<?php

namespace App\Repositories;

use App\Models\BackorderQueue;
use App\Models\BackorderQueueRelease;
use App\Models\Product;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Spatie\Activitylog\Models\Activity;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\QueryBuilder;

class ActivityRepository
{
    public static function index(): Collection|array
    {
        return QueryBuilder::for(Activity::class)
            ->select([
                'activity_log.event',
                'activity_log.subject_type',
                'activity_log.parent_subject_id',
                'activity_log.subject_id',
                'activity_log.properties',
                'activity_log.created_at',
                'users.name as user_name',
            ])
            ->leftJoin('users', function ($join) {
                $join->on('activity_log.causer_id', 'users.id');
                $join->where('activity_log.causer_type', \App\Models\User::class);
            })
            ->allowedFilters([
                'subject_type',
                // filter[subject]=App\\Models\\PurchaseOrder:9,App\\Models\\PurchaseOrder:10
                AllowedFilter::callback('subject', function (Builder $builder, $value) {
                    $subjects = Arr::wrap($value);
                    $builder->where(function (Builder $builder) use ($subjects) {
                        foreach ($subjects as $subject) {
                            $builder->orWhere(function (Builder $builder) use ($subject) {
                                // extract subject type and id
                                $subjectType = explode(':', $subject)[0];
                                $subjectId = explode(':', $subject)[1];
                                // add query to main subject
                                $builder->where(fn ($q) => $q->whereIn('subject_type', static::getSubSubjectTypes($subjectType))->where('parent_subject_id', $subjectId));
                            });
                        }
                    });
                }),
            ])
            ->get();
    }

    private static function getSubSubjectTypes(string $subjectType): array
    {
        $subjectType = stripslashes($subjectType);

        $map = [
            PurchaseOrder::class => [
                PurchaseOrder::class,
                PurchaseInvoice::class,
                PurchaseOrderLine::class
            ],
            SalesOrder::class => [
                SalesOrder::class,
                SalesOrderLine::class,
                SalesOrderFulfillment::class,
                BackorderQueue::class,
                BackorderQueueRelease::class
            ],
            Product::class => [
                Product::class
            ],
        ];

        return $map[$subjectType] ?? [];
    }
}
