<?php

namespace App\Repositories\Accounting;

use App\Abstractions\AbstractRepository;
use App\Data\AccountingBulkReplaceNominalCodesData;
use App\Data\AccountingTransactionBulkEnableSyncData;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Helpers;
use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\NominalCode;
use App\Models\Payment;
use App\Models\Setting;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\DataCollection;
use Spatie\QueryBuilder\AllowedFilter;
use Spatie\QueryBuilder\QueryBuilder;
use Throwable;

class AccountingTransactionRepository extends AbstractRepository
{
    public function deleteTransactionsWithoutLink(): void
    {
        AccountingTransaction::query()->whereDoesntHave('link')
            ->each(function (AccountingTransaction $accountingTransaction) {
                $accountingTransaction->children->each(function (AccountingTransaction $childAccountingTransaction) {
                    $childAccountingTransaction->delete();
                });
                $accountingTransaction->delete();
            });
    }

    public function deleteTransactionLinesWithoutLink(): void
    {
        AccountingTransactionLine::query()->whereDoesntHave('link')
            ->each(function (AccountingTransactionLine $accountingTransactionLine) {
                $accountingTransactionLine->delete();
            });
    }

    /**
     * @throws Exception
     */
    public function getIncomeStatement(string $period, ?int $trailing_days = null): EloquentCollection
    {
        // TODO: Support quarters of the year
        $dateFormat = match ($period) {
            'month' => '%Y-%m',
            'year' => '%Y',
            default => throw new Exception('Unknown period in getSummaryByPeriod()'),
        };

        /**
         * Query to get accounting transaction line totals grouped by period and nominal code where the accounting
         * transaction date is within the specified range.
         */
        $query = QueryBuilder::for(AccountingTransaction::query())
            ->from('accounting_transaction_lines as atl')
            ->selectRaw('nc.id as nominal_code_id, nc.code as nominal_code, nc.name as nominal_code_name, nc.type as nominal_code_type')
            ->selectRaw("DATE_FORMAT(CONVERT_TZ(at.transaction_date, 'UTC', '".Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE)."'), '".$dateFormat."') as transaction_date")
            ->selectRaw('sum(amount * quantity) as total')
            ->join('accounting_transactions as at', 'at.id', '=', 'atl.accounting_transaction_id')
            ->join('nominal_codes as nc', 'nc.id', '=', 'atl.nominal_code_id')
            ->whereIn('nc.type', NominalCode::TYPES_INCOME_STATEMENT)
            ->groupByRaw(DB::raw("DATE_FORMAT(CONVERT_TZ(at.transaction_date, 'UTC', '".Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE)."'), '".$dateFormat."')")->getValue(DB::getQueryGrammar()))
            ->groupBy('nc.id')
            ->groupBy('nc.code')
            ->groupBy('nc.name')
            ->groupBy('nc.type')
            ->havingRaw('sum(amount * quantity) > 0')
            ->orderBy(DB::raw("DATE_FORMAT(CONVERT_TZ(at.transaction_date, 'UTC', '".Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE)."'), '".$dateFormat."')"))
            ->orderBy('nc.code')
            ->allowedFilters([
                AllowedFilter::scope('start_date'),
                AllowedFilter::scope('end_date'),
            ]);

        if (! Arr::has($_REQUEST, 'filter.start_date') && ! Arr::has($_REQUEST, 'filter.end_date') && $trailing_days) {
            $query->trailingDays($trailing_days);
        }

        return $query->get();
    }

    public function setSubmissionStarted(array $ids = []): void
    {
        Log::channel('xero')->debug('Locking transactions for ids: '.implode(', ', $ids));
        $accountingTransaction = new AccountingTransaction();
        $accountingTransaction->timestamps = false;
        $accountingTransaction::query()
            ->whereIn('id', $ids)
            ->update(['submission_started_at' => Carbon::now()->setTimezone('UTC')]);
    }

    public function getPaymentIdsFromTransactions(array $ids = []): array
    {
        return Payment::query()
            ->select('payments.id')
            ->whereHas('link', function (Builder $query) use ($ids) {
                $query->whereHas('accountingTransaction', function (Builder $query) use ($ids) {
                    $query->whereIn('id', $ids);
                });
            })
            ->get()
            ->pluck('id')
            ->toArray();
    }

    public function unLinkFromIntegration(array $ids = []): void
    {
        AccountingTransaction::query()
            ->whereIn('id', $ids)
            ->whereHas('accountingIntegration')
            ->each(function (AccountingTransaction $accountingTransaction) {
                $accountingTransaction->accountingIntegration->delete();
            });
    }

    /**
     * @throws Throwable
     */
    public function saveWithRelations(
        #[DataCollectionOf(AccountingTransactionData::class)]
        DataCollection $data
    ) {
        $data = $data->toCollection();

        return DB::transaction(function () use ($data) {
            $accountingTransactionCollection = $data->map(
                fn ($transaction) => AccountingTransactionData::from(new AccountingTransaction($transaction->toArray()))
            );
            $accountingTransactionCollection = $this->save($accountingTransactionCollection, AccountingTransaction::class);
            $this->saveLines($accountingTransactionCollection, $data);

            return $this->updateParentChildData($accountingTransactionCollection);
        });
    }

    private function saveLines(Collection $accountingTransactionCollection, Collection $data): void
    {
        if ($accountingTransactionCollection->isEmpty()) {
            collect();
            return;
        }

        $accountingTransactionLineCollection = collect();
        $accountingTransactionCollection->each(function ($accountingTransaction) use (
            $accountingTransactionLineCollection,
            $data,
        ) {
            $accountingTransactionData = $data
                ->where('link_id', $accountingTransaction['link_id'])
                ->where('link_type', $accountingTransaction['link_type'])
                ->first();
            $accountingTransactionData->accounting_transaction_lines->each(function (AccountingTransactionLineData $lineData) use (
                $accountingTransactionLineCollection,
                $accountingTransaction,
            ) {
                $lineData->accounting_transaction_id = $accountingTransaction['id'];
                $accountingTransactionLineCollection->add(AccountingTransactionLineData::from((new AccountingTransactionLine($lineData->toArray()))));
            });
        });

        $this->save($accountingTransactionLineCollection, AccountingTransactionLine::class);
    }

    private function updateParentChildData(Collection $accountingTransactionCollection): Collection
    {
        $accountingTransactionCollection = $this->getForValues(
            $accountingTransactionCollection->map(fn ($transaction) => $transaction['id'])->toArray(),
            'id',
            AccountingTransaction::class
        );
        $accountingTransactionCollection = $accountingTransactionCollection->map(function (AccountingTransaction $accountingTransaction) {
            $parent = $accountingTransaction->link->getParentAccountingTransaction();
            if ($parent && ! $accountingTransaction->parent_id) {
                $accountingTransaction->parent_id = $parent->id;
            }

            return $accountingTransaction;
        });

        $parentUpdates = $accountingTransactionCollection->map(fn ($transaction) => $transaction->only(['id', 'parent_id']))->toArray();
        batch()->update(new AccountingTransaction(), $parentUpdates, 'id');

        return $accountingTransactionCollection;
    }

    public function getTransactionsNeedingUpdate(int $limit, int $offset): Collection
    {
        $lockDate = Helpers::setting(Setting::KEY_ACCOUNTING_LOCK_DATE);
        return AccountingTransaction::query()
            ->whereHasMorph('link', ['*'], function (Builder $query, $type) use ($lockDate) {
                $modelInstance = new $type;
                $tableName = $modelInstance->getTable();

                $query
                    ->whereColumn("{$tableName}.updated_at", '>', 'accounting_transactions.updated_at')
                    ->where('is_locked', false)
                    ->accountingReady();

                if ($lockDate) {
                    $query->whereDate("{$tableName}.{$modelInstance->getAccountingDateFieldName()}", '>', $lockDate);
                }
            })
            ->limit($limit)
            ->offset($offset)
            ->orderBy('id', 'desc')
            ->get();
    }

    public function updateLastSyncError(AccountingTransaction $accountingTransaction, array $last_sync_error): AccountingTransaction
    {
        $accountingTransaction->last_sync_error = $last_sync_error;
        $accountingTransaction->update();

        return $accountingTransaction;
    }

    public function clearErrors(array $ids = []): void
    {
        $transactions = AccountingTransaction::query()
            ->whereNotNull('last_sync_error');

        if (!empty($ids)) {
            $transactions->whereIn('id', $ids);
        }

        $transactions = $transactions->get();

        foreach ($transactions as $transaction) {
            $transaction->saveLastSyncError(null);
            if ($accountingLink = $transaction->accountingIntegration)
            {
                $accountingLink->last_error = null;
                $accountingLink->timestamps = false;
                $accountingLink->save();
            }
        }
    }

    public function bulkReplaceNominalCodes(AccountingBulkReplaceNominalCodesData $data): void
    {
        AccountingTransaction::with('accountingTransactionLines')
            ->whereIn('id', $data->ids)->each(function (AccountingTransaction $transaction) use ($data) {
                $lineIsDirty = false;
                $transaction->accountingTransactionLines->where('nominal_code_id', $data->old_nominal_code_id)
                    ->each(function (AccountingTransactionLine $line) use ($data, &$lineIsDirty) {
                        $line->nominal_code_id = $data->new_nominal_code_id;
                        if ($line->isDirty('nominal_code_id'))
                        {
                            $lineIsDirty = true;
                            $line->save();
                        }
                    });
                if ($lineIsDirty && $data->is_locked) {
                    $transaction->is_locked = true;
                    $transaction->save();
                }
            });
    }

    public function bulkEnableSync(AccountingTransactionBulkEnableSyncData $data): void
    {
        AccountingTransaction::whereIn('id', $data->ids)->update(['is_sync_enabled' => $data->status]);
    }
}
