<?php

namespace App\Queries;

use App\Exporters\MapsExportableFields;
use App\Importers\DataImporters\ProductDataImporter;
use App\Models\ProductInventory;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Query\JoinClause;

// use App\Lib\Inspect\Inspect;

class Product extends \App\Models\Product implements MapsExportableFields
{
    /**
     * used to prevent joins with other tables, only use sub-query.
     *
     * @var bool
     */
    protected $relationsByJoin = false;

    /**
     * Get a new query builder for the model's table.
     */
    public function newQuery(): Builder
    {
        $query = parent::newQuery()->select($this->getBasicSelect());

        if ($this->isInventoryRequiredJoin()) {
            $query->leftJoin('products_inventory', function (JoinClause $join) {
                $join->on('products.id', '=', 'products_inventory.product_id')
                    ->where('products_inventory.warehouse_id', ProductInventory::$idForTotalWarehouses);
            })->addSelect($this->getInventorySelect());
        }

        // if inventory fields does not require to the (filter or sort), but they are required in response only, add inventory fields by with
        if ($this->isInventoryRequiredWith() && ! $this->isInventoryRequiredJoin()) {
            $query->with('totalInventory');
        }

        return $query;
    }

    /**
     * Get products columns.
     */
    private function getBasicSelect(): array
    {
        $columns = [
            'sku',
            'name',
            'barcode',
            'mpn',
            'unit_cost',
            'weight',
            'weight_unit',
            'length',
            'width',
            'height',
            'dimension_unit',
            'fba_prep_instructions',
            'case_quantity',
            'case_length',
            'case_width',
            'case_height',
            'case_weight',
            'case_weight_unit',
            'case_dimension_unit',
            'average_cost',
            'daily_average_consumption',
            'shared_children_attributes',
            'proforma_landed_cost_percentage',
            'archived_at',
            'created_at',
            'updated_at',
        ];

        // used in product response
        $except = json_decode(request('excluded', '[]'), true);
        $include = json_decode(request('included', '[]'), true);

        $select = [
            $this->qualifyColumn('id'),
            $this->qualifyColumn('brand_id'),
            $this->qualifyColumn('parent_id'),
            $this->qualifyColumn('sales_nominal_code_id'),
            $this->qualifyColumn('cogs_nominal_code_id'),
            $this->qualifyColumn('is_taxable'),
            $this->qualifyColumn('default_tax_rate_id'),
            $this->qualifyColumn('type'),
        ]; // we need them to relations

        foreach ($columns as $column) {
            if (empty($include) && ! in_array($column, $except)) {
                $select[] = $this->qualifyColumn($column);
            } elseif (empty($except) && in_array($column, $include)) {
                $select[] = $this->qualifyColumn($column);
            }
        }

        return $select;
    }

    private function getInventorySelect()
    {
        return [
            'products_inventory.inventory_total',
            'products_inventory.inventory_reserved',
            'products_inventory.inventory_stock_value',
            'products_inventory.inventory_in_transit',
            'products_inventory.inventory_available',
        ];
    }

    /**
     * Get the class name for polymorphic relations.
     */
    public function getMorphClass(): string
    {
        return \App\Models\Product::class;
    }

    /**
     * Determine if any of the inventory fields are required in the query.
     */
    private function isInventoryRequiredJoin(): bool
    {
        $inventoryColumns = [
            'inventory_total',
            'inventory_reserved',
            'inventory_stock_value',
            'inventory_in_transit',
            'inventory_available',
        ];

        return $this->isRequired($inventoryColumns, true, true, false);
    }

    /**
     * Determine if any of the inventory fields are required in the response.
     */
    private function isInventoryRequiredWith(): bool
    {
        $inventoryColumns = [
            'inventory_total',
            'inventory_reserved',
            'inventory_stock_value',
            'inventory_in_transit',
            'inventory_available',
        ];

        return $this->isRequired($inventoryColumns, false, false, true);
    }

    /**
     * Determine if fields is required in request.
     *
     * @param  array|string  $fields
     */
    private function isRequired($fields, bool $inFilter = true, bool $orInSort = true, bool $orInResponse = true): bool
    {
        $fields = is_array($fields) ? $fields : [$fields];

        // used in filters
        $filterSet = json_decode(request('filters', '{}'), true)['filterSet'] ?? [];
        $inFilters = collect($filterSet)->whereIn('column', $fields)->isNotEmpty();

        // only need total(count)
        if (request('total', 0) == 1) {
            return $inFilters;
        }

        if ($inFilter && ! ($orInSort || $orInResponse)) {
            return $inFilters;
        }

        // used in sorts
        $sortObjs = json_decode(request('sortObjs', '{}'), true);
        $inSorts = collect($sortObjs)->whereIn('column', $fields)->isNotEmpty();

        if ($orInSort && ! ($inFilter || $orInResponse)) {
            return $inSorts;
        }

        if ($orInSort && $inFilter && ! $orInResponse) {
            return $inFilters || $inSorts;
        }

        // used in product response
        $except = json_decode(request('excluded', '[]'), true);
        $include = json_decode(request('included', '[]'), true);

        $exceptCondition = empty($include) && array_diff($fields, $except);
        $includeCondition = empty($except) && array_intersect($fields, $include);

        if ($orInResponse && ! ($inFilter || $orInSort)) {
            return $exceptCondition || $includeCondition || (empty($include) and empty($except));
        }

        return $exceptCondition || $includeCondition || (empty($include) and empty($except)) || $inFilters || $inSorts;
    }

    /**
     * Calculated columns used to filter by "having".
     */
    public function calculatedColumns(): array
    {
        return ['category_main_path', 'category_others_path'];
    }

    public function getCasts()
    {
        $benchmarkCasts = [
            'products_benchmarks.benchmark_date' => 'date',
            'products_benchmarks.inventory_total' => 'float',
            'products_benchmarks.inventory_reserved' => 'float',
            'products_benchmarks.inventory_stock_value' => 'float',
            'products_benchmarks.inventory_in_transit' => 'float',
            'products_benchmarks.inventory_available' => 'float',
            'products_benchmarks.avg_cost' => 'float',
            'days_in_stock' => 'integer',
            'inventory_total' => 'float',
            'inventory_reserved' => 'float',
            'inventory_stock_value' => 'float',
            'inventory_in_transit' => 'float',
            'inventory_available' => 'float',
        ];

        $this->casts = array_merge($this->casts, $benchmarkCasts);

        return parent::getCasts();
    }

    public static function getExportableFields(): array
    {
        return ProductDataImporter::getExportableFields();
    }
}
