<?php

namespace App\Observers;

use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Models\IntegrationInstance;
use App\Models\ProductComponent;
use App\Models\ProductInventory;
use App\Models\ProductListing;
use App\Models\SalesChannel;

class UpdateProductListingQuantityObserver
{
    /**
     * Handle the product listing "saved" event.
     */
    public function saved(ProductInventory $productInventory): void
    {
        // not total warehouse
        if (
            $productInventory->warehouse_id != ProductInventory::$idForTotalWarehouses &&
            (! $productInventory->wasRecentlyCreated || $productInventory->inventory_available != 0)
            && $productInventory->isDirty('inventory_available')
        ) {
            $this->updateProductListingsQuantity($productInventory);

            $salesChannelIdsForProducts = $productInventory->product->productListings()?->groupBy('sales_channel_id')->select('sales_channel_id')->pluck('sales_channel_id')->toArray();

            foreach ($salesChannelIdsForProducts as $salesChannelId) {
                /** @var SalesChannel $salesChannel */
                $salesChannel = SalesChannel::query()->find($salesChannelId);

                $productIds = $productInventory->product->productListings()
                    ->where('sales_channel_id', $salesChannelId)
                    ->pluck('product_id')
                    ->toArray();

                $bundleComponents = ProductComponent::whereIn('component_product_id', $productIds)
                    ->pluck('parent_product_id')
                    ->toArray();

                /** @var IntegrationInstance $integrationInstance */
                $integrationInstance = $salesChannel->integrationInstance;
                customlog('syncInventory', 'UpdateProductListingQuantityObserver: running for integration instance '.$integrationInstance->id.' and sales channel '.$salesChannel->id.' with '.count($productIds).' products and '.count($bundleComponents).' bundle components');
                dispatch(new GenerateCacheProductListingQuantityJob($integrationInstance, array_merge($productIds, $bundleComponents)));
            }
        }
    }

    /**
     * Handle the product listing "deleted" event.
     */
    public function deleted(ProductInventory $productInventory): void
    {
        // we don't delete productInventory, just to be sure
        if ($productInventory->warehouse_id != ProductInventory::$idForTotalWarehouses && $productInventory->inventory_available != 0) {
            $this->updateProductListingsQuantity($productInventory);
        }
    }

    /**
     * @param  ProductInventory  $productInventory
     *
     * query for more performance when updating by warehouse or something else
     *
     * select product_listings.id,
     * product_listings.product_id,
     * product_listings.inventory_rules,
     * product_listings.sales_channel_id,
     * ii.id,
     * ii.integration_settings
     * from product_listings
     * left join sales_channels sc on product_listings.sales_channel_id = sc.id
     * left join integration_instances ii on sc.integration_instance_id = ii.id
     * where product_listings.product_id = 1712
     * and (
     * json_contains(json_extract(`inventory_rules`, '$.selectedWarehouses[*].id'), '1') or
     * json_contains(json_extract(ii.integration_settings, '$.inventory.selectedWarehouses[*].id'), '1') or
     * json_contains(json_extract(ii.integration_settings, '$.inventoryLocations[0].selectedWarehouses[*].id'), '1'));
     */
    private function updateProductListingsQuantity(ProductInventory $productInventory)
    {
        customlog('syncInventory', 'UpdateProductListingQuantityObserver: productInventory');
        ProductListing::with(['salesChannel.integrationInstance.integration', 'product.productInventory'])
            ->where('product_id', $productInventory->product_id)
            ->each(function (ProductListing $productListing) use ($productInventory) {
                if (in_array($productInventory->warehouse_id, $productListing->active_warehouses_ids)) {
                    $productListing->save(['set_price' => false, 'set_quantity' => true]);
                }
            });
    }
}
