<?php

namespace App\Observers;

use App\Helpers;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Repositories\DailyFinancialRepository;
use App\Repositories\ProductRepository;
use App\Repositories\SalesOrderLineFinancialsRepository;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;

class InvalidateFinancialReportingCacheObserver
{
    private DailyFinancialRepository $dailyFinancialRepository;

    private SalesOrderLineFinancialsRepository $salesOrderLineFinancialsRepository;

    private ProductRepository $productRepository;

    public function __construct()
    {
        $this->dailyFinancialRepository = app(DailyFinancialRepository::class);
        $this->salesOrderLineFinancialsRepository = app(SalesOrderLineFinancialsRepository::class);
        $this->productRepository = app(ProductRepository::class);
    }

    public function created(Model $model): void
    {
        if ($model instanceof SalesOrderLine) {
            $this->salesOrderLineFinancialsRepository->invalidateForSalesOrderLineIds($model->id);
            $this->dailyFinancialRepository->invalidateForDates(Helpers::utcToStartOfLocalDateInUtc($model->salesOrder->order_date)->toDateTimeString());
        }
    }

    public function updated(Model $model): void
    {
        if ($model instanceof SalesOrder) {
            if ($model->salesOrderLines->isNotEmpty()) {
                $this->salesOrderLineFinancialsRepository->invalidateForSalesOrderLineIds($model->salesOrderLines->pluck('id')->toArray());
                $this->dailyFinancialRepository->invalidateForDates(Helpers::utcToStartOfLocalDateInUtc($model->order_date)->toDateTimeString());
            }
        }

        if ($model instanceof SalesOrderLine) {
            $this->salesOrderLineFinancialsRepository->invalidateForSalesOrderLineIds($model->id);
            $this->dailyFinancialRepository->invalidateForDates(Helpers::utcToStartOfLocalDateInUtc($model->salesOrder->order_date)->toDateTimeString());
        }

        if ($model instanceof Product && $model->isDirty('unit_cost')) {
            if ($model->salesOrderLines->isNotEmpty()) {
                $this->salesOrderLineFinancialsRepository->invalidateForSalesOrderLineIds($model->salesOrderLines->pluck('id')->toArray());
            }
        }
    }

    public function deleted(Model $model): void
    {
        if ($model instanceof SalesOrderLine) {
            /*
             * TODO: When deleting a large number of records, the following is causing deadlocks.  Needs to be handled in a bulk way
             *  Not through observer
             */
            $this->productRepository->invalidateDailyAverageConsumptionCache(Arr::wrap($model->product_id));
            $this->dailyFinancialRepository->invalidateForDates(Helpers::utcToStartOfLocalDateInUtc($model->salesOrder->order_date)->toDateTimeString());
        }
    }
}
