<?php

namespace App\Notifications;

use App\Data\FinancialAlertSummaryNotificationData;
use App\Enums\FinancialAlertNotificationConditionEnum;
use App\Exceptions\FinancialAlertNotificationConditionTypeException;
use App\Models\FinancialAlert;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class FinancialAlertEmailNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(
        private readonly ?FinancialAlert $financialAlert = null,
        private readonly ?FinancialAlertSummaryNotificationData $data = null,
        private readonly ?FinancialAlertNotificationConditionEnum $condition = null,
    )
    {
    }

    public function via($notifiable): array
    {
        return ['mail'];
    }

    /**
     * @throws FinancialAlertNotificationConditionTypeException
     */
    public function toMail($notifiable): MailMessage
    {
        if (isset($this->financialAlert)) {
            return (new MailMessage)
                ->subject('Low Margin/Loss Alert for Sales Order #' . $this->financialAlert->salesOrderLineFinancial->salesOrderLine->salesOrder->sales_order_number)
                ->view('mail.financial_alert', ['financialAlert' => $this->financialAlert]);
        }

        $alertCount = $this->data->new_alert_count;

        $subject = match($this->condition) {
            FinancialAlertNotificationConditionEnum::DAILY_SUMMARY=> "Daily Summary: $alertCount Low Margin/Loss Alerts",
            FinancialAlertNotificationConditionEnum::WEEKLY_SUMMARY => "Weekly Summary: $alertCount Low Margin/Loss Alerts",
            FinancialAlertNotificationConditionEnum::IMMEDIATELY => throw new FinancialAlertNotificationConditionTypeException,
        };

        return (new MailMessage)
            ->from('noreply@skui.io')
            ->subject($subject)
            ->view('mail.financial_alert_summary', ['data' => $this->data, 'condition' => $this->condition]);
    }
}
