<?php

namespace App\Notifications;

use App\Abstractions\AdvancedShipmentNoticeInterface;
use App\Exports\AdvancedShipmentInvoiceExport;
use App\Helpers;
use App\Models\Setting;
use App\Models\StoreEmailTemplate;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Maatwebsite\Excel\Facades\Excel;
use ReflectionClass;

class AdvancedShipmentNoticesNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(private readonly AdvancedShipmentNoticeInterface $order, private readonly array $lines)
    {
    }

    public function via(): array
    {
        return ['mail'];
    }

    /**
     * @throws Exception
     */
    public function toMail(): MailMessage
    {
        $type = StoreEmailTemplate::TYPE_ASN_NOTES;

        if ($this->order instanceof WarehouseTransfer && $this->order->fromWarehouse->type == Warehouse::TYPE_AMAZON_FBA) {
            $type = StoreEmailTemplate::TYPE_ASN_NOTES_FBA;
        }

        $emailTemplate = StoreEmailTemplate::with([])
            ->where('type', $type)
            ->first();
        if (!$emailTemplate || !$emailTemplate->enabled) {
            throw new Exception('Email template is not enabled.');
        }

        $model = (new ReflectionClass($this->order))->getShortName();

        $data = StoreEmailTemplate::fillTemplate($this->order->id, $model, $emailTemplate->subject,
            $emailTemplate->html_body);

        $fileType = Helpers::setting(Setting::KEY_ASN_FILE_FORMAT);

        $file = Excel::raw(new AdvancedShipmentInvoiceExport($this->lines), $this->getType($fileType));

        $mime = $this->getMime($fileType);

        $reference = $this->order->getReferenceNumber();

        $cc = Helpers::setting(Setting::KEY_ASN_CC_EMAIL);

        $mail = (new MailMessage)
            ->from($this->order->getFromEmail() ?? config('mail.from.address'))
            ->subject($data['subject'])
            ->attachData($file, "ASN $reference.$fileType", ['mime' => $mime])
            ->view('mail.template', ['body' => $data['body']]);

        if($cc) {
            $mail->cc($cc);
        }

        return $mail;
    }

    private function getType(string $type): string
    {
        return match (strtolower($type)) {
            'xlsx' => \Maatwebsite\Excel\Excel::XLSX,
            'xls' => \Maatwebsite\Excel\Excel::XLS,
            default => \Maatwebsite\Excel\Excel::CSV,
        };

    }

    private function getMime(string $type): string
    {
        return match (strtolower($type)) {
            'xlsx' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'xls' => 'application/vnd.ms-excel',
            default => 'text/csv',
        };

    }

    public function toArray(): array
    {
        return [];
    }
}
