<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 7/10/20
 * Time: 8:40 AM.
 */

namespace App\Models;

use App\Abstractions\FinancialDocumentInterface;
use App\Contracts\HasReference;
use App\Data\AccountingTransactionData;
use App\Models\Concerns\HasAccountingTransaction;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionDataFromWarehouseTransferShipment;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Facades\DB;
use Throwable;

/**
 * Class WarehouseTransferShipment.
 *
 * @property-read int $id
 * @property int $shipping_method_id
 * @property float $cost
 * @property int $warehouse_transfer_id
 * @property string $tracking_number
 * @property Carbon $shipped_at
 * @property WarehouseTransfer $warehouseTransfer
 * @property ShippingMethod $shippingMethod
 * @property-read Collection|WarehouseTransferShipmentLine[] $shipmentLines
 * @property-read Collection|WarehouseTransferShipmentReceiptLine[] $receiptLines
 * @property-read Collection|WarehouseTransferShipmentReceipt[] $receipts
 * @property Carbon $fully_received_at
 * @property-read int $total_shipped
 * @property-read int $total_received
 */
class WarehouseTransferShipment extends Model implements HasReference, FinancialDocumentInterface
{
    use HasAccountingTransaction;
    use HasFactory;

    /**
     * @var array
     */
    protected $fillable = [
        'shipping_method_id',
        'cost',
        'warehouse_transfer_id',
        'tracking_number',
        'shipped_at',
    ];

    protected $casts = [
        'shipped_at' => 'datetime',
        'fully_received_at' => 'datetime',
    ];

    /**
     * @throws Throwable
     */
    public function getAccountingTransactionData(): AccountingTransactionData
    {
        return (new BuildAccountingTransactionDataFromWarehouseTransferShipment($this))->handle();
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return null;
    }

    public function scopeAccountingReady(Builder $builder): Builder
    {
        return $builder;
    }

    public function accountingTransaction(): MorphOne
    {
        return $this->morphOne(AccountingTransaction::class, 'link');
    }

    public function getAccountingDateFieldName(): string
    {
        return 'shipped_at';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function warehouseTransfer(): BelongsTo
    {
        return $this->belongsTo(WarehouseTransfer::class);
    }

    public function shippingMethod(): BelongsTo
    {
        return $this->belongsTo(ShippingMethod::class);
    }

    public function shipmentLines(): HasMany
    {
        return $this->hasMany(WarehouseTransferShipmentLine::class);
    }

    public function receipts(): HasMany
    {
        return $this->hasMany(WarehouseTransferShipmentReceipt::class);
    }

    public function receiptLines(): HasManyThrough
    {
        return $this->hasManyThrough(WarehouseTransferShipmentReceiptLine::class, WarehouseTransferShipmentReceipt::class);
    }

    public function getTotalShippedAttribute()
    {
        return $this->shipmentLines()->pluck('quantity')->sum();
    }

    public function getTotalReceivedAttribute()
    {
        return $this->receiptLines()->pluck('quantity')->sum();
    }

    public function isFullyReceived(): bool
    {
        // The total shipped quantity must match the total received quantities
        return $this->total_shipped === $this->total_received;
    }

    public function getReference(): ?string
    {
        return $this->warehouseTransfer->getReference();
    }

    public function fullyReceived($receiptDate): static
    {
        $this->fully_received_at = $receiptDate;
        $this->save();

        return $this;
    }

    public function delete()
    {
        $this->load('shipmentLines', 'receipts', 'receiptLines');

        DB::transaction(function () {
            // Delete Receipts
            if ($this->receipts) {
                $this->receipts->each(function (WarehouseTransferShipmentReceipt $receipt) {
                    $receipt->delete();
                });
            }

            // Delete shipment lines
            if ($this->shipmentLines) {
                $this->shipmentLines->each(function (WarehouseTransferShipmentLine $line) {
                    $line->delete();
                });
            }

            // Delete The shipment
            return parent::delete();
        });
    }
}
