<?php

namespace App\Models;

use App\Abstractions\AdvancedShipmentNoticeInterface;
use App\Abstractions\HasNotesInterface;
use App\Abstractions\UniqueFieldsInterface;
use App\Contracts\HasReference;
use App\Enums\AccountingTransactionTypeEnum;
use App\Exporters\BaseExporter;
use App\Exporters\MapsExportableFields;
use App\Exporters\TransformsExportData;
use App\Importers\DataImporters\WarehouseTransferDataImporter;
use App\Models\Amazon\FBAFulfillmentRemovalOrderDetail;
use App\Models\Amazon\InboundShipment;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasNotesTrait;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Abstractions\AmazonInboundShipmentSourceInterface;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonFbaReportRemovalShipment;

/**
 * Class WarehouseTransfer.
 *
 *
 * @property int $id
 * @property Carbon $transfer_date
 * @property int $product_id
 * @property float $quantity
 * @property int $from_warehouse_id
 * @property int $to_warehouse_id
 * @property int $shipping_method_id
 * @property string $warehouse_transfer_number
 * @property string $transfer_status
 * @property string $shipment_status
 * @property string $receipt_status
 * @property string $shipment_id
 * @property Carbon $eta
 * @property Carbon $created_at
 * @property Carbon $update_at
 * @property Carbon|null $fully_shipped_at
 * @property Carbon|null $fully_received_at
 * @property-read Collection $inventoryMovements
 * @property-read Collection|WarehouseTransferLine[] $warehouseTransferLines
 * @property-read Collection $shipmentLines
 * @property-read Collection $shipmentReceipts
 * @property-read WarehouseTransferShipment $shipment
 * @property-read WarehouseTransferShipment[]|EloquentCollection $shipments
 * @property-read int $total_shipped
 * @property-read int $total_quantity
 * @property-read int $total_received
 * @property-read float $receipt_percent
 * @property-read float $shipment_percent
 * @property Warehouse $fromWarehouse
 * @property Warehouse $toWarehouse
 * @property ShippingMethod $shippingMethod
 * @property-read InboundShipmentRelation[]|EloquentCollection $inboundShipmentRelations
 * @property-read Note[]|EloquentCollection $notes,
 * @property-read AccountingTransaction $receivingDiscrepancy
 * @property Carbon|null $asn_last_sent_at
 */
class WarehouseTransfer extends Model
    implements
    AmazonInboundShipmentSourceInterface,
    Filterable,
    HasNotesInterface,
    HasReference,
    MapsExportableFields,
    Sortable,
    TransformsExportData,
    UniqueFieldsInterface,
    AdvancedShipmentNoticeInterface
{
    use Archive;
    use HasFactory;
    use HasFilters;
    use HasNotesTrait;
    use HasSort;

    const TRANSFER_STATUS_DRAFT = 'draft';

    const TRANSFER_STATUS_OPEN = 'open';

    const TRANSFER_STATUS_CLOSED = 'closed';

    const TRANSFER_STATUSES = [
        self::TRANSFER_STATUS_DRAFT,
        self::TRANSFER_STATUS_OPEN,
        self::TRANSFER_STATUS_CLOSED,
    ];

    const TRANSFER_SHIPMENT_STATUS_UNSHIPPED = 'unshipped';

    const TRANSFER_SHIPMENT_STATUS_PARTIALLY_SHIPPED = 'partially_shipped';

    const TRANSFER_SHIPMENT_STATUS_SHIPPED = 'shipped';

    const TRANSFER_SHIPMENT_STATUSES = [
        self::TRANSFER_SHIPMENT_STATUS_UNSHIPPED,
        self::TRANSFER_SHIPMENT_STATUS_PARTIALLY_SHIPPED,
        self::TRANSFER_SHIPMENT_STATUS_SHIPPED,
    ];

    const TRANSFER_RECEIPT_STATUS_UNRECEIVED = 'unreceived';

    const TRANSFER_RECEIPT_STATUS_PARTIALLY_RECEIVED = 'partially_received';

    const TRANSFER_RECEIPT_STATUS_RECEIVED = 'received';

    const TRANSFER_RECEIPT_STATUSES = [
        self::TRANSFER_RECEIPT_STATUS_UNRECEIVED,
        self::TRANSFER_RECEIPT_STATUS_PARTIALLY_RECEIVED,
        self::TRANSFER_RECEIPT_STATUS_RECEIVED,
    ];

    /**
     * @var array
     */
    protected $fillable = [
        'warehouse_transfer_number',
        'transfer_date',
        'from_warehouse_id',
        'to_warehouse_id',
        'eta',
        'shipping_method_id',
        'transfer_status',
        'shipment_status',
        'receipt_status',
        'fully_shipped_at',
        'fully_received_at',
        'shipment_id',
    ];

    /**
     * @var array
     */
    protected $casts = [
        'transfer_date' => 'datetime',
        'fully_shipped_at' => 'datetime',
        'fully_received_at' => 'datetime',
        'quantity' => 'float',
        'asn_last_sent_at' => 'datetime',
    ];

    public static function getUniqueFields(): array
    {
        return ['warehouse_transfer_number'];
    }

    public static function getLinesRelationName(): string
    {
        return 'warehouseTransferLines';
    }

    public static function getNumberFieldName(): string
    {
        return 'warehouse_transfer_number';
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return null;
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function fromWarehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class, 'from_warehouse_id');
    }

    public function toWarehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class, 'to_warehouse_id');
    }

    public function shippingMethod()
    {
        return $this->belongsTo(ShippingMethod::class);
    }

    public function inventoryMovements(): MorphMany
    {
        return $this->morphMany(InventoryMovement::class, 'link');
    }

    public function inboundShipmentRelations(): MorphMany
    {
        return $this->morphMany(InboundShipmentRelation::class, 'link');
    }

    public function shipmentReceipts(): HasManyThrough
    {
        return $this->hasManyThrough(WarehouseTransferShipmentReceipt::class, WarehouseTransferShipment::class);
    }

    public function warehouseTransferLines(): HasMany
    {
        return $this->hasMany(WarehouseTransferLine::class);
    }

    public function productLines(): HasMany
    {
        return $this->warehouseTransferLines();
    }

    public function shipments()
    {
        return $this->hasMany(WarehouseTransferShipment::class);
    }

    public function shipment(): HasOne
    {
        return $this->hasOne(WarehouseTransferShipment::class);
    }

    public function shipmentLines(): HasManyThrough
    {
        return $this->hasManyThrough(WarehouseTransferShipmentLine::class, WarehouseTransferShipment::class);
    }

    public function removalShipments(): HasMany
    {
        return $this->hasMany(AmazonFbaReportRemovalShipment::class, 'order_id', 'warehouse_transfer_number');
    }

    public function removalShipment(): HasOne
    {
        return $this->hasOne(AmazonFbaReportRemovalShipment::class, 'order_id', 'warehouse_transfer_number');
    }

    public function receivingDiscrepancy(): MorphOne
    {
        return $this->morphOne(AccountingTransaction::class, 'link')
            ->where('type', AccountingTransactionTypeEnum::RECEIVING_DISCREPANCY);
    }

    public function document(): ?Model
    {
        if (! $integrationInstance = IntegrationInstance::amazon()->first()) {
            return null;
        }

        if (! $integrationInstance->isFinalized()) {
            return null;
        }

        if (
            $inboundShipment =
            InboundShipment::query()->where('ShipmentId', $this->warehouse_transfer_number)->first()
        ) {
            return $inboundShipment;
        }

        return
            FBAFulfillmentRemovalOrderDetail::query()
                ->where('order-id', $this->warehouse_transfer_number)
                ->first();
    }

    public function getTotalQuantityAttribute(): int
    {
        return $this->warehouseTransferLines()->pluck('quantity')->sum();
    }

    public function getTotalShippedAttribute(): int
    {
        return $this->shipmentLines()->pluck('quantity')->sum();
    }

    public function getTotalShippedBeforeStartDateAttribute(): int
    {
        return $this->warehouseTransferLines()->pluck('quantity_shipped_before_start_date')->sum();
    }

    public function getReference(): string
    {
        return $this->warehouse_transfer_number;
    }

    public function getTotalReceivedAttribute(): int
    {
        $totalReceived = 0;
        $this->shipmentLines->each(function (WarehouseTransferShipmentLine $shipmentLine) use (&$totalReceived) {
            $totalReceived += $shipmentLine->receiptLines()->pluck('quantity')->sum();
            $totalReceived += $shipmentLine->warehouseTransferLine->adjustments()->pluck('quantity')->sum();
        });

        return $totalReceived;
    }

    /**
     * Gets the percent received.
     *
     * @return float|int
     */
    public function getReceiptPercentAttribute()
    {
        if (($totalQuantity = $this->warehouseTransferLines()->sum('quantity')) === 0) {
            return 0;
        }

        return round(($this->total_received / $totalQuantity) * 100, 2);
    }

    /**
     * Gets the percent shipped.
     *
     * @return float|int
     */
    public function getShipmentPercentAttribute()
    {
        if (($totalQuantity = $this->warehouseTransferLines()->sum('quantity')) === 0) {
            return 0;
        }

        return round(($this->total_shipped / $totalQuantity) * 100, 2);
    }

    public function save(array $options = []): bool
    {
        // SKU-2435
        if ($this->toWarehouse && $this->toWarehouse->isAmazonFBA()) {
            $this->shipping_method_id = null;
        }

        return parent::save($options);
    }

    public function totalShippedForProduct($productId): float
    {
        if ($this->isDraft() || ! $this->shipmentLines) {
            return 0;
        }

        return $this->shipmentLines()
            ->select('warehouse_transfer_line_id', 'quantity')
            ->whereHas('warehouseTransferLine', function (Builder $builder) use ($productId) {
                $builder->where('product_id', $productId);
            })->pluck('quantity')->sum();
    }

    public function totalQuantityForProduct($productId): Collection
    {
        return $this->warehouseTransferLines()->where('product_id', $productId)->pluck('quantity')->sum();
    }

    public function totalQuantityReceivedForProduct($productId)
    {
        if ($this->isDraft() || ! $this->shipmentLines) {
            return 0;
        }

        $shipmentLinesForProduct = $this->shipmentLines()->whereHas('warehouseTransferLine', function (Builder $builder) use ($productId) {
            $builder->where('warehouse_transfer_lines.product_id', $productId);
        });

        // Find receipt lines for the product shipment lines and sum up the quantities.
        return WarehouseTransferShipmentReceiptLine::with([])
            ->whereIn(
                'warehouse_transfer_shipment_line_id',
                $shipmentLinesForProduct->pluck('warehouse_transfer_shipment_lines.id')->toArray()
            )
            ->pluck('quantity')->sum();
    }

    public function setTransferNumber($number)
    {
        $this->warehouse_transfer_number = $number;
        $this->save();
    }

    /**
     * Checks if the warehouse transfer is draft.
     */
    public function isDraft(): bool
    {
        return $this->transfer_status === static::TRANSFER_STATUS_DRAFT;
    }

    public function isClosed(): bool
    {
        return $this->transfer_status === static::TRANSFER_STATUS_CLOSED;
    }

    /**
     * Sets the status of the warehouse transfer to open.
     */
    public function open(): static
    {
        $this->transfer_status = static::TRANSFER_STATUS_OPEN;
        $this->save();

        return $this;
    }

    public function totallyShipped($lastShipmentDate): static
    {
        $this->shipment_status = static::TRANSFER_SHIPMENT_STATUS_SHIPPED;
        $this->fully_shipped_at = $lastShipmentDate;
        $this->save();

        return $this;
    }

    public function partiallyShipped(): static
    {
        $this->shipment_status = static::TRANSFER_SHIPMENT_STATUS_PARTIALLY_SHIPPED;
        $this->save();

        return $this;
    }

    public function totallyReceived($lastReceiptDate): static
    {
        $this->receipt_status = static::TRANSFER_RECEIPT_STATUS_RECEIVED;
        $this->fully_received_at = $lastReceiptDate;
        $this->save();

        return $this;
    }

    public function partiallyReceived(): static
    {
        $this->receipt_status = static::TRANSFER_RECEIPT_STATUS_PARTIALLY_RECEIVED;
        $this->transfer_status = static::TRANSFER_STATUS_OPEN;
        $this->save();

        return $this;
    }

    /**
     * Completes the warehouse transfer.
     */
    public function close(): static
    {
        $this->transfer_status = static::TRANSFER_STATUS_CLOSED;
        $this->save();

        return $this;
    }

    /**
     * {@inheritDoc}
     */
    public function delete()
    {
        //Load deletable relationships

        $this->load('warehouseTransferLines', 'shipments');

        return DB::transaction(function () {
            // Delete related models

            // Delete shipment
            if ($this->shipments->isNotEmpty()) {
                $this->shipments->each(function (WarehouseTransferShipment $shipment) {
                    $shipment->delete();
                });
            }

            // Delete transfer lines
            if ($this->warehouseTransferLines) {
                $this->warehouseTransferLines->each(function (WarehouseTransferLine $line) {
                    $line->delete();
                });
            }

            /*
             * @deprecated See notes in InboundShipmentRelation
             */
            //            // Delete inbound shipping relations
            //            $this->inboundShipmentRelations->each(function (InboundShipmentRelation $shipmentRelation) {
            //                $shipmentRelation->delete();
            //            });

            // Clear relationship with Amazon FBA Inbound Shipment
            AmazonFbaInboundShipment::query()
                ->whereHas('skuLink', function ($query) {
                    $query->where('sku_link_id', $this->id);
                    $query->where('sku_link_type', get_class($this));
                })->update(['sku_link_id' => null, 'sku_link_type' => null]);

            // Delete the warehouse transfer
            return parent::delete();
        });
    }

    public function isUsed(): bool|array
    {
        if ($this->receipt_status == static::TRANSFER_RECEIPT_STATUS_UNRECEIVED) {
            return false;
        }

        foreach ($this->warehouseTransferLines as $transferLine) {
            DB::beginTransaction();
            try {
                $transferLine->delete();
            } catch (\Throwable $exception) {
                return [__('messages.warehouse.transfer_used', ['id' => $this->warehouse_transfer_number, 'sku' => $transferLine->product->sku])];
            } finally {
                DB::rollBack();
            }
        }

        return false;
    }

    public function scopeFilterAsnSent(Builder $builder, $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        $builder->{$function}(function(Builder $builder) use ($value, $operator) {
            $function = $value ? 'whereNotNull' : 'whereNull';
            $builder->{$function}('asn_last_sent_at');
            if(!$value) {
                $builder->whereHas('toWarehouse', function (Builder $builder) {
                        $builder->where('type', Warehouse::TYPE_3PL);
                    });
            }
        });
    }

    public function scopeAccountingReady(Builder $builder): Builder
    {
        return $builder;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.warehouse_transfer.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return [];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public static function getExportableFields(): array
    {
        return WarehouseTransferDataImporter::getExportableFields();
    }

    public static function transformExportData(array $data): array
    {
        return BaseExporter::groupByLines($data, 'warehouse_transfer_number', 'items');
    }

    public function getReferenceNumber(): string
    {
        return $this->warehouse_transfer_number;
    }

    public function getDestinationAddress(): Address
    {
        return $this->toWarehouse->address;
    }

    public function getFromEmail(): string
    {
        return Store::whereNotNull('email')->first()->email;
    }


}
