<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Exceptions\DropshipWithOpenOrdersException;
use App\Exporters\MapsExportableFields;
use App\Importers\DataImporter;
use App\Importers\DataImporters\WarehouseDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Modules\Amazon\Abstractions\AmazonFbaInboundFromInterface;

/**
 * Class Warehouse.
 *
 *
 * @property int $id
 * @property string $name
 * @property string $type
 * @property int $supplier_id
 * @property int $integration_instance_id
 * @property int $address_id
 * @property string $order_fulfillment
 * @property int $nominal_code_id
 * @property bool $dropship_enabled
 * @property bool $direct_returns
 * @property bool $customer_returns
 * @property bool $auto_routing_enabled
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read WarehouseLocation|null $defaultLocation
 * @property Collection $warehouseLocations
 * @property-read Address|null $address
 * @property Supplier $supplier
 * @property Collection $supplierInventory
 * @property Collection $inventoryMovements
 * @property Collection $fifoLayers
 * @property-read string|null $supplier_name
 * @property-read bool $is_dropship
 * @property-read bool $is_fba
 * @property-read bool $is_archivable
 * @property-read IntegrationInstance $integrationInstance
 * @property-read StockTake $initialCountStockTake
 * @property-read array|Collection|mixed $aging_inventory
 * @property-read NominalCode $nominalCode
 */
class Warehouse extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable, UniqueFieldsInterface, AmazonFbaInboundFromInterface
{
    use Archive {
        archive as baseArchive;
        unarchived as baseUnarchive;
    }
    use HasFactory;
    use HasFilters, HasSort;

    public static $automatedWarehouses;

    /**
     * Warehouse Types.
     */
    const TYPE_DIRECT = 'direct';

    const TYPE_3PL = '3pl';

    const TYPE_SUPPLIER = 'supplier';

    const TYPE_AMAZON_FBA = 'amazon_fba';

    const TYPES = [
        self::TYPE_DIRECT,
        self::TYPE_3PL,
        self::TYPE_SUPPLIER,
        self::TYPE_AMAZON_FBA,
    ];

    const DEFAULT_WAREHOUSE_NAME = 'Default Warehouse';

    protected $casts
        = [
            'archived_at' => 'datetime',
            'dropship_enabled' => 'boolean',
            'direct_returns' => 'boolean',
            'customer_returns' => 'boolean',
            'is_default' => 'boolean',
            'auto_routing_enabled' => 'boolean',
        ];

    protected $fillable = [
        'name',
        'type',
        'supplier_id',
        'order_fulfillment',
        'dropship_enabled',
        'direct_returns',
        'customer_returns',
        'address_id',
        'integration_instance_id',
        'auto_routing_enabled',
        'nominal_code_id'
    ];

    public static function getUniqueFields(): array
    {
        return ['name', 'type', 'supplier_id', 'integration_instance_id'];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function inventoryMovements(): HasMany
    {
        return $this->hasMany(InventoryMovement::class);
    }

    public function salesCredits(): HasMany
    {
        return $this->hasMany(SalesCredit::class, 'to_warehouse_id');
    }

    public function purchaseOrders(): HasMany
    {
        return $this->hasMany(PurchaseOrder::class, 'destination_warehouse_id');
    }

    public function outgoingPurchaseOrders(): HasMany
    {
        return $this->hasMany(PurchaseOrder::class, 'supplier_warehouse_id');
    }

    public function salesOrderFulfillments(): HasMany
    {
        return $this->hasMany(SalesOrderFulfillment::class);
    }

    public function salesOrderLines(): HasMany
    {
        return $this->hasMany(SalesOrderLine::class);
    }

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(IntegrationInstance::class);
    }

    public function warehouseLocations(): HasMany
    {
        return $this->hasMany(WarehouseLocation::class);
    }

    public function defaultLocation(): HasOne
    {
        return $this->hasOne(WarehouseLocation::class)->where('is_default', true);
    }

    public function sourceTransfers(): HasMany
    {
        return $this->hasMany(WarehouseTransfer::class, 'from_warehouse_id');
    }

    public function destinationTransfers(): HasMany
    {
        return $this->hasMany(WarehouseTransfer::class, 'to_warehouse_id');
    }

    public function withDefaultLocation(): self{
        WarehouseLocation::query()
            ->create([
                'warehouse_id' => $this->id,
                'is_default' => true,
                'aisle' => 'Default',
            ]);
        return $this;
    }

    /**
     * @return MorphMany
     */
    public function address(): BelongsTo
    {
        return $this->belongsTo(Address::class);
    }

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function supplierInventory(): HasMany
    {
        return $this->hasMany(SupplierInventory::class);
    }

    public function fromWarehouseTransfers(): HasMany
    {
        return $this->hasMany(WarehouseTransfer::class, 'from_warehouse_id');
    }

    public function toWarehouseTransfers(): HasMany
    {
        return $this->hasMany(WarehouseTransfer::class, 'to_warehouse_id');
    }

    public function inventoryAssemblies(): HasMany
    {
        return $this->hasMany(InventoryAssembly::class, 'warehouse_id');
    }

    public function fifoLayers(): HasMany
    {
        return $this->hasMany(FifoLayer::class);
    }

    public function nominalCode(): BelongsTo
    {
        return $this->belongsTo(NominalCode::class);
    }

    public function initialCountStockTake(): HasOne
    {
        return $this->hasOne(StockTake::class)->where('is_initial_count', true);
    }

    public function isSupplierWarehouse(): bool
    {
        return $this->supplier_id != null;
    }

    public function isAmazonFBA(): bool
    {
        return $this->type === self::TYPE_AMAZON_FBA;
    }

    public function currentStockLevelForProduct($productId): int
    {
        return $this->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('product_id', $productId)
            ->pluck('quantity')
            ->sum();
    }

    public function initialInventoryForProduct($productId, $unitCost = 0): array
    {
        $results = $this->initialCountStockTake?->stockTakeItems
            ->filter(fn (StockTakeItem $stockTakeItem) => $stockTakeItem->product_id === $productId)
            ->map(function (StockTakeItem $stockTakeItem) use ($unitCost) {
                return [
                    'quantity' => $stockTakeItem->qty_counted,
                    'unit_cost' => $stockTakeItem->unit_cost ?? $unitCost,
                ];
            });

        if (!$results || $results->isEmpty()) {
            return [
                'quantity' => 0,
                'unit_cost' => 0,
            ];
        }

        return [
            'quantity' => $results->sum('quantity'),
            'unit_cost' => $results->sum('unit_cost'),
        ];
    }

    public function setAddressId($id)
    {
        $this->address_id = $id;
        $this->save();
    }

    public function getSupplierNameAttribute()
    {
        return $this->supplier->name ?? null;
    }

    public function getIsDropshipAttribute()
    {
        return $this->supplier_id && $this->dropship_enabled;
    }

    public function getIsFbaAttribute()
    {
        return $this->type == self::TYPE_AMAZON_FBA;
    }

    public function setTypeAttribute($type)
    {
        if (strtolower($type) === strtolower(self::TYPE_3PL)) {
            $type = self::TYPE_3PL;
        }
        $this->attributes['type'] = $type;
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * {@inheritDoc}
     */
    public function delete()
    {
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return DB::transaction(function () {
            if ($this->isSupplierWarehouse()) {
                $this->load('supplierInventory');
            }

            // If it's supplier warehouse,
            // we delete the supplier inventory
            if ($this->isSupplierWarehouse()) {
                $this->supplierInventory()->delete();
            }

            // Remove from warehouse priority
            $this->removeFromWarehousePriority();

            if ($deleted = parent::delete()) {
                // Delete address
                $this->address?->delete();
            }

            return $deleted;
        });
    }

    /**
     * @throws DropshipWithOpenOrdersException
     */
    public function save(array $options = []): bool
    {
        // If removing dropship status, make sure no
        // open dropship orders exist
        if ($this->isSupplierWarehouse()) {
            if($this->isDirty('dropship_enabled') && !$this->dropship_enabled){
                $openOrderLines = $this->salesOrderLines()
                    ->whereHas('salesOrder', function (Builder $builder) {
                        $builder->where('order_status', SalesOrder::STATUS_OPEN);
                    })->count();
                if ($openOrderLines > 0) {
                    throw new DropshipWithOpenOrdersException;
                }
            }

            // Ensure supplier warehouse has supplier name appended.
            $nameSplit = explode('-', $this->name);
            if(!empty($nameSplit) && $nameSplit[0] !== $this->supplier_name){
                $this->name = $this->supplier_name . '-' .$this->name;
            }
        }

        /** @var Warehouse|null $warehouse */
        if($warehouse = self::query()->where('name', $this->name)->first()){
            // If the name exists, we simply set the id to force an update.
            $this->id = $warehouse->id;
        }

        return parent::save($options);
    }

    /**
     * Removes warehouse from warehouse priority.
     */
    public function removeFromWarehousePriority()
    {
        $setting = Setting::with([])->where('key', Setting::KEY_WAREHOUSE_PRIORITY)->first();
        $priority = $setting ? json_decode($setting->value) : [];

        if ($setting) {
            $setting->value = json_encode(
                array_values(array_diff($priority, [$this->id]))
            );
            $setting->save();
        }
    }

    /**
     * Gets the aging inventory info for the warehouse.
     *
     * @return Collection|\Illuminate\Support\Collection
     */
    public function getAgingInventoryAttribute()
    {
        return $this->fifoLayers
            ->map(function (FifoLayer $fifoLayer) {
                return [
                    'id' => $fifoLayer->id,
                    'aging_quantity' => $fifoLayer->available_quantity,
                    'unit_cost' => $fifoLayer->avg_cost,
                    'date' => $fifoLayer->fifo_layer_date,
                ];
            });
    }

    public function hasStockAvailable(): bool
    {
        return ProductInventory::with([])->where('warehouse_id', $this->id)
            ->sum('inventory_available') > 0;
    }

    /**
     * @return mixed
     */
    public function archive(?callable $callback = null)
    {
        // If warehouse has stock, we prevent archiving it.
        // SKU-2525
        if ($this->hasStockAvailable()) {
            return __('messages.warehouse.not_archived_has_stock', [
                'resource' => 'warehouse',
                'id' => $this->name,
            ]);
        }

        return DB::transaction(function () use ($callback) {
            // Archive the warehouse
            $this->baseArchive($callback);

            // Remove from warehouse priority
            $this->removeFromWarehousePriority();

            return true;
        });
    }

    /**
     * @return mixed
     */
    public function unarchived(?callable $callback = null)
    {
        return DB::transaction(function () use ($callback) {
            // Unarchive the warehouse
            $this->baseUnarchive($callback);

            // Add to end of warehouse priority
            $this->appendToWarehousePriority();

            return true;
        });
    }

    public function appendToWarehousePriority()
    {
        $setting = Setting::with([])->where('key', Setting::KEY_WAREHOUSE_PRIORITY)->first();
        $priority = $setting ? json_decode($setting->value) : [];

        if (! in_array($this->id, $priority)) {
            $priority[] = $this->id;
        }

        $setting->value = json_encode(array_values($priority));
        $setting->save();
    }

    public function isUsed()
    {
        $relatedRelations = [
            'salesOrderLines',
            'purchaseOrders',
            'inventoryMovements',
            'fromWarehouseTransfers',
            'toWarehouseTransfers',
            'integrationInstance',
        ];

        $this->loadCount($relatedRelations);

        $usage = [];

        foreach ($relatedRelations as $relatedRelation) {
            // property count name
            $countKeyName = str_replace('-', '_', Str::kebab($relatedRelation)).'_count';
            if ($this->{$countKeyName}) {
                $relatedName = Str::singular(str_replace('-', ' ', Str::kebab($relatedRelation)));

                $usage[$relatedRelation] = trans_choice('messages.currently_used', $this->{$countKeyName}, [
                    'resource' => $relatedName,
                    'model' => 'warehouse('.$this->name.')',
                ]);
            }
        }

        return count($usage) ? $usage : false;
    }

    public function isArchivable()
    {
        if (isset($this->integration_instance_id)) {
            return ['fbaWarehouse' => trans('messages.warehouse.not_archived_fba', ['name' => $this->name])];
        }

        return true;
    }

    public static function getAutomatedWarehouses(bool $reload = false)
    {
        if (! $reload && isset(static::$automatedWarehouses)) {
            return static::$automatedWarehouses;
        }

        $shippingProviderInstances = IntegrationInstance::with(['integration'])->whereHas('integration', function (Builder $builder) {
            $builder->where('integration_type', 'shipping_provider');
        })->get();

        static::$automatedWarehouses = [];
        foreach ($shippingProviderInstances as $shippingProviderInstance) {
            $integrationSettings = $shippingProviderInstance->integration_settings;
            foreach ($integrationSettings['fulfillment']['automatedWarehousesIds'] ?? [] as $automatedWarehouseId) {
                static::$automatedWarehouses[$automatedWarehouseId] = [
                    'warehouse_id' => $automatedWarehouseId,
                    'integration_instance_id' => $shippingProviderInstance->id,
                    'integration_name' => $shippingProviderInstance->integration->name,
                ];
            }
        }

        return static::$automatedWarehouses;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.warehouse.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name', 'type'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new WarehouseDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return WarehouseDataImporter::getExportableFields();
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeStandard(Builder $builder): Builder
    {
        return $builder->whereIn('type', [
            self::TYPE_DIRECT,
            self::TYPE_3PL,
        ]);
    }

    public function scopeFilterOwned(Builder $builder): Builder
    {
        return $builder->whereIn('type', [
            self::TYPE_DIRECT,
            self::TYPE_3PL,
            self::TYPE_AMAZON_FBA
        ]);
    }
}
