<?php

namespace App\Models;

use App\Models\Concerns\Has2FAAuth;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

/**
 * Class User.
 *
 *
 * @property int $id
 * @property string $name
 * @property string $email
 * @property string $password
 * @property string|null $remember_token
 * @property Carbon|null $email_verified_at
 * @property string|null $activation_token
 * @property Carbon|null $activated_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $deleted_at
 */
class User extends Authenticatable implements Filterable, Sortable
{
    use Has2FAAuth;
    use HasApiTokens;
    use HasFactory;
    use HasFilters;
    use Notifiable;
    use SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'is_power_user',
        'is_admin',
        'last_login_at',
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'activated_at' => 'datetime',
        'is_admin' => 'boolean',
        'is_power_user' => 'boolean',
    ];

    /**
     * Get the identifier that will be stored in the subject claim of the JWT.
     *
     * @return mixed
     */
    public function getJWTIdentifier()
    {
        return $this->getKey();
    }

    /**
     * Return a key value array, containing any custom claims to be added to the JWT.
     */
    public function getJWTCustomClaims(): array
    {
        return [];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function alertSettings()
    {
        return $this->hasMany(UserAlertSetting::class);
    }

    public function alerts()
    {
        return $this->hasMany(UserAlert::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeListenAlert($query, string $alertType)
    {
        return $query->whereHas('alertSettings', function ($query) use ($alertType) {
            $query->where('alert_type', $alertType);
        });
    }

    public function scopeWithoutMe($query)
    {
        return $query->where('id', '!=', auth()->id());
    }

    public function newQuery(): Builder
    {
        return parent::newQuery()->withTrashed();
    }

    public function availableColumns()
    {
        return config('data_table.user.columns');
    }
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return [
            'name',
            'email'
        ];
    }


}
