<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Arr;

trait Upsert
{
    /**
     * Insert new record or update the existing one.
     *
     * @param  array  $uniqueBy just required if you want to set the id when updating
     * @param  array|null  $update null means update all attributes
     */
    public function upsertModel(array $uniqueBy = [], ?array $update = null, bool $setId = true, array $options = []): bool
    {
        // if the model already exists, we just save it (update)
        if ($this->exists) {
            return $this->save($options);
        }

        $query = $this->newModelQuery();

        $saved = $this->performUpsert($query, $uniqueBy, $update, $setId);

        if (! $this->getConnectionName() &&
            $connection = $query->getConnection()) {
            $this->setConnection($connection->getName());
        }

        if ($saved) {
            $this->finishSave($options);
        }

        return $saved;
    }

    /**
     * Perform "insert...on duplicate key update" query and get the id of the model
     */
    protected function performUpsert(Builder $query, array $uniqueBy = [], ?array $update = null, bool $setId = true): bool
    {
        // we will consider the model is a new model
        if ($this->fireModelEvent('creating') === false) {
            return false;
        }

        // touch the creation and update timestamps on this model
        if ($this->usesTimestamps()) {
            $this->updateTimestamps();
        }

        $attributes = $this->getAttributesForInsert();

        if (empty($attributes)) {
            return true;
        }

        // "insert...on duplicate key update" query
        $result = $query->upsert($attributes, $uniqueBy, $update);

        // inserted
        if ($result == 1) {
            // If the model has an incrementing key, we can use the "lastInsertId" method on
            // the query builder, which will give us back the final inserted ID for this
            // table from the database. Not all tables have to be incrementing though.
            if ($setId && $this->getIncrementing()) {
                $keyName = $this->getKeyName();
                $id = $query->getConnection()->getPdo()->lastInsertId($keyName);
                $this->setAttribute($keyName, is_numeric($id) ? (int) $id : $id);
            }

            $this->wasRecentlyCreated = true;

            $this->fireModelEvent('created', false);
        }

        // updated
        if ($result == 2) {
            // If the model updated, and it has an incrementing key, we can use the "uniqueBy" columns to get the model id.
            if ($setId && ! empty($uniqueBy) && $this->getIncrementing()) {
                $keyName = $this->getKeyName();
                $this->setAttribute($keyName, $this->newModelQuery()->where(Arr::only($attributes, $uniqueBy))->value($keyName));
            }
            // fire "updated" event, and we will keep the "wasRecentlyCreated" value to false
            $this->fireModelEvent('updated', false);
        }

        // set the exists property to true.
        $this->exists = true;

        return true;
    }
}
