<?php

namespace App\Models;

use App\Enums\TrackedJobLogStatusEnum;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphTo;

/**
 * @property int $id
 * @property ?string $job_id
 * @property string $job_type
 * @property string $queue
 * @property string $job_name
 * @property TrackedJobLogStatusEnum $status
 * @property ?string $related_model_type
 * @property ?int $related_model_id
 * @property ?string $messages
 * @property array $job_data
 * @property ?Carbon $attempted_at
 * @property ?Carbon $completed_at
 * @property float $duration // in seconds to 2 decimal places
 * @property ?Carbon $created_at
 * @property ?Carbon $updated_at
 * @property-read Model $relatedModel
 */
class TrackedJobLog extends Model implements Filterable, Sortable
{
    use HasFactory;
    use HasFilters;
    use HasSort;

    protected $casts = [
        'job_data' => 'array',
        'status' => TrackedJobLogStatusEnum::class,
    ];

    protected $guarded = [];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function relatedModel(): MorphTo
    {
        return $this->morphTo('related_model');
    }

    /*
    |--------------------------------------------------------------------------
    | DataTable methods
    |--------------------------------------------------------------------------
    */

    public function availableColumns()
    {
        return config('data_table.tracked_job_log.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return ['id', 'job_name'];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }
}
