<?php

namespace App\Models;

use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphTo;

/**
 * Class Tag.
 *
 *
 * @property int $id
 * @property string $name
 * @property float $rate
 * @property int $accounting_integration_id
 * @property string $accounting_integration_type
 * @property-read mixed $accountingIntegration
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 */
class TaxRate extends Model implements Filterable, Sortable
{
    use Archive, HasFactory, HasFilters, HasSort;

    protected $fillable = [
        'name',
        'rate',
        'accounting_integration_id',
        'accounting_integration_type',
    ];

    /**
     * Casting.
     */
    protected $casts = [
        'rate' => 'float',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesOrders()
    {
        return $this->hasMany(SalesOrder::class);
    }

    public function salesOrderLines()
    {
        return $this->hasMany(SalesOrderLine::class);
    }

    public function purchaseOrders()
    {
        return $this->hasMany(PurchaseOrder::class);
    }

    public function purchaseOrderLines()
    {
        return $this->hasMany(PurchaseOrderLine::class);
    }

    public function salesCredits()
    {
        return $this->hasMany(SalesCredit::class);
    }

    public function salesCreditLines()
    {
        return $this->hasMany(SalesCreditLine::class);
    }

    public function taxRateLines()
    {
        return $this->hasMany(TaxRateLine::class);
    }

    public function accountingIntegration(): MorphTo
    {
        return $this->morphTo('accountingIntegration');
    }

    /**
     * Other functions
     */
    /**
     * Get available columns to show in datatable.
     */
    public function availableColumns(): array
    {
        return config('data_table.tax_rate.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name', 'rate'];
    }

    /**
     * Get available columns to show in datatable.
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }
}
