<?php

namespace App\Models\TaskStatus;

use App\Models\User;
use Auth;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * Class TaskStatus.
 *
 *
 * @property int $id
 * @property int $user_id
 * @property string $title
 * @property string $info
 * @property string $status
 * @property int $progress_start
 * @property int $progress_end
 * @property int $progress
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read User $user
 */
class TaskStatus extends Model
{
    /*
     * This class reports the progress of a background task to the UI.
     *
     * First, the controller must create a TaskStatus object and return it's ID
     * to the frontend:
     *
     * $taskinfo = new TaskStatus();
     * $taskinfo->title = "Importing File";
     * $taskinfo->subtitle = "";
     *
     * return response()->json( [ 'task_id' => $taskinfo->id ] );
     *
     * But before returning, the controller will also dispatch a background
     * task that will be reporting to that TaskStatus object:
     *
     * dispatch(new ImportListings(\Auth::user()->id, $filepath, $taskinfo->id));
     *
     * That task will re-create the TaskStatus object:
     *
     * $this->taskinfo = TaskStatus::where('user_id', $owner_user_id)->find($taskinfo_id);
     *
     * and then it will be using its methods to report to it.
     *
     * $this->taskStatus()->addMessage("File uploaded successfully.");
     * $this->taskStatus()->progress_start = 0;
     * $this->taskStatus()->progress_end = 100;
     * $this->taskStatus()->progress = 0; // Adds bar
     * $this->taskStatus()->progress = null; // Removes bar
     * $this->taskStatus()->progress = 30; // Manually sets bar
     * $this->taskStatus()->progressAdvance(); // Sets bar +1
     * $this->taskStatus()->addErrorMessage("Error message.  Will appear in red.");
     * $this->taskStatus()->status = TaskStatus::STATUS_FAILED;
     * $this->taskStatus()->status = TaskStatus::STATUS_COMPLETED;
     *
     * Then the frontend should display a TaskProgress Vue component:
     *
     *   <TaskStatus
     *      v-if="task_id"
     *      :id="task_id"
     *      @error="catchError"
     *      @success="onCompleted"
     *   />
     *
     * And the background can make changes to the task progress
     * and the widget would show it in the UI.
     */

    const STATUS_CREATED = 'created';

    const STATUS_STARTED = 'started';

    const STATUS_IDLE = 'idle';

    const STATUS_BLOCKED = 'blocked';

    const STATUS_COMPLETED = 'completed';

    const STATUS_FAILED = 'failed';

    const ALLOWED_STATUSES = [
        'created',
        'started',
        'idle',
        'blocked',
        'completed',
        'failed',
    ];

    // Use separate connection just in case we need to
    // report live in the middle of a transaction.
    // TODO:: This is causing problems in tests, will revisit this.
//    protected $connection = 'mysql_taskstatus';

    protected $table = 'task_status';

    protected $status_object = [];

    protected $formatted = false;

    public $output_to_stdout = false;

    protected static function boot()
    {
        static::saving(function ($model) {
            if (is_null($model->user_id) || ! isset($model->user_id)) {
                if (is_null(Auth::user())) {
                    $model->user_id = 1;
                } else {
                    $model->user_id = Auth::user()->id;
                }
            }
            $model->info = json_encode($model->status_object, JSON_PRETTY_PRINT);
        });
        parent::boot();
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function getStatus()
    {
        $this->format();

        return $this->status_object;
    }

    public function addMessage($message, $options = [])
    {
        $this->format();
        $message_object = ['time' => date('Y-m-d H:i:s'), 'message' => $message];
        $prefix = '';

        if (! empty($options['class']) && is_string($options['class'])) {
            $message_object['class'] = $options['class'];
            $prefix = '['.strtoupper($options['class']).'] ';
        }

        $this->status_object['messages'][] = $message_object;
        $this->save();
        if ($this->output_to_stdout) {
            if (is_string($message)) {
                echo $prefix.$message."\n";
            } else {
                echo $prefix."\n";
                var_export($message);
                echo "\n";
            }
        }

        return $this;
    }

    public function addErrorMessage($message)
    {
        return $this->addMessage($message, ['class' => 'error']);
    }

    public function addWarningMessage($message)
    {
        return $this->addMessage($message, ['class' => 'warning']);
    }

    public function addSuccessMessage($message)
    {
        return $this->addMessage($message, ['class' => 'success']);
    }

    public function setTitleAttribute($title)
    {
        $this->format();
        $this->status_object['title'] = $title;
        $this->save();
    }

    public function setSubtitleAttribute($title)
    {
        $this->format();
        $this->status_object['subtitle'] = $title;
        $this->save();
    }

    public function setStatusAttribute($status)
    {
        if (! in_array($status, self::ALLOWED_STATUSES)) {
            throw new TaskStatusException('Error: Trying to set an unrecognized status code.');
        }
        $this->format();
        $this->status_object['status'] = $status;
        $this->save();
    }

    /**
     * @throws TaskStatusException
     */
    public function complete()
    {
        $this->setStatusAttribute(self::STATUS_COMPLETED);
    }

    public function setProgressStartAttribute($start)
    {
        $this->format();
        $this->status_object['progress_start'] = $start;
    }

    public function setProgressEndAttribute($end)
    {
        $this->format();
        $this->status_object['progress_end'] = $end;
    }

    public function setProgressAttribute($current_progress)
    {
        $this->format();
        if ($current_progress === null && isset($this->status_object['progress'])) {
            unset($this->status_object['progress']);
        } else {
            $this->status_object['progress'] = $current_progress;
        }
        $this->save();
    }

    public function progressAdvance()
    {
        $this->format();
        if (! isset($this->status_object['progress'])) {
            $this->status_object['progress'] = 0;
        }
        $this->status_object['progress']++;
        $this->save();
    }

    public function reset()
    {
        $this->formatted = false;
        $this->info = '{}';
        $this->format();

        return $this;
    }

    protected function format()
    {
        if ($this->formatted) {
            return;
        }

        $this->formatted = true;

        $this->status_object = json_decode($this->info, true);

        if (empty($this->status_object)) {
            $this->status_object = [];
        }
        if (empty($this->status_object['messages'])) {
            $this->status_object['messages'] = [];
        }
        if (empty($this->status_object['status'])) {
            $this->status_object['status'] = 'default';
        }

        return $this;
    }
}
