<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Helpers;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\Pivot;
use Illuminate\Support\Facades\DB;

/**
 * Class SupplierProduct.
 *
 *
 * @property int $id
 * @property int $product_id
 * @property int $supplier_id
 * @property string $supplier_sku
 * @property int $leadtime
 * @property float $minimum_order_quantity
 * @property float $product_minimum_order_quantity
 * @property bool $is_default
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Product $product
 * @property Supplier $supplier
 * @property SupplierProductPricing $supplierProductPricing
 * @property SupplierProductPricing $default_supplier_pricing
 * @property int $product_leadtime
 */
class SupplierProduct extends Pivot implements Filterable, UniqueFieldsInterface
{
    use BulkImport, HasFactory, HasFilters;

    public $incrementing = true;

    protected $table = 'supplier_products';

    protected $casts = [
        'minimum_order_quantity' => 'float',
        'is_default' => 'boolean',
    ];

    protected $fillable = [
        'supplier_id',
        'product_id',
        'supplier_sku',
        'leadtime',
        'minimum_order_quantity',
        'is_default',
        'default_tax_rate_id',
    ];

    protected $touches = ['product'];

    public static function getUniqueFields(): array
    {
        return ['product_id', 'supplier_id'];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }

    public function supplierPricingTiers()
    {
        return $this->belongsToMany(SupplierPricingTier::class, 'supplier_product_pricing', 'supplier_product_id', 'supplier_pricing_tier_id')
            ->using(SupplierProductPricing::class)
            ->withTimestamps()
            ->withPivot('price');
    }

    public function supplierProductPricing()
    {
        return $this->hasMany(SupplierProductPricing::class, 'supplier_product_id');
    }

    public function supplierInventory()
    {
        return $this->hasOne(SupplierInventory::class, 'product_id', 'product_id')
            ->whereColumn('supplier_products.supplier_id', 'supplier_inventory.supplier_id');
    }

    public function getDefaultSupplierPricing($defaultSupplierPricingTierId = null): ?SupplierProductPricing
    {
        if (!$defaultSupplierPricingTierId)
        {
            $defaultSupplierPricingTierId = SupplierPricingTier::default()->id;
        }
        return $this->supplierProductPricing->firstWhere('supplier_pricing_tier_id', $defaultSupplierPricingTierId);
    }

    public function save(array $options = [])
    {
        if ($this->is_default) {
            // reset default supplier to product
            if (($this->exists && ! $this->getOriginal('is_default')) || ! $this->exists) {
                self::with([])->where('product_id', $this->product_id)->update(['is_default' => false]);
            }
        }

        return parent::save($options);
    }

    public function setPricing(array $prices, bool $detaching)
    {
        if ($detaching) {
            $this->supplierPricingTiers()->sync($prices);
        } else {
            foreach ($prices as $pricing) {
                $operation = $pricing['operation'] ?? Helpers::OPERATION_UPDATE_CREATE;
                if ($operation == Helpers::OPERATION_UPDATE_CREATE) {
                    $this->supplierPricingTiers()->sync([$pricing['supplier_pricing_tier_id'] => ['price' => $pricing['price']]], false);
                } elseif ($operation == Helpers::OPERATION_DELETE) {
                    $this->supplierPricingTiers()->detach($pricing['supplier_pricing_tier_id']);
                }
            }
        }
    }

    /**
     * Gets the product's leadtime.
     *
     * @return int|mixed|null
     */
    public function getProductLeadtimeAttribute()
    {
        // The hierarchy is to find leadtime on the product, then at the supplier level,
        // then at the global level
        if ($this->leadtime) {
            return $this->leadtime;
        } elseif ($this->supplier->leadtime) {
            return $this->supplier->leadtime;
        }

        // Return global leadtime
        return Helpers::setting(Setting::KEY_PO_LEAD_TIME, null, true);
    }

    /**
     * Gets the product's MOQ.
     *
     * @return int|mixed|null
     */
    public function getProductMinimumOrderQuantityAttribute()
    {
        // The hierarchy is to find leadtime on the product, then at the supplier level,
        // then at the global level
        if ($this->minimum_order_quantity) {
            return $this->minimum_order_quantity;
        } elseif ($this->supplier->minimum_order_quantity) {
            return $this->supplier->minimum_order_quantity;
        }

        // Return global MOQ
        return Helpers::setting(Setting::KEY_SUPPLIER_MOQ, null, true);
    }

    public function delete()
    {
        // must delete it after fetch it from DB to check is_default attribute
        if (empty($this->attributes[$this->getKeyName()])) {
            return $this->newQueryForRestoration($this->getQueueableId())->first()->delete();
        }

        return DB::transaction(function () {
            // If default supplier, we set the next supplier as default if any
            if ($this->is_default) {
                $nextSupplierProduct = self::with([])->where('id', '!=', $this->id)
                    ->where('product_id', $this->product_id)->first();
                if ($nextSupplierProduct) {
                    $nextSupplierProduct->is_default = 1;
                    $nextSupplierProduct->save();
                }
            }

            // Delete all supplier inventory for this supplier product.
            SupplierInventory::with([])->where('supplier_id', $this->supplier_id)->where('product_id', $this->product_id)->delete();

            return parent::delete();
        });
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return [];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return ['product_name', 'product_sku'];
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return [];
    }
}
