<?php

namespace App\Models;

use App\Exporters\MapsExportableFields;
use App\Importers\DataImporter;
use App\Importers\DataImporters\SupplierPricingTierDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

/**
 * Class SupplierPricingTier.
 *
 *
 * @property int $id
 * @property string $name
 * @property string $currency_code
 * @property string $supplier_id
 * @property bool $is_default
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $archived_at
 */
class SupplierPricingTier extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable
{
    use Archive, HasFilters, HasSort;
    use HasFactory;

    protected $casts = ['is_default' => 'boolean'];

    protected $fillable = ['name', 'supplier_id', 'currency_code', 'is_default'];

    private static $default = null;

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function supplierProducts()
    {
        return $this->belongsToMany(SupplierProduct::class, 'supplier_product_pricing')
            ->using(SupplierProductPricing::class)
            ->withTimestamps()
            ->withPivot('price');
    }

    public function supplierProductPrices()
    {
        return $this->hasMany(SupplierProductPricing::class);
    }

    public function supplier(): BelongsToMany
    {
        return $this->belongsToMany(Supplier::class, 'supplier_to_pricing_tier');
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * {@inheritDoc}
     */
    public function save(array $options = [])
    {
        if (empty($this->currency_code)) {
            $this->currency_code = Currency::default()->code;
        }

        if ($this->is_default) {
            // reset default product pricing tier
            if (($this->exists && ! $this->getOriginal('is_default')) || ! $this->exists) {
                self::with([])->update(['is_default' => false]);
            }
        }

        return parent::save($options);
    }

    /**
     * {@inheritDoc}
     */
    public function delete()
    {
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        $this->supplierProducts()->detach();

        return parent::delete();
    }

    public function isUsed()
    {
        $usage = [];
        if ($this->is_default) {
            $usage['is_default'] = __('messages.supplier.delete_pricing_failed', ['id' => $this->name]);
        }

        $this->loadCount('supplierProductPrices');
        if ($this->supplier_product_prices_count) {
            $usage['supplierProducts'] = trans_choice('messages.currently_used', $this->supplier_product_prices_count, [
                'resource' => 'supplier product',
                'model' => 'supplier pricing tier('.$this->name.')',
            ]);
        }

        // is it used in integration instance import mappings
        $mappingCount = DataImportMapping::query()->whereJsonContains('mapping',
            ['sku_field' => "supplier_pricing.{$this->name}.value"])->count();
        if ($mappingCount) {
            $usage['import_mappings'] = trans_choice('messages.currently_used', $mappingCount, [
                'resource' => 'integration instance listing mapping',
                'model' => 'supplier pricing tier('.$this->name.')',
            ]);
        }

        return count($usage) ? $usage : false;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.supplier_pricing_tier.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name', 'currency_code', 'supplier_id'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    /**
     * Get default supplier pricing tier.
     *
     * if the user does not set default supplier pricing tier, return null
     */
    public static function default(): self|null
    {
        if (! static::$default) {
            static::$default = self::with([])->where('is_default', true)->first();
        }

        return static::$default;
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new SupplierPricingTierDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return SupplierPricingTierDataImporter::getExportableFields();
    }
}
