<?php
/**
 * Created by PhpStorm.
 * User: brightantwiboasiako
 * Date: 7/17/20
 * Time: 10:25 AM.
 */

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Exporters\MapsExportableFields;
use App\Importers\DataImporter;
use App\Importers\DataImporters\SupplierInventoryDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * Class SupplierInventory.
 *
 * @property-read int $id
 * @property int $product_id
 * @property int $supplier_id
 * @property int $warehouse_id
 * @property int $quantity
 * @property bool $in_stock
 * @property bool $discontinued
 * @property Carbon $eta
 * @property string $source
 * @property Product $product
 * @property Supplier $supplier
 * @property Warehouse $warehouse
 * @property SupplierProduct $supplierProduct
 * @property-read float|string $stock
 * @property-read string $supplier_sku
 */
class SupplierInventory extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable, UniqueFieldsInterface
{
    use Archive, BulkImport, HandleDateTimeAttributes, HasFactory, HasFilters, HasSort;

    const SOURCE_USER = 'user';

    const SOURCE_API = 'api';

    const SOURCES = [
        self::SOURCE_USER,
        self::SOURCE_API,
    ];

    /**
     * @var string
     */
    protected $table = 'supplier_inventory';

    protected $fillable = ['warehouse_id', 'quantity', 'in_stock', 'discontinued', 'eta', 'source', 'supplier_id', 'product_id'];

    protected $casts = ['in_stock' => 'boolean', 'discontinued' => 'boolean'];

    public static function getUniqueFields(): array
    {
        return ['product_id', 'supplier_id', 'warehouse_id'];
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    /**
     * @return mixed|null
     */
    public function getSupplierSkuAttribute()
    {
        $supplierProduct = SupplierProduct::with([])->where([
            'product_id' => $this->product_id,
            'supplier_id' => $this->supplier_id,
        ])->first();

        return $supplierProduct ? $supplierProduct->supplier_sku : null;
    }

    public function getStockAttribute()
    {
        if ($this->quantity) {
            return $this->quantity;
        }

        return $this->in_stock ? 'in_stock' : 0;
    }

    public function setQuantityAttribute($quantity)
    {
        if ($quantity === '' || is_null($quantity)) {
            $this->attributes['quantity'] = null;
        } else {
            $this->attributes['quantity'] = $quantity;
        }
    }

    public function getInStockAttribute($inStock)
    {
        if (is_null($this->attributes['quantity'])) {
            return (int) ($inStock);
        }

        return null;
    }

    public function setEtaAttribute($eta)
    {
        if (empty($eta) || trim($eta) == '') {
            $this->attributes['eta'] = null;
        } // Necessary for imported inventories.
        else {
            $this->attributes['eta'] = $this->asDateTime($eta);
        }
    }

    public function save(array $options = [])
    {
        if (! $this->source) {
            $this->source = self::SOURCE_USER;
        }

        return parent::save($options);
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.supplier_inventory.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->filterableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function scopeFilterSupplier(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        if (count($keyExploded = explode('.', $relation['combined_key'])) > 2) {
            $relation = implode('.', array_slice($keyExploded, 0, 2));
            $lastKey = array_slice($keyExploded, -1)[0];

            if ($relation === 'supplier.supplierProducts') {
                $relation = 'product.supplierProducts';
            } // This allows fetching based on supplier sku

            return $builder->{$function.'Has'}($relation, function (Builder $builder) use ($lastKey, $value, $operator) {
                $builder->filterKey([
                    'key' => $builder->qualifyColumn($lastKey),
                    'is_relation' => false,
                ], $operator, $value);
            });
        }
    }

    public function scopeSortSupplier(Builder $builder, array $relation, bool $ascending): Builder
    {
        // TODO: Pending decision on complex fields
        return $builder;
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new SupplierInventoryDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return SupplierInventoryDataImporter::getExportableFields();
    }
}
