<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Exporters\MapsExportableFields;
use App\Helpers;
use App\Importers\DataImporter;
use App\Importers\DataImporters\SupplierDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Abstractions\AmazonFbaInboundFromInterface;

/**
 * Class Supplier.
 *
 *
 * @property int $id
 * @property string $name
 * @property string $company_name
 * @property string|null $primary_contact_name
 * @property string|null $email
 * @property string|null $purchase_order_email
 * @property string|null $phone
 * @property string|null $website
 * @property int|null $leadtime
 * @property float|null $minimum_order_quantity
 * @property int|null $minimum_purchase_order
 * @property string|null $purchase_order_format
 * @property string|null $currency_code
 * @property array|null $po_batch_schedule
 * @property string $po_processing_method
 * @property int $default_warehouse_id
 * @property int $default_pricing_tier_id
 * @property int $default_store_id
 * @property bool $auto_fulfill_dropship
 * @property int $auto_submit_dropship_po
 * @property bool $auto_generate_backorder_po
 * @property bool $auto_submit_backorder_po
 * @property bool $auto_split_backorder_po_by_brand
 * @property bool $auto_receive_backorder_po
 * @property array|null $backorder_po_schedule
 * @property string $default_stock_level
 * @property Collection $supplierProducts
 * @property Collection $backorderQueues
 * @property ShippingMethod $defaultShippingMethod
 * @property-read SupplierPricingTier $defaultPricingTier
 * @property Warehouse $defaultWarehouse
 * @property string $timezone
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read Address $address
 * @property-read Store $defaultStore
 */
class Supplier extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable, UniqueFieldsInterface, AmazonFbaInboundFromInterface
{
    use Archive, BulkImport, HasFilters, HasSort, Notifiable;
    use HasFactory;

    const DEFAULT_WAREHOUSE_NAME = 'Main Warehouse';

    const DEFAULT_ADDRESS_LABEL = 'Warehouse Address';

    const OFFICE_ADDRESS_LABEL = 'Office';

    const PO_DELIVERY_METHOD_INDIVIDUALLY = 'individually';

    const PO_DELIVERY_METHOD_BATCH = 'batch';

    const PO_DELIVERY_METHODS = [
        self::PO_DELIVERY_METHOD_INDIVIDUALLY,
        self::PO_DELIVERY_METHOD_BATCH,
    ];

    const STOCK_LEVEL_ZERO = '0';

    const STOCK_LEVEL_IN_STOCK = 'in_stock';

    const STOCK_LEVEL_NOT_IN_STOCK = 'not_in_stock';

    const STOCK_LEVELS = [self::STOCK_LEVEL_ZERO, self::STOCK_LEVEL_IN_STOCK, self::STOCK_LEVEL_NOT_IN_STOCK];

    protected $casts = [
        'minimum_order_quantity' => 'float',
        'po_batch_schedule' => 'array',
        'backorder_po_schedule' => 'array',
    ];

    protected $fillable = [
        'name',
        'company_name',
        'primary_contact_name',
        'email',
        'purchase_order_email',
        'website',
        'leadtime',
        'minimum_order_quantity',
        'minimum_purchase_order',
        'address_id',
        'purchase_order_format',
        'default_pricing_tier_id',
        'default_warehouse_id',
        'default_shipping_method_id',
        'default_store_id',
        'po_processing_method',
        'po_batch_schedule',
        'auto_submit_dropship_po',
        'auto_fulfill_dropship',
        'auto_generate_backorder_po',
        'auto_submit_backorder_po',
        'auto_split_backorder_po_by_brand',
        'auto_receive_backorder_po',
        'backorder_po_schedule',
        'default_stock_level',
        'default_tax_rate_id',
        'timezone',
    ];

    public static function getUniqueFields(): array
    {
        return ['name'];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function purchaseInvoices()
    {
        return $this->hasMany(PurchaseInvoice::class);
    }

    public function purchaseOrders()
    {
        return $this->hasMany(PurchaseOrder::class);
    }

    public function supplierProducts()
    {
        return $this->hasMany(SupplierProduct::class);
    }

    public function products()
    {
        return $this->belongsToMany(Product::class, 'supplier_products')
            ->withPivot('supplier_sku', 'leadtime', 'minimum_order_quantity');
    }

    public function address(): BelongsTo
    {
        return $this->belongsTo(Address::class);
    }

    public function pricingTiers(): BelongsToMany
    {
        return $this->belongsToMany(SupplierPricingTier::class, 'supplier_to_pricing_tier');
    }

    public function warehouses()
    {
        return $this->hasMany(Warehouse::class);
    }

    public function defaultStore()
    {
        return $this->belongsTo(Store::class);
    }

    public function backorderQueues()
    {
        return $this->hasMany(BackorderQueue::class);
    }

    public function defaultWarehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class, 'default_warehouse_id');
    }

    public function defaultPricingTier(): BelongsTo
    {
        return $this->belongsTo(SupplierPricingTier::class, 'default_pricing_tier_id');
    }

    public function defaultShippingMethod(): BelongsTo
    {
        return $this->belongsTo(ShippingMethod::class, 'default_shipping_method_id');
    }

    public function getPurchaseOrderEmail(): array
    {
        if (empty($this->purchase_order_email ?: $this->email)) {
            return [];
        }

        $emails = [];
        foreach (explode(',', $this->purchase_order_email ?: $this->email) as $email) {
            $emails[$email] = $this->name;
        }

        return $emails;
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getDefaultAddressAttribute()
    {
        return $this->address;
    }

    public function getAllAddressesAttribute()
    {
        return collect($this->address)->toArray();
    }

    public function handleSchedule($schedule): ?array
    {
        if (empty($schedule)) {
            return null;
        }

        foreach ($schedule as $key => $scheduleItem) {
            $schedule[$key]['time'] = Helpers::timeToTimezone($scheduleItem['time'], Helpers::getAppTimezone(), config('app.timezone'));

            $localTime = now(Helpers::getAppTimezone())->setTimeFromTimeString($scheduleItem['time']);

            $localTimeInUtc = $localTime->copy()->timezone('UTC');
            $dayDifference = $localTimeInUtc->day - $localTime->day;

            // If the UTC time is on the next day, adjust the day of week.
            if ($dayDifference > 0) {
                foreach ($scheduleItem['days'] as $i => $day) {
                    // Parse the day, add one day and get the new day name.
                    $newDay = Carbon::parse($day)->addDay()->format('l');

                    // Lowercase and replace the original day.
                    $schedule[$key]['days'][$i] = strtolower($newDay);
                }
            } elseif ($dayDifference < 0) {  // If the UTC time is on the previous day, adjust the day of week.
                foreach ($scheduleItem['days'] as $i => $day) {
                    // Parse the day, subtract one day and get the new day name.
                    $newDay = Carbon::parse($day)->subDay()->format('l');

                    // Lowercase and replace the original day.
                    $schedule[$key]['days'][$i] = strtolower($newDay);
                }
            }
        }

        return $schedule;
    }

    public function setBackorderPoScheduleAttribute($value): void
    {
        $value = $this->handleSchedule($value);
        $this->attributes['backorder_po_schedule'] = $value ? json_encode($value) : null;
    }

    public function setPoBatchScheduleAttribute($value): void
    {
        $value = $this->handleSchedule($value);
        $this->attributes['po_batch_schedule'] = $value ? json_encode($value) : null;
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function receivesOrdersIndividually(): bool
    {
        return $this->po_processing_method === self::PO_DELIVERY_METHOD_INDIVIDUALLY;
    }

    public function canSubmitOrderNow(PurchaseOrder $order): bool
    {
        return $this->auto_submit_dropship_po && $order->dropshipping;
    }

    public function save(array $options = [])
    {
        if (empty($this->timezone))
        {
            $this->timezone = Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE, null, true);
        }

        if (isset($this->name) && empty(trim($this->name))) {
            throw new \Exception("The supplier name can't be an empty string");
        }

        return parent::save($options);
    }

    /**
     * {@inheritDoc}
     */
    public function delete()
    {
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return DB::transaction(function () {
            // Delete supplier products
            $this->supplierProducts()->each(function (SupplierProduct $supplierProduct) {
                $supplierProduct->delete();
            });

            // Delete supplier inventory from every
            // warehouse of the supplier
            $this->warehouses()->each(function (Warehouse $warehouse) {
                $warehouse->supplierInventory()->delete();
                // Delete supplier warehouses
                $warehouse->delete();
            });

            // Delete supplier to pricing tier
            $this->pricingTiers()->detach();

            if ($deleted = parent::delete()) {
                // Delete supplier address
                $this->address?->delete();
            }

            return $deleted;
        });
    }

    /**
     * Determine if the supplier is used.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        $usage = [];

        // used in purchase orders
        $this->loadCount('purchaseOrders');
        if ($this->purchase_orders_count) {
            $usage['purchaseOrders'] = trans_choice('messages.currently_used', $this->purchase_orders_count, [
                'resource' => 'purchase order',
                'model' => 'supplier('.$this->name.')',
            ]);
        }

        return count($usage) ? $usage : false;
    }

    public function attachPricingTiers(array $tiers)
    {
        $this->pricingTiers()->sync($tiers);

        $this->supplierProducts->each(function (SupplierProduct $supplierProduct) use ($tiers) {
            // Sync supplier product pricing
            foreach ($tiers as $tierId) {
                $pricing = $supplierProduct->supplierProductPricing()->where('supplier_pricing_tier_id', $tierId)->first();
                if (! $pricing) {
                    // Pricing doesn't already exist, we create the pricing.
                    // This check is needed so we don't override existing prices
                    $supplierProduct->supplierProductPricing()->create([
                        'supplier_pricing_tier_id' => $tierId,
                        'price' => 0.0,
                    ]);
                }
            }

            // Delete prices for removed pricing tiers
            $supplierProduct->supplierProductPricing()->whereNotIn('supplier_pricing_tier_id', $tiers)->delete();
        });
    }

    public function getLeadtimeAttribute()
    {
        if (isset($this->attributes['leadtime']) && (! is_null($this->attributes['leadtime']) || $this->attributes['leadtime'] != '')) {
            return $this->attributes['leadtime'];
        }

        return Helpers::setting(Setting::KEY_PO_LEAD_TIME, null, true);
    }

    /**
     * Checks if the supplier already has warehouses.
     */
    public function hasWarehouses(): bool
    {
        return $this->warehouses->isNotEmpty();
    }

    public function setDefaultWarehouse($warehouseId)
    {
        $this->default_warehouse_id = $warehouseId;
        $this->save();
    }

    public function updateOrCreateAddress(array $data)
    {
        $address = $this->address;
        if (! $address) {
            $address = new Address();
        }
        $address->fill($data);
        $address->save();

        // Set address id if it's just created
        if ($address->wasRecentlyCreated) {
            $this->address_id = $address->id;
            $this->save();
        }
    }

    /**
     * Get the next backorder schedule.
     *
     * This function looks for the next scheduled time from the current time, iterating through the next seven days.
     * If the schedule for the current day has passed, it will find the next available schedule on the upcoming days.
     * The function returns null if there are no upcoming schedules in the next seven days.
     */
    public function getNextBackorderSchedule(): ?Carbon
    {
        // Return null if there is no auto_generate_backorder_po flag or if the backorder_po_schedule is empty
        if (! $this->auto_generate_backorder_po || empty($this->backorder_po_schedule)) {
            return null;
        }

        // Get the current time in the user's timezone
        $nowByUserTimezone = now(Helpers::getAppTimezone());

        // Loop over the next 7 days, including today
        for ($daysCount = 0; $daysCount <= 7; $daysCount++) {
            // Get the current day of the week in lowercase (e.g., 'monday')
            $dayString = strtolower($nowByUserTimezone->format('l'));

            // Filter the schedule to only include the times for the current day of the week
            $matched = collect($this->backorder_po_schedule)
                ->filter(function ($row) use ($dayString, $nowByUserTimezone, $daysCount) {
                    // Convert the schedule's time to the user's timezone
                    $timeByUserTimezone = Helpers::timeToTimezone($row['time'], config('app.timezone'), Helpers::getAppTimezone());

                    // Create a Carbon instance of the scheduled time
                    $scheduleTime = $nowByUserTimezone->clone()->setTimeFromTimeString($timeByUserTimezone);

                    // If we are checking the schedules for today ($daysCount == 0), we only consider times that are later than the current time.
                    // For other days, we include all the times.
                    return in_array($dayString, $row['days'] ?? []) &&
                        ($daysCount != 0 || $nowByUserTimezone->lt($scheduleTime));
                });

            // If there are any times scheduled for the current day, return the earliest one
            if ($matched->isNotEmpty()) {
                return min(
                    $matched->map(function ($row) use ($nowByUserTimezone) {
                        // Convert the scheduled time to the user's timezone and create a Carbon instance of it
                        $timeByUserTimezone = Helpers::timeToTimezone($row['time'], config('app.timezone'), Helpers::getAppTimezone());

                        return $nowByUserTimezone->clone()->setTimeFromTimeString($timeByUserTimezone)->timezone(config('app.timezone'));
                    })->toArray()
                );
            }

            // If there are no times scheduled for the current day, proceed to the next day
            $nowByUserTimezone->addDay();
        }

        // If there are no schedules for the next seven days, return null
        return null;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.supplier.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name', 'company_name', 'primary_contact_name', 'email'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new SupplierDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return SupplierDataImporter::getExportableFields();
    }
}
