<?php

namespace App\Models;

use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;

/**
 * Class StoreEmailTemplate.
 *
 *
 * @property int $id
 * @property int $store_id
 * @property string $type
 * @property string $subject
 * @property string $html_body
 * @property bool $enabled
 * @property bool $attach_pdf
 * @property string $sent_type
 * @property bool $is_system_template
 * @property array $cc
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read Store store
 */
class StoreEmailTemplate extends Model implements Filterable, Sortable
{
    use Archive, HasFilters, HasSort;
    use HasFactory;

    const SENT_TYPE_AUTOMATICALLY = 'automatically';

    const SENT_TYPE_MANUALLY = 'manually';

    const SENT_TYPES = [
        self::SENT_TYPE_AUTOMATICALLY,
        self::SENT_TYPE_MANUALLY,
    ];

    const TYPE_PURCHASE_ORDER = 'purchase_order';

    const TYPE_PURCHASE_ORDER_UPDATE = 'purchase_order_updated';

    const TYPE_BULK_PURCHASE_ORDER = 'bulk_purchase_order';

    const TYPE_SALES_ORDER_CONFIRMATION = 'sales_order_confirmation';

    const TYPE_SALES_ORDER_UPDATED = 'sales_order_updated';

    const TYPE_SALES_ORDER_FULFILLED = 'sales_order_fulfilled';

    const TYPE_SALES_CREDIT_ISSUED = 'sales_credit_issued';

    const TYPE_ASN_NOTES = 'asn_notes';

    const TYPE_ASN_NOTES_FBA = 'asn_notes_fba';

    const TYPE_CONTEXT_SALES_ORDER = 'user_sales_order';

    const TYPE_CONTEXT_PURCHASE_ORDER = 'user_sales_order';

    protected $fillable = [
        'store_id',
        'type',
        'subject',
        'html_body',
        'enabled',
        'attach_pdf',
        'sent_type',
        'cc'
    ];

    protected $casts = [
        'enabled' => 'bool',
        'cc' => 'array',
        'is_system_template' => 'bool'
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function store()
    {
        return $this->belongsTo(Store::class);
    }

    /**
     * Get available columns to show in datatable.
     */
    public function availableColumns(): array
    {
        return config('data_table.store_email_template.columns');
    }

    /**
     * Get filterable columns, must be inside available columns.
     */
    public function filterableColumns(): array
    {
        if (Str::startsWith(func_get_arg(0), [
            'store.',
        ])) {
            return func_get_args();
        }

        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['subject', 'type'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public static function fillTemplate($id, $type, $subject, $body, bool $v = false): array
    {
        $data = self::getModelVariables($id, $type);
        $variables = config('store-email-templates.'.$type);

        preg_match_all("/\{\{([^\}]*)\}\}/", $body, $matches);
        $replaces = $matches[0];
        $extractedVariables = [];
        foreach ($replaces as $replace) {
            if (array_key_exists($replace, $variables)) {
                if ($variables[$replace] == 'sales_order_lines') {
                    $orderLines = self::getSalesOrderLines($id);
                    $body = str_replace($replace, $orderLines, $body);
                    $extractedVariables[$replace] = $orderLines;

                    continue;
                }
                if ($variables[$replace] == 'purchase_order_lines') {
                    $pOrderLines = self::getPurchaseOrderLines($id);
                    $body = str_replace($replace, $pOrderLines, $body);
                    $extractedVariables[$replace] = $pOrderLines;

                    continue;
                }
                if ($variables[$replace] == 'sales_credit_lines') {
                    $scLines = self::getSalesCreditLines($id);
                    $body = str_replace($replace, $scLines, $body);
                    $extractedVariables[$replace] = $scLines;

                    continue;
                }
                if (array_key_exists($variables[$replace], $data)) {
                    $replaced = $data[$variables[$replace]];
                    if (is_array($replaced) && empty($replaced)) {
                        $replaced = null;
                    }

                    if (Str::contains($variables[$replace], 'logo')) {
                        $url = $replaced;
                        if (Str::startsWith($url, ['/images', '/storage/images'])) {
                            $url = env('APP_URL').$url;
                        }

                        $replaced = "<img src='$url' height='100px' width='100px'/>";
                    }
                    $body = str_replace($replace, $replaced, $body);
                    $extractedVariables[$replace] = $replaced;

                    continue;
                }
            }
            $extractedVariables[$replace] = '';
            $body = str_replace($replace, '', $body);
        }
        preg_match_all("/\{\{([^\}]*)\}\}/", $subject, $matches);
        $replaces = $matches[0];

        foreach ($replaces as $replace) {
            if (array_key_exists($replace, $variables) && array_key_exists($variables[$replace], $data)) {
                $subject = str_replace($replace, $data[$variables[$replace]], $subject);
                $extractedVariables[$replace] = $data[$variables[$replace]];

                continue;
            }
            $extractedVariables[$replace] = '';
            $subject = str_replace($replace, '', $subject);
        }

        if ($v) {
            return $extractedVariables;
        }

        return ['subject' => $subject, 'body' => $body];
    }

    public static function getModelVariables($id, $type)
    {
        $variables = config('store-email-templates.'.$type);

        /** @var Model $model */
        $model = self::getTemplateModel($type);

        $load = collect($variables)->filter(function ($var) {
            return Str::contains($var, '.');
        })->map(function ($var) {
            return Str::camel(explode('.', $var)[0]);
        })->values()->unique()->filter(function ($var) use ($type) {
            return $type !== 'SalesCredit' || $var !== 'shippingAddress';
        })->toArray();

        $mutators = self::getMutatorMethods($model);

        $mutators = collect($mutators)->map(function ($match) {
            return lcfirst(Str::snake($match));
        });

        $mutators = $mutators->intersect(array_values($variables ?? []))->filter(function ($q) {
            return isset($q) && strlen($q);
        })->toArray();

        $data = $model::with($load)->findOrFail($id);
        if ($type === 'SalesOrder') {
            $data->load('salesOrderLines');
        } elseif ($type === 'PurchaseOrder') {
            $data->load('purchaseOrderLines');
        }
        $data = $data->setAppends($mutators)->toArray();

        return Arr::dot($data);
    }

    public static function getModelMutator($model)
    {
        preg_match_all('/(?<=^|;)get([^;]+?)Attribute(;|$)/', implode(';', get_class_methods($model)), $matches);

        return collect($matches[1])->map(function ($match) {
            return lcfirst(Str::snake($match));
        });
    }

    /**
     * @throws Exception
     */
    public static function getTemplateModel($type)
    {
        switch ($type) {
            case 'SalesOrder':
                return SalesOrder::class;
            case 'PurchaseOrder':
                return PurchaseOrder::class;
            case 'SalesCredit':
                return SalesCredit::class;
            case 'Supplier': // For bulk purchase orders
                return Supplier::class;
            case 'WarehouseTransfer':
                return WarehouseTransfer::class;
            default:
                throw new Exception('Model '.$type.' not supported.');
        }
    }

    private static function getSalesOrderLines($id)
    {
        $salesOrderLines = SalesOrderLine::with(['product'])->where('sales_order_id', $id)->get();

        $table = "<table class='et-sales-order-table'>";
        $table .= '<thead><tr><th>SKU</th><th>Product</th><th>QTY</th><th>Price</th><th>Disc%</th><th>Tax%</th><th>Total</th></tr></thead>';
        $table .= '<tbody>';
        foreach ($salesOrderLines as $line) {
            $total = floatval($line->amount) * floatval($line->quantity);
            $sku = $line->product ? $line->product->sku : null;
            $table .= '<tr>';
            $table .= "<td>{$sku}</td>";
            $table .= "<td>{$line->description}</td>";
            $table .= "<td>{$line->quantity}</td>";
            $table .= "<td>{$line->amount}</td>";
            $table .= "<td>{$line->discount}</td>";
            $table .= "<td>{$line->tax_value}</td>";
            $table .= "<td>{$total}</td>";
            $table .= '</tr>';
        }
        $table .= '</tbody>';
        $table .= '</table>';

        return $table;
    }

    private static function getPurchaseOrderLines($id)
    {
        $purchaseOrderLines = PurchaseOrderLine::with(['product'])->where('purchase_order_id', $id)->get();

        $table = "<table class='et-purchase-order-table'>";
        $table .= '<thead><tr><th>SKU</th><th>Product</th><th>QTY</th><th>Price</th><th>Disc%</th><th>Tax%</th><th>Total</th></tr></thead>';
        $table .= '<tbody>';
        foreach ($purchaseOrderLines as $line) {
            $total = floatval($line->amount) * floatval($line->quantity);
            $sku = $line->product ? $line->product->sku : null;
            $table .= '<tr>';
            $table .= "<td>{$sku}</td>";
            $table .= "<td>{$line->description}</td>";
            $table .= "<td>{$line->quantity}</td>";
            $table .= "<td>{$line->amount}</td>";
            $table .= "<td>{$line->discount}</td>";
            $table .= "<td>{$line->tax}</td>";
            $table .= "<td>{$total}</td>";
            $table .= '</tr>';
        }
        $table .= '</tbody>';
        $table .= '</table>';

        return $table;
    }

    private static function getSalesCreditLines($id)
    {
        $salesCreditLines = SalesCreditLine::with(['product'])->where('sales_credit_id', $id)->get();

        $table = "<table class='et-purchase-order-table'>";
        $table .= '<thead><tr><th>SKU</th><th>Product</th><th>QTY</th><th>Price</th><th>Disc%</th><th>Tax%</th><th>Total</th></tr></thead>';
        $table .= '<tbody>';
        foreach ($salesCreditLines as $line) {
            $total = floatval($line->amount) * floatval($line->quantity);
            $sku = $line->product ? $line->product->sku : null;
            $table .= '<tr>';
            $table .= "<td>{$sku}</td>";
            $table .= "<td>{$line->description}</td>";
            $table .= "<td>{$line->quantity}</td>";
            $table .= "<td>{$line->amount}</td>";
            $table .= "<td>{$line->discount}</td>";
            $table .= "<td>{$line->tax}</td>";
            $table .= "<td>{$total}</td>";
            $table .= '</tr>';
        }
        $table .= '</tbody>';
        $table .= '</table>';

        return $table;
    }
}
