<?php

namespace App\Models;

use App\Abstractions\FinancialDocumentInterface;
use App\Contracts\HasReference;
use App\Data\AccountingTransactionData;
use App\Data\StockTakeItemData;
use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Exporters\BaseExporter;
use App\Exporters\MapsExportableFields;
use App\Exporters\TransformsExportData;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasAccountingTransaction;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionDataFromStockTake;
use Carbon\Carbon;
use Illuminate\Config\Repository;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Spatie\LaravelData\DataCollection;
use Throwable;

/**
 * Class StockTake.
 *
 * @property-read int $id
 * @property int $warehouse_id
 * @property string $notes
 * @property float $value_change
 * @property Carbon|string $date_count
 * @property string $status
 * @property bool is_initial_count
 * @property Warehouse $warehouse
 * @property Collection|StockTakeItem[] $stockTakeItems
 */
class StockTake extends Model implements Filterable, HasReference, FinancialDocumentInterface, MapsExportableFields, Sortable, TransformsExportData
{
    use Archive, HandleDateTimeAttributes, HasAccountingTransaction, HasFactory, HasFilters, HasSort;

    const STOCK_TAKE_STATUS_DRAFT = 'draft';

    const STOCK_TAKE_STATUS_OPEN = 'open';

    const STOCK_TAKE_STATUS_CLOSED = 'closed';

    const STOCK_TAKE_STATUSES = [
        self::STOCK_TAKE_STATUS_DRAFT,
        self::STOCK_TAKE_STATUS_OPEN,
        self::STOCK_TAKE_STATUS_CLOSED,
    ];

    /**
     * @var array
     */
    protected $fillable = ['date_count',
        'status',
        'is_initial_count',
        'warehouse_id',
        'notes',
        'value_change',
        'updated_at'];

    protected $casts = [
        'date_count' => 'datetime',
        'value_change' => 'float',
        'is_initial_count' => 'bool',
    ];

    /**
     * @throws Throwable
     */
    public function getAccountingTransactionData(): AccountingTransactionData
    {
        return (new BuildAccountingTransactionDataFromStockTake($this))->handle();
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return null;
    }

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function stockTakeItems(): HasMany
    {
        return $this->hasMany(StockTakeItem::class);
    }

    public function accountingTransaction()
    {
        return $this->morphOne(AccountingTransaction::class, 'link');
    }

    public function getReference(): ?string
    {
        return $this->id;
    }

    public function getDrawer(): Model
    {
        return $this;
    }

    public function getAccountingDateFieldName(): string
    {
        return 'date_count';
    }

    public function isClosed(): bool
    {
        return $this->status === static::STOCK_TAKE_STATUS_CLOSED;
    }

    public function isOpen(): bool
    {
        return $this->status === static::STOCK_TAKE_STATUS_OPEN;
    }

    public function isDraft(): bool
    {
        return $this->status === static::STOCK_TAKE_STATUS_DRAFT;
    }

    /**
     * Closes the stock take.
     */
    public function close()
    {
        $this->status = static::STOCK_TAKE_STATUS_CLOSED;
        $this->save();
    }

    /**
     * Opens the stock count.
     */
    public function open()
    {
        $this->status = static::STOCK_TAKE_STATUS_OPEN;
        $this->save();
    }

    public function addValueChanged($value): self
    {
        $currentValue = $this->value_change ?? 0;

        return $this->setValueChanged($currentValue + $value);
    }

    public function setValueChanged($value): static
    {
        $this->value_change = $value;
        $this->save();

        return $this;
    }

    /**
     * Reverts the stock take to draft mode.
     */
    public function revertToDraft(): static
    {
        $this->status = static::STOCK_TAKE_STATUS_DRAFT;
        $this->value_change = null;
        $this->stockTakeItems()->update(['snapshot_inventory' => null]);
        $this->save();

        return $this;
    }

    public function hasFullQuantities(): bool
    {
        return $this->stockTakeItems()->whereNull('qty_counted')->count() === 0;
    }

    /**
     * @return array|bool|mixed
     */
    public function isUsed()
    {
        if ($this->status != self::STOCK_TAKE_STATUS_CLOSED) {
            return false;
        }

        // Only limitation is that fifo
        // layers created from lines should
        // not have been used up.
        $reasons = [];
        $this->stockTakeItems->each(function (StockTakeItem $item, $index) use (&$reasons) {
            $usedFifoQuantity = $item->fifoLayers()->where('fulfilled_quantity', '>', 0)->count();
            if ($usedFifoQuantity > 0) {
                $reasons[$index] = "Stock Take Item: {$item->product->sku} has {$usedFifoQuantity} used fifo layer(s) and cannot be deleted.";
            }
        });

        return count($reasons) ? $reasons : false;
    }

    public function delete()
    {
        if ($this->status == self::STOCK_TAKE_STATUS_CLOSED) {
            $this->load('stockTakeItems.fifoLayers');
        }

        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return DB::transaction(function () {
            // Delete stock take items
            $this->stockTakeItems->each(function (StockTakeItem $item) {
                $item->setRelation('stockTake', $this);
                $item->delete();
            });

            return parent::delete();
        });
    }

    /**
     * @return Repository|mixed
     */
    public function availableColumns()
    {
        return config('data_table.stock_take.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return ['date_count', 'item_name', 'item_sku', 'notes', 'status'];
    }

    /**
     * @return mixed
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', false, 'ID'),
            'date_count' => Exporter::makeExportableField('count_date', false),
            'warehouse.name' => Exporter::makeExportableField('warehouse_name', false),
            'status' => Exporter::makeExportableField('status', false),
            'value_change' => Exporter::makeExportableField('value_change', false),
            'notes' => Exporter::makeExportableField('notes', false),
            'item_sku' => Exporter::makeExportableField('item_sku', false),
            'item_quantity' => Exporter::makeExportableField('item_quantity', false),
            'item_name' => Exporter::makeExportableField('item_name', false),
            'item_unit_cost' => Exporter::makeExportableField('item_unit_cost', false),
            'created_at' => Exporter::makeExportableField('created_at', false),
            'updated_at' => Exporter::makeExportableField('updated_at', false),
        ];
    }

    public static function transformExportData(array $data): array
    {
        return BaseExporter::groupByLines($data, 'id', 'items');
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeAccountingReady(Builder $builder): Builder
    {
        return $builder
            // TODO: The initial inventory should already be in the accounting software, but the SKU balance sheet will be off without this... need to think through this
            ->where('is_initial_count', false)
            ->where('status', self::STOCK_TAKE_STATUS_CLOSED);
    }
}
