<?php

namespace App\Models\Shopify;

use App\Collections\SalesOrderFulfillmentCollection;
use App\Console\Commands\Inventory\Integrity\CleanupExtraSalesCreditsCommand;
use App\Contracts\SalesChannels\FulfillableInterface;
use App\Data\FinancialLineData;
use App\Data\SalesOrderLineData;
use App\Data\UpdateSalesOrderData;
use App\Data\UpdateSalesOrderPayloadData;
use App\DTO\SalesChannelFulfillmentDto;
use App\DTO\SalesOrderFulfillmentDto;
use App\Enums\FinancialLineClassificationEnum;
use App\Exceptions\ExternallyFulfilledCantBeFulfilledException;
use App\Exceptions\IntegrationInstance\Shopify\UnsupportedFinancialStatusException;
use App\Exceptions\SalesOrder\InvalidProductWarehouseRouting;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentException;
use App\Helpers;
use App\Http\Resources\Shopify\ShopifyOrderResource;
use App\Integrations\SalesChannelOrder;
use App\Integrations\Shopify;
use App\Jobs\AutomatedSalesOrderFulfillmentJob;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\Currency;
use App\Models\IntegrationInstance;
use App\Models\PaymentType;
use App\Models\ProductListing;
use App\Models\ReturnReason;
use App\Models\SalesChannel;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturn;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\ShippingMethodMappingsSalesChannelToSku;
use App\Repositories\FinancialLineRepository;
use App\Repositories\SettingRepository;
use App\Repositories\Shopify\ShopifyOrderMappingRepository;
use App\Repositories\Shopify\ShopifyTransactionRepository;
use App\Repositories\WarehouseRepository;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use App\Services\SalesOrder\FulfillSalesOrderService;
use App\Services\SalesOrder\SalesOrderManager;
use App\Services\StockTake\OpenStockTakeException;
use App\Services\TaxRate\TaxRateService;
use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Throwable;

/**
 * @property int $id
 * @property array $json_object
 * @property array $fulfillmentsMap
 * @property array $refundsMap
 * @property array $refunds
 * @property int $refund_lines_count
 * @property array $fulfillments
 * @property array $transactions
 * @property array $line_items
 * @property string $taxes_included
 * @property string $name
 * @property int $number
 * @property int $shopify_order_id
 * @property int|null $sku_sales_order_id
 * @property string $error_log
 * @property SalesOrder|null $salesOrder
 * @property Carbon|null $transactions_updated_at
 * @property Carbon|null $archived_at
 * @property ?string $created_at
 * @property ?string $updated_at
 * @property Carbon|null $createdAtUtc
 * @property Carbon|null $updatedAtUtc
 * @property Carbon|null $sku_updated_at
 * @property Carbon|null $sku_order_updated_at
 * @property Carbon|null $processedAtUtc
 * @property-read IntegrationInstance $integrationInstance
 * @property-read Collection|ShopifyOrderMapping[] $orderMappings
 *
 * @method static Builder|static latestOrder(int $integrationInstanceId, string $downloadedBy = null, string $column = 'updated_at')
 * @method static dataFeedBulkImport()
 */
class ShopifyOrder extends Model implements Filterable, FulfillableInterface, SalesChannelOrder, Sortable
{
    use Archive, HandleDateTimeAttributes, HasFactory, HasFilters, HasSort;

    // used for getting listing data
    const LINE_ITEMS_QUERY_ID = 'id';

    const LINE_ITEMS_QUERY = 'line_items';

    const PRODUCT_LOCAL_QUERY_IDENTIFER = 'variant_id';

    const PRODUCT_LOCAL_FORIEN_IDENTIFER = 'variant_id';

    const DOWNLOADED_BY_WEBHOOK = 'Webhook';

    const DOWNLOADED_BY_GET_ORDERS_JOB = 'GetOrdersJob';

    const DOWNLOADED_BY_COMMAND = 'Command';

    const DOWNLOADED_BY_SYNC_MISSING_ORDERS = 'SyncMissingOrders';

    const DOWNLOADED_BY_BACKUP = 'GetOrdersJobBackup';

    // override the default timestamps columns because the shopify has the same names
    public const UPDATED_AT = 'sku_updated_at';

    public const CREATED_AT = 'sku_created_at';

    protected $table = 'shopify_orders';

    protected $fillable = [
        'order_number',
        'integration_instance_id',
        'json_object',
        'transactions',
        'sku_created_at',
        'sku_updated_at',
        'sku_order_updated_at',
        'created_at',
        'updated_at',
        'createdAtUtc',
        'updatedAtUtc',
        'error_log',
        'processedAtUtc',
    ];

    protected $casts = [
        'json_object' => 'array',
        'name' => 'string',
        'shipping_lines' => 'array',
        'billing_address' => 'array',
        'shipping_address' => 'array',
        'email' => 'string',
        'discount_applications' => 'array',
        'total_price' => 'float',
        'customer' => 'array',
        'refunds' => 'array',
        'line_items' => 'array',
        'transactions' => 'array',
        'fulfillmentsMap' => 'array',
        'fulfillments' => 'array',
        'refundsMap' => 'array',
        'errors' => 'array',
        'sku_created_at' => 'datetime',
        'sku_updated_at' => 'datetime',
        'sku_order_updated_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'error_log' => 'string',
        'processedAtUtc' => 'datetime',
    ];

    public $dataTableKey = 'shopify.order';

    private array $noRestockFulfillmentCache = [];

    public function orderLineItems()
    {
        return $this->hasMany(ShopifyOrderLineItem::class, 'shopify_order_id');
    }

    public function integrationInstance()
    {
        return $this->belongsTo(IntegrationInstance::class);
    }

    public function salesOrder()
    {
        return $this->belongsTo(SalesOrder::class, 'sku_sales_order_id');
    }

    public function availableColumns()
    {
        return config('data_table.shopify.order.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return ['order_number', 'name', 'email'];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public static function getResource()
    {
        return ShopifyOrderResource::class;
    }

    public static function getOrderDateAttributeName(): string
    {
        return 'processed_at';
    }

    public function toArray()
    {
        $order = parent::toArray();
        foreach ($order['json_object']['line_items'] as $key => $lineItem) {
            $order['json_object']['line_items'][$key]['shop_url'] = $this->integrationInstance->connection_settings['shop_url'] ?? null;
            $listing = ShopifyProduct::with([])->where('variant_id', $lineItem['variant_id'])->first();
            if (! $listing) {
                continue;
            }
            $order['json_object']['line_items'][$key]['handle'] = $listing['handle'];
        }

        return $order;
    }

    public function getShopifyOrderId()
    {
        return $this->shopify_order_id;
    }

    public function getSkuOrderLines(?array $lineItems = null): array
    {
        return $this->getSkuSalesOrderLines($lineItems);
    }

    public function findMatchingSku($orderLines, $sku)
    {
        return $orderLines->firstWhere('sku', $sku);
    }

    public function getSkuShippingMethodId()
    {
        return $this->getShippingMethodId();
    }

    public function isFullyShipped(): bool
    {
        return $this['fulfillment_status'] === 'fulfilled';
    }

    public function isPartiallyShipped(): bool
    {
        return $this['fulfillment_status'] === 'partial';
    }

    public function partiallyFulfill(SalesOrder $salesOrder)
    {
        // TODO: Implement partiallyFulfill() method.
    }

    public function createLineItems()
    {
        if (! is_null($this->line_items)) {
            foreach ($this->line_items as $lineItem) {
                DB::transaction(function () use ($lineItem) {
                    ShopifyOrderLineItem::updateOrCreate([
                        'shopify_order_id' => $this->id,
                        'line_id' => $lineItem['id'],
                    ], [
                        'shopify_order_id' => $this->id,
                        'line_item' => $lineItem,
                    ]);
                }, 2);
            }
        }
    }

    //Replace functions

    public function isCancelled(?SalesOrder $salesOrder = null): bool
    {
        $salesOrder = $salesOrder ?: $this->salesOrder;

        return ! empty($this->json_object['cancelled_at']) ||
           ($salesOrder && $salesOrder->salesOrderLines()->where('is_product', true)->sum('quantity') == 0
             && $salesOrder->salesOrderLines()->where('is_product', true)->sum('canceled_quantity') > 0);
    }

    /**
     * @throws Throwable
     */
    protected function handleOnHoldStatus(SalesOrder $salesOrder): SalesOrder
    {
        $salesOrder->reserve();

        return $salesOrder;
    }

    public function isOnHold(): bool
    {
        return (! $this['fulfillment_status'] || $this['fulfillment_status'] == 'null') && collect($this['line_items'])->sum('fulfillable_quantity') == 0;
    }

    public function fulfilled(): bool
    {
        return $this['fulfillment_status'] === 'fulfilled';
    }

    public function paid(): bool
    {
        return $this['financial_status'] === 'paid';
    }

    /**
     * @throws Exception
     */
    public function watchUnsupportedFinancialStatuses()
    {
        $unsupportedStatuses = [
            'voided',
        ];
        if (in_array($this['financial_status'], $unsupportedStatuses)) {
            throw new UnsupportedFinancialStatusException('Unsupported financial status: '.$this['financial_status'].' for sales order '.$this['name'].'. This needs to be analyzed.');
        }
    }

    public function hasRefunds(): bool
    {
        return $this['financial_status'] === 'refunded' || $this['financial_status'] === 'partially_refunded' || ! empty($this['refunds']);
    }

    /**
     * @throws Throwable
     */
    protected function createFulfillments(SalesOrder $salesOrder): SalesOrder
    {
        // delete sales order fulfillments that canceled in Shopify
        if ($this->salesOrder->shopifyOrder->isPreAuditTrail()) {
            return $salesOrder;
        }

        // All products must be mapped and warehoused
        if (! $salesOrder->allProductsWarehoused()) {
            return $salesOrder;
        }

        // delete sales order fulfillments that canceled in Shopify
        $this->purgeCancelledFulfillments();

        // Fulfill sales order
        foreach ($this->getFulfillments($salesOrder) as $fulfillment) {
            if (! $this->isFulfillmentProcessed($fulfillment['shopify_fulfillment_id'])) {
                $salesOrderFulfillment = FulfillSalesOrderService::make($salesOrder)->fulfillWithInputs($fulfillment);
                // failed to create the fulfillment
                if (is_array($salesOrderFulfillment)) {
                    continue;
                }

                /*
                 * We do this to avoid resending fulfillments to Shopify when running Submit Tracking Info command
                 */
                $salesOrderFulfillment->submitted_to_sales_channel_at = now();
                $salesOrderFulfillment->save();

                $this->markFulfillmentAsProcessed($fulfillment['shopify_fulfillment_id'], $salesOrderFulfillment->id);
            }
        }

        return $salesOrder;
    }

    protected function purgeCancelledFulfillments(): void
    {
        $canceledFulfillmentIds = collect($this->fulfillments ?: [])
            ->where('status', 'cancelled')
            ->pluck('id')
            ->toArray();

        app(ShopifyOrderMappingRepository::class)
            ->getMappingsForLinks(ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS, $canceledFulfillmentIds)
            ->each(function (ShopifyOrderMapping $mapping) {
                /** @var SalesOrderFulfillment|null $fulfillment */
                $fulfillment = SalesOrderFulfillment::with([])->find($mapping->sku_link_id);
                if ($fulfillment) {
                    $fulfillment->setRelation('salesOrder', $this->salesOrder)
                        ->delete(true);
                    app(ShopifyOrderMappingRepository::class)->deleteWithSiblings($mapping);
                }
            });
    }

    protected function isFulfillmentProcessed($fulfillmentId): bool
    {
        $totalUnitsProcessed = app(ShopifyOrderMappingRepository::class)->getTotalFulfillmentUnitsProcessed($fulfillmentId);

        if ($totalUnitsProcessed == 0) {
            return false;
        }

        return collect($this->fulfillments ?: [])
            ->where('id', $fulfillmentId)
            ->pluck('line_items')
            ->collapse()
            ->sum('quantity') <= $totalUnitsProcessed;
    }

    protected function markFulfillmentAsProcessed($fulfillmentId, $skuLinkId): void
    {
        $fulfillment = collect($this->fulfillments ?: [])
            ->where('id', $fulfillmentId)
            ->first();

        if (! $fulfillment) {
            return;
        }

        collect($fulfillment['line_items'])->each(function ($fulfillmentLineItem) use ($fulfillment, $skuLinkId) {
            ShopifyOrderMapping::with([])
                ->updateOrCreate(
                    [
                        'shopify_order_id' => $this->id,
                        'link_id' => $fulfillment['id'],
                        'link_type' => ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS,
                        'line_id' => $fulfillmentLineItem['id'],
                        'line_type' => ShopifyOrderMapping::LINE_TYPE_LINE_ITEM,
                    ],
                    [
                        'processed_at' => $fulfillment['created_at'],
                        'quantity' => $fulfillmentLineItem['quantity'],
                        'sku_link_id' => $skuLinkId,
                        'sku_link_type' => SalesOrderFulfillment::class,
                    ]);
        });

    }

    protected function getFulfillments(SalesOrder $salesOrder): array
    {
        $fulfillments = [];

        collect($this['fulfillments'])->whereIn('status', ['open', 'success'])
            ->where('service', '!=', 'gift_card')
            ->map(function ($fulfillment) use ($salesOrder, &$fulfillments) {
                // Line ids
                $lineItems = collect($fulfillment['line_items']);
                $orderLines = $salesOrder->salesOrderLines()
                    ->whereIn('sales_channel_line_id', $lineItems->pluck('id')->toArray())
                    ->where('is_product', 1)
                    // TODO:
                    // May want to add a condition here where quantity - externally_fulfilled_quantity - fulfilled_quantity > 0
                    // This could also be a case for delaying sales order fulfillment creation and letting the sales order manager
                    // handle it.  Here we would just build the DTO.
                    ->get()
                    ->groupBy(['warehouse_id']);

                foreach ($orderLines as $warehouseId => $salesOrderLines) {
                    $fulfillments[] = [
                        'shopify_fulfillment_id' => $fulfillment['id'],
                        'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
                        'fulfilled_at' => $fulfillment['updated_at'],
                        'shipping_method_id' => $salesOrder->shipping_method_id,
                        'fulfilled_shipping_method' => $fulfillment['tracking_company'],
                        'requested_shipping_method' => $salesOrder->requested_shipping_method,
                        'tracking_number' => count($fulfillment['tracking_numbers']) ? implode(', ', $fulfillment['tracking_numbers']) : null,
                        'warehouse_id' => $warehouseId,
                        'fulfillment_lines' => $salesOrderLines->map(function ($orderLine) use ($lineItems) {
                            $lineItem = $lineItems->firstWhere('id', $orderLine->sales_channel_line_id);

                            return [
                                'sales_order_line_id' => $orderLine->id,
                                'quantity' => $lineItem['quantity'] - ($lineItem['externally_fulfilled_quantity'] ?? 0),
                            ];
                        })->toArray(),
                    ];
                }
            })->toArray();

        return $fulfillments;
    }

    /**
     * @throws Throwable
     */
    protected function getSkuFulfillments(SalesOrder $salesOrder, array $salesOrderData): array
    {
        /*
         * First update the sales order data to include the warehouse id
         */
        $shopifySalesOrderLines = [];
        foreach ($salesOrderData['sales_order_lines'] as $shopifySalesOrderLine) {
            if (! $shopifySalesOrderLine['is_product']) {
                continue;
            }
            /** @var SalesOrderLine $salesOrderLine */
            $salesOrderLine = $salesOrder->salesOrderLines()
                ->where('sales_channel_line_id', $shopifySalesOrderLine['sales_channel_line_id'])
                ->first();

            // Exception thrown because all sales order lines should be mapped to their Shopify equivalent line by using sales_channel_line_id
            throw_if(! $salesOrderLine?->product, new Exception('Sales order line not found for '.$salesOrder->sales_order_number.' for shopify sales order line '.$shopifySalesOrderLine['sales_channel_line_id']));

            // Exception thrown because we should never enter this method if the sales order has unmapped line
            throw_if(! $salesOrderLine?->product, new Exception('Product not found for sales order line due to it being unmapped. '.$salesOrder->sales_order_number.', '.$salesOrderLine?->id));

            $shopifySalesOrderLine['warehouse_id'] = $salesOrderLine->warehouse_id;
            $shopifySalesOrderLines[] = $shopifySalesOrderLine;
        }
        $salesOrderData['sales_order_lines'] = $shopifySalesOrderLines;

        $fulfillments = [];

        collect($this['fulfillments'])->whereIn('status', ['open', 'success'])
            ->where('service', '!=', 'gift_card')
            //This filter makes sure already processed fulfillments are not included.
            ->filter(fn ($fulfillment) => ! $this->isFulfillmentProcessed($fulfillment['id']))
            // Don't include fulfillments that are prior to inventory start date
            ->filter(function ($fulfillment) {
                return Carbon::parse($fulfillment['created_at'])->utc()->greaterThanOrEqualTo(SettingRepository::getInventoryStartDate());
            })
            ->map(function ($fulfillment) use ($salesOrderData, &$fulfillments) {
                // Line ids
                $shopifyFulfillmentLineItems = collect($fulfillment['line_items']);

                $skuOrderLines = collect($salesOrderData['sales_order_lines'])
                    ->whereIn('sales_channel_line_id', $shopifyFulfillmentLineItems->pluck('id')->toArray())
                    ->where('is_product', true)
                    ->groupBy(['warehouse_id']);

                foreach ($skuOrderLines as $warehouseId => $salesOrderLines) {
                    $fulfillments[] = [
                        'sales_channel_fulfillment_id' => $fulfillment['id'],
                        'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
                        'fulfilled_at' => $fulfillment['updated_at'],
                        'shipping_method_id' => $salesOrderData['shipping_method_id'],
                        'fulfilled_shipping_method' => $fulfillment['tracking_company'],
                        'requested_shipping_method' => $salesOrderData['requested_shipping_method'],
                        'tracking_number' => count($fulfillment['tracking_numbers']) ? implode(', ', $fulfillment['tracking_numbers']) : null,
                        'warehouse_id' => $warehouseId,
                        'fulfillment_lines' => $salesOrderLines->map(function ($skuSalesOrderLine) use ($shopifyFulfillmentLineItems) {
                            $skuFulfillmentLine = $shopifyFulfillmentLineItems->firstWhere('id', $skuSalesOrderLine['sales_channel_line_id']);

                            return [
                                // sku sales order line doesn't have an id yet because it is not yet created, so we use the sales_channel_line_id as the link
                                'sales_channel_line_id' => $skuSalesOrderLine['sales_channel_line_id'],
                                'quantity' => $skuFulfillmentLine['quantity'],
                            ];
                        })->toArray(),
                    ];
                }
            })->toArray();

        return $fulfillments;
    }

    private function syncFulfillmentsBeforeCancellation()
    {
        // First, attempt to create fulfillments.
        // Note that any existing fulfillments will
        // be skipped as they're tracked in fulfillments map.
        // Also, any pre-audit trail fulfillments will be
        // skipped.
        $this->createFulfillments($this->salesOrder);

        // Next, since we're cancelling the order, we remove
        // any fulfillments not marked as fulfilled.
        $this->salesOrder->salesOrderFulfillments
            ->whereNotIn('status', [SalesOrderFulfillment::STATUS_FULFILLED])
            ->each
            ->delete(true);
    }

    /**
     * Marks the sales order as cancelled.
     */
    public function markSalesOrderAsCancelled(SalesOrder $salesOrder): SalesOrder
    {

        // We synchronise the sales channel and sku fulfillments
        // to ensure that already fulfilled lines are not removed.
        $this->syncFulfillmentsBeforeCancellation();

        $salesOrder->cancel($this['cancelled_at'] ?? now());
        $salesOrder->close();

        return $salesOrder;
    }

    /**
     * @throws Throwable
     */
    public function handleOrderStatuses(SalesOrder $salesOrder): SalesOrder
    {
        if ($this->isCancelled($salesOrder) || (in_array($this['financial_status'], ['refunded', 'voided']) && $salesOrder->fulfillment_status !== SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC)) {
            // Cancel the sales order
            return $this->markSalesOrderAsCancelled($salesOrder);
        }

        if ($this->isOnHold()) {
            return $this->handleOnHoldStatus($salesOrder);
        }

        $this->watchUnsupportedFinancialStatuses();

        return $salesOrder;
    }

    public function createPaymentsFromCapturedTransactions(SalesOrder $salesOrder): SalesOrder
    {
        $shopifyPaymentType = PaymentType::with([])->firstOrCreate(['name' => PaymentType::PAYMENT_TYPE_SHOPIFY]);

        // Try to re-fetch transactions if shop money is not set
        if (collect($this['transactions'])->reject(function ($transaction) {
            return isset($transaction['amountSet']['shopMoney']);
        })->isNotEmpty())
        {
            $this->fetchOrderTransactions();
            $this->refresh();
        }

        $payments = collect($this['transactions'])
            ->whereIn('kind', ['CAPTURE', 'SALE', 'REFUND'])
            ->where('status', 'SUCCESS')
            ->map(function ($transaction) use ($shopifyPaymentType, $salesOrder) {
                $amountSet = @$transaction['amountSet']['shopMoney'];
                if (!$amountSet) {
                    throw new Exception('Amount set not found in transaction for order '.$this['name']);
                }
                $salesOrder->payments()->delete(); // Clear any existing payments

                if ($transaction['kind'] == 'REFUND') {
                    $salesCredit = $salesOrder->salesCredits()->where('total_credit', abs($amountSet['amount']))->first();
                    if (! $salesCredit instanceof SalesCredit) {
                        $salesCredit = new SalesCredit([
                            'sales_order_id' => $salesOrder->id,
                            'total_credit' => abs($amountSet['amount']),
                            'credit_date' => Carbon::parse($transaction['processedAt']),
                        ]);
                        $salesOrder->addSalesCredit($salesCredit);
                    } else {
                        $salesCredit->payments()->delete();
                    }

                    $salesCredit->payments()->create(
                        [
                            'payment_date' => $transaction['processedAt'],
                            'payment_type_id' => $shopifyPaymentType->id,
                            'amount' => abs($amountSet['amount']),
                            'cost' => $this->getTransactionCost($transaction),
                            'currency_code' => $amountSet['currencyCode'] ?? ($this['currency'] ?? Currency::default()->code),
                        ]
                    );

                    return false;
                }

                return [
                    'payment_date' => $transaction['processedAt'],
                    'payment_type_id' => $shopifyPaymentType->id,
                    'amount' => $amountSet['amount'],
                    'cost' => $this->getTransactionCost($transaction),
                    'currency_code' => $amountSet['currencyCode'] ?? ($this['currency'] ?? Currency::default()->code),
                    'external_reference' => $transaction['id'],
                ];
            })->reject(function ($value) {
                return $value === false;
            })->toArray();

        $salesOrder->payments()->createMany($payments);
        $salesOrder->load('payments');
        $salesOrder->setPaymentStatus();

        return $salesOrder;
    }

    public function fetchOrderTransactions(): void
    {
        $shopify = new Shopify($this->integrationInstance);
        $shopifyOrderTransactions = $shopify->getOrderTransactions($this->shopify_order_id);
        $order = collect($shopifyOrderTransactions['data']['orders']['edges'])->pluck('node')->values()[0];
        $unified_transactions = [];
        foreach($order['transactions'] as $transaction) {
            $transaction['order_id'] = $this->shopify_order_id;
            $unified_transactions[] = $transaction;
        }
        app(ShopifyTransactionRepository::class)->bulkSave($this->integrationInstance, $unified_transactions);
    }

    /**
     * Extract the transaction fee
     *
     *
     * @return mixed|null
     */
    public function getTransactionCost(array $saleTransaction)
    {

        if (empty($saleTransaction['gateway'])) {
            return null;
        }

        // paypal
        if ($saleTransaction['gateway'] == 'paypal') {
            if (is_string($receipt = $saleTransaction['receipt'])) {
                $receipt = str_replace(['=>', 'nil'], [':', 'null'], $receipt);
                $receipt = json_decode($receipt, 1);
            }

            return $receipt['fee_amount'] ?? ($receipt['fee_refund_amount'] ?? 0);
        }

        // shopify_payments
        if ($saleTransaction['gateway'] == 'shopify_payments') {
            return collect($saleTransaction['fees'])->sum('amount.amount');
        }

        return null;
    }

    public function isRefundProcessed($refundId): bool
    {
        $refundMappingLines = app(ShopifyOrderMappingRepository::class)->getMappingsForLinks(ShopifyOrderMapping::LINK_TYPE_REFUNDS, $refundId);

        $refund = collect($this->refunds)->firstWhere('id', $refundId);
        // check refund line items
        foreach ($refund['refund_line_items'] as $lineItem) {
            if (! $refundMappingLines->firstWhere('line_id', $lineItem['id'])) {
                return false;
            }
        }

        foreach ($refund['order_adjustments'] as $adjustment) {
            if (! $refundMappingLines->firstWhere('line_id', $adjustment['id'])) {
                return false;
            }
        }

        return true;
    }

    /**
     * @throws Exception|Throwable
     */
    public function handleRefunds(): ?SalesOrder
    {
        $shopifyRefunds = $this['refunds'];

        // There are no refunds
        if (empty($shopifyRefunds)) {
            return $this->salesOrder;
        }

        // TODO: @Bright, the following is for cancellations and cancellations are handled in SalesOrderManager.  It seems unnecessary?

        $refundLineItemIds = collect($shopifyRefunds)
            ->filter(fn ($refund) => ! $this->isRefundProcessed($refund['id']))
            ->pluck('refund_line_items')
            ->collapse()
            ->pluck('line_item_id')
            ->unique()
            ->toArray();

        foreach ($refundLineItemIds as $lineItemId) {
            $salesOrderLine = $this->getSalesOrderLineByChannelLineId($lineItemId);
            if (! $salesOrderLine) {
                continue;
            }

            // We cancel quantities removed from the order (refund_type = cancel)
            // or those with refund_type = no_restock but without fulfillments.
            $totalLineItemQty = $this->getLineItemQty($lineItemId);
            $totalUnfulfilledForLine = $this->getTotalUnfulfilledForLine($lineItemId);
            $canceled = $this->getTotalCanceledForLine($lineItemId);
            $totalNoRestockRefunds = $this->getTotalNoRestockRefundsForLineItem($lineItemId);
            $totalFulfilled = $this->getTotalFulfilledForLine($lineItemId);

            $totalCanceledInSalesChannel = min(
                $totalNoRestockRefunds,
                max(0, $totalLineItemQty - $totalFulfilled - $totalUnfulfilledForLine)
            ) + $canceled;

            $extraCancellationQty = max(0, $totalCanceledInSalesChannel - $salesOrderLine->canceled_quantity);
            if ($extraCancellationQty > 0) {
                $salesOrderLine->updateCanceledQuantity(
                    $totalCanceledInSalesChannel,
                    $totalLineItemQty,
                    $salesOrderLine->canceled_quantity ?? 0
                );
            }
        }

        // we handle refunds with sales credits.
        $this->handleReturnsInRefunds($this['refunds']);

        return $this->salesOrder;
    }

    private function getTotalCanceledForLine($lineItemId): int
    {
        return collect($this->refunds ?? [])
            ->pluck('refund_line_items')
            ->collapse()
            ->where('restock_type', 'cancel')
            ->where('line_item_id', $lineItemId)
            ->sum('quantity');
    }

    private function getTotalUnfulfilledForLine($lineItemId): int
    {
        return collect($this->line_items)->where('id', $lineItemId)->sum('fulfillable_quantity');
    }

    private function getTotalFulfilledForLine($lineItemId): int
    {
        return collect($this->fulfillments)
            ->where('status', 'success')
            ->pluck('line_items')
            ->collapse()
            ->where('id', $lineItemId)
            ->sum('quantity');
    }

    private function getTotalNoRestockRefundsForLineItem($lineItemId): int
    {
        return collect($this->refunds ?? [])
            ->pluck('refund_line_items')
            ->collapse()
            ->where('restock_type', 'no_restock')
            ->where('line_item_id', $lineItemId)
            ->sum('quantity');
    }

    private function getLineItemQty($lineItemId): int
    {
        return collect($this->line_items)
            ->where('id', $lineItemId)
            ->sum('quantity');
    }

    private function getSalesOrderLineByChannelLineId($salesOrderChannelLineId): SalesOrderLine|Model|null
    {
        return $this->salesOrder
            ->salesOrderLines()
            ->where('sales_channel_line_id', $salesOrderChannelLineId)
            ->first();
    }

    private function handleReturnsInRefunds(array $refunds): SalesOrder
    {
        if (empty($refunds)) {
            return $this->salesOrder;
        }
        // Get fulfilled line item IDs from successful fulfillments.
        $fulfilled_line_item_ids = collect($this['fulfillments'])
            ->where('status', 'success')
            ->pluck('line_items')
            ->flatten(1)
            ->pluck('id')
            ->unique();

        // Get the sales order associated with the current context.
        $salesOrder = $this->salesOrder;

        // Iterate through each refund and process accordingly.
        foreach ($refunds as $refund) {

            // Skip if the refund has already been processed.
            if ($this->isRefundProcessed($refund['id'])) {
                continue;
            }

            // Filter refund line items for credits related to return or no restock types.
            /** @see CleanupExtraSalesCreditsCommand::syncRefunds please see to this command when changing this condition */
            $credits = collect($refund['refund_line_items'])
                ->whereIn('restock_type', ['return', 'no_restock'])
                ->whereIn('line_item_id', $fulfilled_line_item_ids);

            // cancellation or adjustment refund, handle it as adjustment refund
            // in this part its handle adjustment refunds or non-returnable line items if there are no "return" or "no restock" line items.
            if ($credits->isEmpty()) {
                $this->handleAdjustmentRefund($refund);

                continue;
            }

            // Filter refund line items to exclude credited line items.
            $nonCreditedRefundLineItems = collect($refund['refund_line_items'])
                ->whereNotIn('id', $credits->pluck('id'));

            if (! $nonCreditedRefundLineItems->isEmpty()) {
                $adjustedRefund = $refund;
                $adjustedRefund['refund_line_items'] = $nonCreditedRefundLineItems->toArray();
                // when credits not empty and non-credited refund line items not empty,
                // handle adjustment refund  and just process refund line items to prevent dublication
                $this->handleAdjustmentRefund($adjustedRefund, ProcessedAdjustmentRefund: false);
            }

            // Create a new SalesCredit instance for the refund.
            $salesCredit = new SalesCredit([
                'credit_status' => SalesCredit::CREDIT_STATUS_OPEN,
                'return_status' => SalesCredit::RETURN_STATUS_NOT_RETURNED,
                'credit_date' => Carbon::parse($refund['processed_at']),
                'store_id' => $this->integrationInstance->salesChannel->store_id,
                'sales_credit_note' => $refund['note'] ?? '',
                'to_warehouse_id' => Setting::getValueByKey(Setting::KEY_SC_DEFAULT_WAREHOUSE),
            ]);

            // Associate the SalesCredit with the sales order.
            $salesOrder->addSalesCredit($salesCredit);

            $salesOrder->refresh();

            $salesOrder->load('salesOrderLines');

            $returnableLines = [];
            $nonReturnableLines = [];
            foreach ($credits as $credited_refund_line_item) {
                $orderLine = $this->getSalesOrderLineByChannelLineId($credited_refund_line_item['line_item_id']);

                /*
                 * Create sales credit
                 * no_restock - mark received as not to be returned
                 * return - mark received as added back to stock
                 */
                $creditLine = app(ShopifyOrderMappingRepository::class)->getRefundMapByLineItemId($this, $refund['id'], $credited_refund_line_item['id']);
                if ($creditLine && SalesCreditLine::find($creditLine->sku_link_id)) {
                    continue;
                }

                $line = [
                    'quantity' => $credited_refund_line_item['quantity'],
                    'description' => $credited_refund_line_item['line_item']['name'] ?? '',
                    'tax' => $credited_refund_line_item['total_tax'] ?? 0,
                    'amount' => $credited_refund_line_item['subtotal'] / $credited_refund_line_item['quantity'],
                    'sales_order_line_id' => $orderLine->id ?? null,
                    'product_id' => $orderLine->product_id ?? null,
                ];

                if ($credited_refund_line_item['restock_type'] == SalesCreditReturnLine::ACTION_NO_RESTOCK || $this->isPreAuditTrail()) {
                    $nonReturnableLines[] = $this->createAndMapSalesCreditLines($salesCredit, [$line], $refund['id'], $credited_refund_line_item['id'], ShopifyOrderMapping::LINE_TYPE_LINE_ITEM);
                } else {
                    $returnableLines[] = $this->createAndMapSalesCreditLines($salesCredit, [$line], $refund['id'], $credited_refund_line_item['id'], ShopifyOrderMapping::LINE_TYPE_LINE_ITEM);
                }
            }

            // process adjustment refunds when credits not empty and Adjustment Refunds not empty
            $this->processAdjustmentRefunds($refund, $salesCredit);

            // Only process returns if there are returns to process.
            if (! empty($returnableLines) || ! empty($nonReturnableLines)) {
                $salesCreditReturn = new SalesCreditReturn([
                    'warehouse_id' => $salesCredit->to_warehouse_id,
                    'shipped_at' => $salesCredit->credit_date,
                    'received_at' => $salesCredit->credit_date,
                ]);
                $salesCredit->salesCreditReturns()->save($salesCreditReturn);

                // add its lines and add to inventory
                $salesCreditReturn->setReturnLines(
                    $salesCredit->salesCreditLines()
                        ->whereNotNull('product_id')
                        ->whereIn('id', $returnableLines)
                        ->get()
                        ->map(function (SalesCreditLine $salesCreditLine) {
                            return [
                                'sales_credit_line_id' => $salesCreditLine->id,
                                'quantity' => $salesCreditLine->quantity,
                                'product_id' => $salesCreditLine->product_id ?? null,
                                'action' => SalesCreditReturnLine::ACTION_ADD_TO_STOCK,
                                'reason_id' => (ReturnReason::with([])->firstOrCreate(['name' => 'Other']))->id,
                            ];
                        })
                        ->toBase()
                        ->merge(
                            $salesCredit->salesCreditLines()
                                ->whereIn('id', $nonReturnableLines)
                                ->whereNotNull('product_id')
                                ->get()
                                ->map(function (SalesCreditLine $salesCreditLine) {
                                    return [
                                        'sales_credit_line_id' => $salesCreditLine->id,
                                        'quantity' => $salesCreditLine->quantity,
                                        'product_id' => $salesCreditLine->product_id,
                                        'action' => SalesCreditReturnLine::ACTION_NO_RESTOCK,
                                        'reason_id' => (ReturnReason::with([])->firstOrCreate(['name' => 'Other']))->id,
                                    ];
                                })
                        )->toArray()
                );

                $salesCredit->returned(! empty($refund['processed_at']) ? Carbon::parse($refund['processed_at']) : null);

            }

            // Update the processed refund information.
            $this->markRefundAsProcessed($refund['id']);
        }

        return $salesOrder;
    }

    public function handleAdjustmentRefund(array $refund, $ProcessedAdjustmentRefund = true)
    {
        // Check if the refund has already been processed, and return early if it has.
        if ($this->isRefundProcessed($refund['id'])) {
            return;
        }

        // Create a new SalesCredit instance for the refund.
        $salesCredit = new SalesCredit([
            'credit_status' => SalesCredit::CREDIT_STATUS_CLOSED,
            'return_status' => SalesCredit::RETURN_STATUS_NOT_RETURNED,
            'credit_date' => Carbon::parse($refund['processed_at']),
            'store_id' => $this->integrationInstance->salesChannel->store_id,
            'sales_credit_note' => $refund['note'] ?? '',
        ]);
        // Associate the SalesCredit instance with the current sales order.
        $this->salesOrder->addSalesCredit($salesCredit);

        // Process adjustment refunds and create associated sales credit lines.
        if ($ProcessedAdjustmentRefund && ! empty($refund['order_adjustments'])) {
            $this->processAdjustmentRefunds($refund, $salesCredit);
        }

        // Process refund line items and create associated sales credit lines.
        if (! empty($refund['refund_line_items'])) {
            $this->processRefundLineItems($refund, $salesCredit);
        }
        // Mark the SalesCredit as closed and save changes.
        $salesCredit->credit_status = SalesCredit::CREDIT_STATUS_CLOSED;
        $salesCredit->save();

        // Mark the refund as processed ;
        $this->markRefundAsProcessed($refund['id']);
    }

    private function processRefundLineItems(array $refund, SalesCredit $salesCredit)
    {
        $refundLineItems = collect($refund['refund_line_items']);
        // Iterate through each refund line item and create corresponding sales credit lines.
        foreach ($refundLineItems as $refundLineItem) {

            $orderLine = $this->getSalesOrderLineByChannelLineId($refundLineItem['line_item_id']);
            $line = [
                'quantity' => $refundLineItem['quantity'],
                'description' => $refundLineItem['line_item']['name'] ?? '',
                'tax' => $refundLineItem['total_tax'] ?? 0,
                'amount' => $refundLineItem['subtotal'] ?? 0 / $refundLineItem['quantity'] ?? 1,
                'sales_order_line_id' => $orderLine->id ?? null,
                'product_id' => $orderLine->product_id ?? null,
            ];
            // Create and map sales credit lines with relevant attributes.
            $this->createAndMapSalesCreditLines(
                $salesCredit, [$line], $refund['id'], $refundLineItem['id'], ShopifyOrderMapping::LINE_TYPE_LINE_ITEM
            );
        }
    }

    private function processAdjustmentRefunds(array $refund, SalesCredit $salesCredit)
    {
        // Iterate through each adjustment refund and create corresponding sales credit lines.
        collect(@$refund['order_adjustments'])->each(function ($adjustment) use ($refund, $salesCredit) {
            $creditLine = [
                'quantity' => 1,
                'description' => $adjustment['reason'],
                'tax' => -($adjustment['tax_amount'] ?? 0),
                'amount' => -$adjustment['amount'],
            ];
            // Create and map sales credit lines for adjustments.
            $this->createAndMapSalesCreditLines(
                $salesCredit, [$creditLine], $refund['id'], $adjustment['id'], ShopifyOrderMapping::LINE_TYPE_REFUND_ADJUSTMENT
            );
        });
    }

    private function createAndMapSalesCreditLines(SalesCredit $salesCredit, array $creditLines, $refundId, $lineId, $lineType)
    {
        // Create sales credit lines and associate them
        // with appropriate mappings that have link type refunds.

        $savedCreditLines = [];
        // Iterate through each credit line and create corresponding mappings.
        foreach ($creditLines as $creditLine) {
            // Create sales credit lines for the sales credit instance.
            $salesCreditLines = $salesCredit->setSalesCreditLines([$creditLine], false);
            $savedCreditLine = $salesCreditLines[0];

            // Define attributes for the ShopifyOrderMapping.
            $matchingAttributes = [
                'shopify_order_id' => $this->id,
                'line_id' => $lineId,
                'line_type' => $lineType,
                'link_id' => $refundId,
                'link_type' => ShopifyOrderMapping::LINK_TYPE_REFUNDS,
            ];

            $shopifyOrderMappingData = [
                'quantity' => 1,
                'processed_at' => Carbon::now(),
                'sku_link_id' => $savedCreditLine,
                'sku_link_type' => SalesCreditLine::class,
            ];
            // Update or create the ShopifyOrderMapping.
            app(ShopifyOrderMappingRepository::class)->upsert($matchingAttributes, $shopifyOrderMappingData);

            // Store the saved credit line for further use.
            $savedCreditLines[] = $savedCreditLine;
        }

        return $savedCreditLines;
    }

    public function orderMappings(): HasMany
    {
        return $this->hasMany(ShopifyOrderMapping::class, 'shopify_order_id');
    }

    public function clearRefundMappings(): void
    {
        app(ShopifyOrderMappingRepository::class)->deleteRefundMappingsForShopifyOrder($this);
    }

    public function markRefundAsProcessed($refundId, $skuLinkId = null, $skuLinkType = null, $adjustmentSkuLinkId = null, $adjustmentSkuLinkType = null)
    {
        $refund = collect($this->refunds)->where('id', $refundId)->first();
        if (! $refund) {
            return;
        }

        // map refund line items
        foreach ($refund['refund_line_items'] as $refundLineItem) {
            $noRestockFulfillmentsApplied = 0;
            if (isset($this->noRestockFulfillmentCache[$refundId]) && isset($this->noRestockFulfillmentCache[$refundId][$refundLineItem['id']])) {
                $noRestockFulfillmentsApplied = $this->noRestockFulfillmentCache[$refundId][$refundLineItem['id']];
            }

            app(ShopifyOrderMappingRepository::class)
                ->upsert(
                    [
                        'shopify_order_id' => $this->id,
                        'line_id' => $refundLineItem['id'],
                        'line_type' => ShopifyOrderMapping::LINE_TYPE_LINE_ITEM,
                        'link_id' => $refund['id'],
                        'link_type' => ShopifyOrderMapping::LINK_TYPE_REFUNDS,
                    ],
                    [
                        'quantity' => $refundLineItem['quantity'],
                        'processed_at' => $refund['processed_at'],
                        'no_restock_fulfillments_applied' => $noRestockFulfillmentsApplied,
                    ]
                );
        }

        // map adjustments
        foreach ((@$refund['order_adjustments'] ?: []) as $orderAdjustment) {
            app(ShopifyOrderMappingRepository::class)
                ->upsert(
                    [
                        'shopify_order_id' => $this->id,
                        'line_id' => $orderAdjustment['id'],
                        'line_type' => ShopifyOrderMapping::LINE_TYPE_REFUND_ADJUSTMENT,
                        'link_id' => $refund['id'],
                        'link_type' => ShopifyOrderMapping::LINK_TYPE_REFUNDS,
                    ],
                    [
                        'quantity' => 1,
                        'processed_at' => $refund['processed_at'],
                    ]
                );
        }

    }

    public function needsUpdating(): bool
    {
        return ! $this->sku_order_updated_at || $this->sku_order_updated_at->lt($this->sku_updated_at);
    }

    /**
     * @throws BindingResolutionException
     */
    protected function getPriorityWarehouseId($product, $item, ?bool $defaultToFirst = true): ?int
    {

        if (! $product) {
            return null;
        }

        /** @var WarehouseRepository $warehouses */
        $warehouses = app()->make(WarehouseRepository::class);

        $product = $product instanceof ProductListing ? $product->product : $product;

        return $warehouses->getPriorityWarehouseIdForProduct($product, $item['quantity'], $defaultToFirst,
            $this->getSkuCustomerAddress());
    }

    protected function getWarehousesWithInventoryForProduct($product, $item, ?bool $defaultToFirst = true)
    {
        if (! $product) {
            return null;
        }

        /** @var WarehouseRepository $warehouses */
        $warehouses = app()->make(WarehouseRepository::class);

        $product = $product instanceof ProductListing ? $product->product : $product;

        return $warehouses->getWarehousesWithInventory($product, $item['quantity'], $this->getSkuShippingAddress());
    }

    private function getTaxAllocation(array $shopifyOrderLineItem): float
    {
        return max(collect(@$shopifyOrderLineItem['tax_lines'])->sum('price') - $this->getCanceledTaxAmount($shopifyOrderLineItem), 0);
    }

    public function getSkuSalesOrderLines(?array $shopifyOrderLineItems = null): array
    {
        $salesOrderLineCollection = new Collection();

        $salesChannel = $this->integrationInstance->salesChannel;

        $shopifyOrderLineItems = collect($shopifyOrderLineItems ?? $this['line_items']);

        // Filter out gift cards and line items without sku or that don't require shipping
        $shopifyOrderLineItemsForProducts = $shopifyOrderLineItems->filter(fn ($shopifyOrderLineItem) => ! (
            (array_key_exists('gift_card', $shopifyOrderLineItem) && $shopifyOrderLineItem['gift_card']) ||
            ! @$shopifyOrderLineItem['sku'] ||
            (array_key_exists('requires_shipping', $shopifyOrderLineItem) && ! $shopifyOrderLineItem['requires_shipping'] || ((array_key_exists('product_exists', $shopifyOrderLineItem) && ! $shopifyOrderLineItem['product_exists']) && (array_key_exists('product_id', $shopifyOrderLineItem) && ! $shopifyOrderLineItem['product_id'])))
        )
        );

        $shopifyOrderLineItemsForProducts->each(function ($shopifyOrderLineItem) use ($salesChannel, &$salesOrderLineCollection) {
            $taxLines = [];
            if ($taxRate = @$shopifyOrderLineItem['tax_lines'][0]) {
                $taxLines[] = [
                    'rate' => $taxRate['rate'] * 100,
                    'name' => $taxRate['title'],
                ];
            }

            $taxAllocation = $this->getTaxAllocation($shopifyOrderLineItem);

            /*
             * See SKU-6433.  Shopify passes tax rates even when the tax amount is 0.  We will handle this just by not using the tax rate if it not actually used.
             */
            $taxRate = $taxAllocation > 0 ? (new TaxRateService())->getTaxRateForArray($taxLines) : null;
            /** @var ProductListing $listing */
            $listing = ProductListing::with(['product'])->where('sales_channel_listing_id', $shopifyOrderLineItem['variant_id'])
                ->where('sales_channel_id', $salesChannel->id)
                ->first();

            // IN PROGRESS
            $salesOrderLineCollection->add(SalesOrderLineData::from([
                'sales_channel_line_id' => $shopifyOrderLineItem['id'],
                'product_id' => $listing?->product_id,
                'quantity' => $shopifyOrderLineItem['quantity'],
                'externally_fulfilled_quantity' => @$shopifyOrderLineItem['externally_fulfilled_quantity'] ?? 0,
                'description' => $shopifyOrderLineItem['name'],
                'is_product' => true,
                'is_taxable' => (bool) (@$shopifyOrderLineItem['tax_lines']),
                'amount' => $this->getShopifyOrderLineAmount($shopifyOrderLineItem),
                'tax_allocation' => $taxAllocation,
                'tax_rate_id' => $taxRate?->id,
                'tax_rate' => $taxRate?->rate ?? 0,
            ]));
        });

        $shopifyOrderLineItemsNotForProducts = $shopifyOrderLineItems->filter(fn ($shopifyOrderLineItem) => (array_key_exists('gift_card', $shopifyOrderLineItem) && $shopifyOrderLineItem['gift_card']) ||
            ! @$shopifyOrderLineItem['sku'] ||
            (array_key_exists('requires_shipping', $shopifyOrderLineItem) && ! $shopifyOrderLineItem['requires_shipping'] || ((array_key_exists('product_exists', $shopifyOrderLineItem) && ! $shopifyOrderLineItem['product_exists']) && (array_key_exists('product_id', $shopifyOrderLineItem) && ! $shopifyOrderLineItem['product_id'])))
        );

        $nonProductLineTotal = 0;
        $shopifyOrderLineItemsNotForProducts->each(function ($shopifyOrderLineItemNotForProduct) use (&$nonProductLineTotal) {
            $nonProductLineTotal += $this->getShopifyOrderLineAmount($shopifyOrderLineItemNotForProduct);
        });

        return $salesOrderLineCollection->toArray();
    }

    private function getShopifyOrderLineAmount(array $shopifyOrderLineItem): float
    {
        return @$shopifyOrderLineItem['price'] - (collect(@$shopifyOrderLineItem['discount_allocations'])->sum('amount') / $shopifyOrderLineItem['quantity']);
    }

    private function getCanceledTaxAmount(array $shopifyOrderLineItem)
    {
        return collect($this['refunds'])->pluck('refund_line_items')
            ->collapse()
            ->whereIn('restock_type', [
                'cancel',
                'no_restock',
            ])
            ->where('line_item_id', $shopifyOrderLineItem['id'])
            ->sum('total_tax');
    }

    public function getSkuFinancialLines(): array
    {
        $shopifyOrderLineItems = collect($this['line_items']);
        $financialLineCollection = new Collection();

        $nonProductLines = $shopifyOrderLineItems->filter(fn ($shopifyOrderLineItem) => (array_key_exists('gift_card', $shopifyOrderLineItem) && $shopifyOrderLineItem['gift_card']) ||
            ! @$shopifyOrderLineItem['sku'] ||
            (array_key_exists('requires_shipping', $shopifyOrderLineItem) && ! $shopifyOrderLineItem['requires_shipping'] || ((array_key_exists('product_exists', $shopifyOrderLineItem) && ! $shopifyOrderLineItem['product_exists']) && (array_key_exists('product_id', $shopifyOrderLineItem) && ! $shopifyOrderLineItem['product_id'])))
        );

        $nonProductLines->each(function ($shopifyOrderLineItem) use (&$financialLineCollection) {
            $taxLines = [];
            $totalRate = 0;
            if ($taxRate = @$shopifyOrderLineItem['tax_lines'][0]) {
                foreach ($shopifyOrderLineItem['tax_lines'] as $taxLine) {
                    $taxLines[] = [
                        'rate' => $taxLine['rate'] * 100,
                        'name' => $taxLine['title'],
                    ];
                    $totalRate += $taxLine['rate'];
                }
            }

            $taxRate = (new TaxRateService())->getTaxRateForArray($taxLines);

            $financialLineType = @$shopifyOrderLineItem['gift_card'] ? 'Gift Cards' : 'Other Revenue';
            $financialLineCollection->add(FinancialLineData::from([
                'financial_line_type_id' => app(FinancialLineRepository::class)
                    ->getOrCreateFinancialLineType($financialLineType, FinancialLineClassificationEnum::REVENUE)
                    ->id,
                'description' => $shopifyOrderLineItem['name'],
                'quantity' => $shopifyOrderLineItem['quantity'],
                'amount' => $this->getShopifyOrderLineAmount($shopifyOrderLineItem),
                'tax_allocation' => $this->getTaxAllocation($shopifyOrderLineItem),
                'tax_rate_id' => $taxRate?->id,
                'tax_rate' => $totalRate * 100,
                'sales_channel_line_id' => $shopifyOrderLineItem['id'],
            ]));
        });

        if (isset($this['shipping_lines'])) {
            $shippingLines = collect($this['shipping_lines']);
            $taxLines = [];
            if ($taxRate = @$shippingLines->first()['tax_lines'][0]) {
                $taxLines[] = [
                    'rate' => $taxRate['rate'] * 100,
                    'name' => $taxRate['title'],
                ];
            }

            // Shipping refunds total is negative
            $shippingRefundsTotal = -(collect($order['refunds'] ?? [])
                ->flatMap(function ($refund) {
                    return collect($refund['order_adjustments'] ?? []);
                })
                ->filter(function ($item) {
                    return $item['kind'] === 'shipping_refund';
                })
                ->sum(function ($item) {
                    return $item['amount'] + $item['tax_amount'];
                }));

            $shippingLines = $shippingLines->map(function ($shippingLine) {
                $shippingLine['price'] -= collect(@$shippingLine['discount_allocations'])?->sum('amount');

                return $shippingLine;
            });

            if (($amount = ($shippingLines->sum('price') - $shippingRefundsTotal)) > 0) {
                $shippingTaxTotal = $shippingLines->sum(function ($shippingLine) {
                    return collect($shippingLine['tax_lines'] ?? [])->sum('price');
                });

                $taxRate = (new TaxRateService())->getTaxRateForArray($taxLines);

                $financialLineCollection->add(FinancialLineData::from([
                    'financial_line_type_id' => app(FinancialLineRepository::class)
                        ->getOrCreateFinancialLineType('Shipping', FinancialLineClassificationEnum::REVENUE, Helpers::setting(Setting::KEY_NC_MAPPING_SHIPPING_SALES_ORDERS))
                        ->id,
                    'description' => 'Shipping',
                    'amount' => $amount,
                    'tax_allocation' => $shippingTaxTotal,
                    'tax_rate_id' => $taxRate?->id,
                    'tax_rate' => $taxRate?->rate ?? 0,
                ]));
            }
        }

        $integrationSettings = is_string($this->integrationInstance->integration_settings) ? [] : $this->integrationInstance->integration_settings;

        if ($paymentCostPercentage = @$integrationSettings['proforma_payment_cost_percentage']) {
            $financialLineCollection->add(FinancialLineData::from([
                'financial_line_type_id' => app(FinancialLineRepository::class)
                    ->getOrCreateFinancialLineType('Payment Fee', FinancialLineClassificationEnum::COST)
                    ->id,
                'description' => 'Shopify Payment Fee',
                'amount' => $this['total_price'] * $paymentCostPercentage,
            ]));
        }

        if ($marketplaceCostPercentage = @$integrationSettings['proforma_marketplace_cost_percentage']) {
            $financialLineCollection->add(FinancialLineData::from([
                'financial_line_type_id' => app(FinancialLineRepository::class)
                    ->getOrCreateFinancialLineType('Marketplace Fee', FinancialLineClassificationEnum::COST)
                    ->id,
                'description' => 'Shopify Marketplace Fee',
                'amount' => $this['total_price'] * ($marketplaceCostPercentage / 100),
            ]));
        }

        return $financialLineCollection->toArray();
    }

    public function getSkuBillingAddress(): ?array
    {
        return $this->billing_address ? (array_merge(
            $this['billing_address'],
            [
                'name' => isset($this['billing_address']['first_name']) ? "{$this['billing_address']['first_name']} {$this['billing_address']['last_name']}" : null,
                'email' => $this->email,
            ]
        )) : null;
    }

    protected function getShippingMethodId(): ?int
    {
        if (empty($this['shipping_lines'])) {
            return null;
        }

        $mapping = ShippingMethodMappingsSalesChannelToSku::firstOrCreate([
            'sales_channel_id' => $this->integrationInstance->salesChannel->id,
            'sales_channel_method' => $this['shipping_lines'][0]['code'],
        ]);

        return $mapping->shipping_method_id;
    }

    public function getSkuShippingAddress(): ?array
    {
        return $this->shipping_address ? (array_merge(
            $this['shipping_address'],
            [
                'name' => isset($this['shipping_address']['first_name']) ? "{$this['shipping_address']['first_name']} {$this['shipping_address']['last_name']}" : null,
                'email' => $this->email,
            ]
        )) : null;
    }

    public function getTags(): array
    {
        if (! empty($this['tags'])) {
            return explode(', ', $this['tags']);
        }

        return [];
    }

    public function getSkuCustomerAddress(): ?array
    {
        if (empty($this->customer)) {
            return $this->getSkuShippingAddress();
        }

        return [
            'name' => @$this['customer']['name'] ?? (
                isset(
                    $this['customer']['first_name']
                ) ? "{$this['customer']['first_name']}".(isset($this['customer']['last_name']) ? ' '.$this['customer']['last_name'] : '') :
                    null
            ),
            'email' => $this['customer']['email'] ?? null,
            'phone' => $this['customer']['phone'] ?? $this['customer']['default_address']['phone'] ?? null,
            'address1' => $this['customer']['default_address']['address1'] ?? null,
            'address2' => $this['customer']['default_address']['address2'] ?? null,
            'address3' => $this['customer']['default_address']['address3'] ?? null,
            'city' => $this['customer']['default_address']['city'] ?? null,
            'province' => $this['customer']['default_address']['province'] ?? null,
            'province_code' => $this['customer']['default_address']['province_code'] ?? null,
            'zip' => $this['customer']['default_address']['zip'] ?? null,
            'country_code' => $this['customer']['default_address']['country_code'] ?? null,
        ];
    }

    public function getTaxLines()
    {
        if ($this->tax_lines) {
            return collect($this->tax_lines)->map(function ($taxRate) {
                return [
                    'rate' => $taxRate['rate'] * 100,
                    'name' => $taxRate['title'],
                ];
            })
                ->toArray();
        }

        return [];
    }

    /**
     * @return null
     */
    protected function getShipByDate()
    {
        return null;
    }

    public static function getOrderStatus(array $order)
    {
        if (collect($order['line_items'])->sum('fulfillable_quantity')) {
            return 'open';
        } else {
            return 'closed';
        }
    }

    private function bindExternallyFulfilledQuantity(array $orderLines): array
    {

        /*
         * TODO: handle fulfillments that aren't externally fulfilled (need a method to compare fulfillments between sales channel and order lines)
         */
        return collect($orderLines)->map(function (array $orderLine) {
            if (! $orderLine['is_product']) {
                return $orderLine;
            }

            $externallyFulfilledQuantity = 0;
            if (! empty($this->fulfillments)) {
                $externallyFulfilledQuantity = collect($this['fulfillments'])
                    ->whereIn('status', ['open', 'success'])
                    ->where('service', '!=', 'gift_card')
                    /*
                     * This filter makes sure already processed fulfillments are not counted.
                     * TODO: I'm not sure if this filter makes sense... wouldn't we want to include already processed fulfillments?
                     */
                    ->filter(fn ($fulfillment) => ! $this->isFulfillmentProcessed($fulfillment['id']))
                    /*
                     * This filter makes sure external fulfilled quantity are only those considering those fulfillments
                     * prior to the inventory start date
                     */
                    ->filter(fn ($fulfillment) => Carbon::parse($fulfillment['created_at'])->utc()
                        ->lessThan(SettingRepository::getInventoryStartDate()))
                    ->pluck('line_items')
                    ->collapse()
                    ->where('id', $orderLine['sales_channel_line_id'])
                    ->sum('quantity');
            }
            $orderLine['externally_fulfilled_quantity'] = $externallyFulfilledQuantity;

            return $orderLine;
        })->toArray();
    }

    /**
     * @return SalesOrder|false|Builder|Model|mixed|object|null
     *
     * @throws BindingResolutionException
     * @throws Throwable
     * @throws InvalidProductWarehouseRouting|OpenStockTakeException
     */
    public function createSKUOrder(): mixed
    {
        customlog('shopifyOrdersBenchmark', $this->name.': shopify prep create order');
        // for old orders
        if (! $this->integrationInstance) {
            return false;
        }

        $existingOrder = $this->salesOrder;
        if (! $existingOrder) {
            /** @var SalesOrder|null $existingOrder */
            $existingOrder = SalesOrder::with([])->where('sales_order_number', $this['name'])
                ->whereHas('salesChannel', function ($builder) {
                    $builder->where('integration_instance_id', $this->integrationInstance->id);
                })->first();

            if ($existingOrder) {
                $this->sku_sales_order_id = $existingOrder->id;
                $this->save();
            }
        }

        if ($existingOrder) {
            return $existingOrder;
        }

        $currencyId = Currency::with([])->where('code', $this->currency)->value('id') ?: Currency::default()->id;

        /**
         * We must ensure that the sales-channel/sales-order-number uniqueness holds.
         * However, Shopify may have multiple orders with the same name. For actual duplicates,
         * we append the shopify order id to the name attribute. If the order isn't an actual
         * duplicate, thus, the name and the shopify order ids match, we update the order.
         *
         * @see SKU-4448
         */
        $existingOrder = SalesOrder::with([])
            ->where('sales_order_number', $this['name'])
            ->where('sales_channel_id', $this->integrationInstance->salesChannel->id)
            ->first();
        $salesOrderNumber = $this['name'];
        if ($existingOrder) {
            if ($existingOrder->shopifyOrder->getShopifyOrderId() !== $this->getShopifyOrderId()) {
                // Different orders with same name.
                $salesOrderNumber .= ' - '.$this->getShopifyOrderId();
            } else {
                // Same order as before, we simply update the order.
                return $this->updateSKUOrder();
            }
        }

        // We net out canceled quantity
        $lineItems = collect($this->bindExternallyFulfilledQuantity($this->getSkuOrderLines()))
            ->map(function ($lineItem) {
                if (! $lineItem['is_product']) {
                    return $lineItem;
                }

                $lineItemId = $lineItem['sales_channel_line_id'];

                $totalLineItemQty = $this->getLineItemQty($lineItemId);
                $totalUnfulfilledForLine = $this->getTotalUnfulfilledForLine($lineItemId);
                $canceled = $this->getTotalCanceledForLine($lineItemId);
                $totalNoRestockRefunds = $this->getTotalNoRestockRefundsForLineItem($lineItemId);
                $totalFulfilled = $this->getTotalFulfilledForLine($lineItemId);

                /*
                 * TODO: Right now we are netting out canceled quantity which means we lose the original quantity in sales order lines
                 *   We should figure out how to do this the right way or refactor how we deal with canceled quantity
                 */
                $quantityToCancel = min(
                    $totalNoRestockRefunds,
                    max(0, $totalLineItemQty - $canceled - $totalFulfilled - $totalUnfulfilledForLine)
                ) + $canceled;

                // Commenting this out and using quantity to cancel instead which should trigger a cancellation in the updateCanceledQuantity method
                //$lineItem['quantity'] -= $quantityToCancel;
                $lineItem['quantity_to_cancel'] = $quantityToCancel;

                return $lineItem;
            })->toArray();

        // Create the Order in Sku
        $data = [
            'sales_order_number' => $salesOrderNumber,
            'sales_channel_id' => $this->integrationInstance->salesChannel->id,
            'currency_id' => $currencyId,
            'order_date' => Carbon::parse($this['processed_at'])->setTimezone('UTC'),
            'store_id' => $this->integrationInstance->salesChannel->store_id,
            'shipping_method_id' => $this->getShippingMethodId(),
            'requested_shipping_method' => $this['shipping_lines'][0]['code'] ?? null,
            'ship_by_date' => $this->getShipByDate(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'is_tax_included' => $this->taxes_included == 'true',
            'sales_order_lines' => $lineItems,
            'financial_lines' => $this->getSkuFinancialLines(),
            'shipping_address' => $this->getSkuShippingAddress(),
            'billing_address' => $this->getSkuBillingAddress(),
            'customer' => $this->getSkuCustomerAddress(),
            'tags' => $this->getTags(),
            'tax_total' => (float) $this->current_total_tax,
        ];

        // FulfillmentManager extends SalesOrderManager
        /** @var FulfillmentManager $manager */
        $manager = app(FulfillmentManager::class);
        $salesOrder = $manager->createOrder($data);

        // Link Sku order to Shopify order
        $this->archived_at = null;
        $this->sku_sales_order_id = $salesOrder->id;
        $this->sku_order_updated_at = Carbon::now();
        $this->save();
        $this->setRelation('salesOrder', $salesOrder);
        if ($this->hasSalesChannelFulfillmentsOutOfSync()) {
            if (app(WarehouseRepository::class)->hasSingleShippableWarehouse() && $salesOrder->allProductsMapped()) {
                // We can be confident that the single shippable warehouse is already associated to the lines appropriately
                // since they have gone through the advanced routing logic
                // TODO: Need to manage shopify order mappings but that is a post processing step

                /*
                 * TODO: $data does not have the warehouse id.  Warehouse id is not assigned at this point.
                 *  Need another way to build fulfillment data
                 */
                $data['fulfillments'] = $this->getSkuFulfillments($salesOrder, $data);

                // Create Fulfillments and map Shopify Order Mappings for Fulfillments
                foreach ($data['fulfillments'] as $fulfillmentData) {
                    try {
                        // How can fulfillments be created here without a warehouse?
                        $fulfillment = $manager->fulfill($salesOrder, $fulfillmentData, false, false);
                        $shopifyFulfillment = collect($this->fulfillments)->where('id', $fulfillmentData['sales_channel_fulfillment_id'])->first();
                        $this->mapFulfillmentToSkuFulfillment($shopifyFulfillment, $fulfillment->id);
                    } catch (SalesOrderFulfillmentException $e) {
                        customlog('out-of-sync', $salesOrder->sales_order_number.' was determined to be out of sync (for order creation).  Sales order fulfillment exception occurred, so marking out of sync');
                        $salesOrder->notes()->create(['note' => 'Out of Sync: '.$e->getMessage()]);
                        $salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC;
                        $salesOrder->save();
                    } catch (ExternallyFulfilledCantBeFulfilledException $e) {
                        customlog('out-of-sync', $salesOrder->sales_order_number.' was determined to be out of sync (for order creation).  Not enough stock to fulfill, so marking out of sync');
                        $salesOrder->notes()->create(['note' => 'Out of Sync: '.$e->getMessage()]);
                        $salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC;
                        $salesOrder->save();
                    }
                }
            } else {
                // SKU-6178, Not sure why processable quantity affects whether to mark as out of sync or not.  If it is out of sync it is out of sync.  Adding an always true condition.
                if ($salesOrder->has_processable_quantity || true) {
                    customlog('out-of-sync', $salesOrder->sales_order_number.' was determined to be out of sync (for order creation).  Not a single shippable warehouse or not all products mapped, so marking out of sync');
                    $salesOrder->notes()->create(['note' => 'Out of Sync: Was determined to be out of sync (for order creation). Not a single shippable warehouse or not all products mapped']);
                    $salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC;
                    $salesOrder->save();
                }
            }
        }

        $this->createPaymentsFromCapturedTransactions($salesOrder);

        return $this->afterOrderProcessing();
    }

    /**
     * @throws Throwable
     */
    private function afterOrderProcessing(): SalesOrder
    {
        $this->salesOrder->refresh();
        $salesOrder = $this->salesOrder->load('warehousedProductLines');

        $salesOrder->taxRate()->associate((new TaxRateService())->getTaxRateForArray($this->getTaxLines()));

        // Moved this to before salesOrderManager->updateOrder so that the logic works based on payment status
        // Record order payments from transactions
        //$this->createPaymentsFromCapturedTransactions($salesOrder);

        if (! $salesOrder->isDraft()) {
            // We handle the refunds of the order.
            $salesOrder = $this->handleRefunds();
        }

        if ($salesOrder->isOpen()) {
            // We handle the statuses of the order.
            $this->handleOrderStatuses($salesOrder);
            if ($salesOrder->is_fully_fulfillable) {
                dispatch(new AutomatedSalesOrderFulfillmentJob($salesOrder))->onQueue('automatedFulfillments');
            }
        }

        return $salesOrder;
    }

    public function handleCancellations(array $lineItem): array
    {
        if (! $lineItem['is_product']) {
            return $lineItem;
        }

        /** @var SalesOrderLine|null $existingLine */
        $existingLine = $this->salesOrder->salesOrderLines()
            ->where('sales_channel_line_id', $lineItem['sales_channel_line_id'])
            ->first();
        if ($existingLine) {
            $lineItem['id'] = $existingLine->id;
            // We bind in the existing line warehouse id if available
            // since warehouse id is null by default from the sales order line DTO.
            $lineItem['warehouse_id'] = $existingLine->warehouse_id;
            $knownCancellation = $existingLine->canceled_quantity;
        } else {
            $knownCancellation = 0;
        }

        $lineItemId = $lineItem['sales_channel_line_id'];

        $shopifyOriginalQty = $this->getLineItemQty($lineItemId);
        $shopifyFulfillableQty = $this->getTotalUnfulfilledForLine($lineItemId);
        $shopifyCanceledRestockedQty = $this->getTotalCanceledForLine($lineItemId);
        $shopifyRefundedNoRestockQty = $this->getTotalNoRestockRefundsForLineItem($lineItemId);

        /*
         * We want to calculate quantity fulfilled in sku in case the update got processed late and the line already
         * got fulfilled.
         *
         * If the sku fulfilled quantity + shopify canceled quantity > original line quantity, we know a fulfillment occurred
         * on a cancelled line
         */
        $skuFulfillableQty = 0;
        if ($existingLine) {
            $skuFulfilledQty = $existingLine
                ->salesOrderFulfillmentLines()->whereHas('salesOrderFulfillment', function ($query) {
                    $query->where('status', SalesOrderFulfillment::STATUS_FULFILLED);
                })->sum('quantity');
            $skuFulfillableQty = max(0, $existingLine->quantity + $existingLine->canceled_quantity - $skuFulfilledQty);
            if (($shopifyCanceledRestockedQty + $shopifyRefundedNoRestockQty) > $skuFulfillableQty) {
                $lineItem['is_fulfilled_on_cancelled_line'] = true;
            }
        }
        $shopifyFulfilledQty = $this->getTotalFulfilledForLine($lineItemId);

        /*
         * TODO: Right now we are netting out canceled quantity which means we lose the original quantity in sales order lines
         *   We should figure out how to do this the right way or refactor how we deal with canceled quantity
         */

        /*
         * $shopifyRefundedNoRestockQty may contain both fulfilled and unfulfilled no restock refunds.  Fulfilled no restock refunds
         * are not cancellations, because cancellations must be pre-fulfillment.  To back into what is likely the correct no restock pre
         * fulfillment quantity, we take the lesser between the no restock quantity and the unfulfilled/uncanceled (by regular cancellation)/unfulfillable quantity
         *
         * This is an imperfect way to back into the pre-fulfillment no restock quantity, but practically it has worked out so far
         */
        // Shopify unfulfillable quantity is the max potential cancellable quantity
        $shopifyUnfulfillableQty = max(0,
            $shopifyOriginalQty - // Original Qty
            $shopifyCanceledRestockedQty - // Already clearly canceled accordingly to Shopify
            $shopifyFulfilledQty - // Shopify fulfilled quantity by nature is not cancelled
            $shopifyFulfillableQty // If it is fulfillable, it is not possible to be cancelled
        );
        $probableShopifyCanceledNoRestockQty = min($shopifyRefundedNoRestockQty, $shopifyUnfulfillableQty);
        $quantityToCancel = min($skuFulfillableQty, $probableShopifyCanceledNoRestockQty + $shopifyCanceledRestockedQty);
        $quantityToCancel = max(0, $quantityToCancel);

        //                customlog('cancelQty', $this->salesOrder->sales_order_number . ' ' . $lineItem['description'] . ': shopify cancel calculations', [
        //                    'shopifyOriginalQty' => $shopifyOriginalQty,
        //                    'shopifyFulfillableQty' => $shopifyFulfillableQty,
        //                    'shopifyFulfilledQty' => $shopifyFulfilledQty,
        //                    'shopifyCanceledRestockedQty' => $shopifyCanceledRestockedQty,
        //                    'shopifyRefundedNoRestockQty' => $shopifyRefundedNoRestockQty,
        //                    'skuFulfillableQty' => $skuFulfillableQty,
        //                    'shopifyUnfulfillableQty' => $shopifyUnfulfillableQty,
        //                    'probableShopifyCanceledNoRestockQty' => $probableShopifyCanceledNoRestockQty,
        //                    'quantityToCancel' => $quantityToCancel,
        //                ]);

        $lineItem['quantity_to_cancel'] = $quantityToCancel;

        // If updating an already fulfilled order, we don't want to reset fulfilled quantity
        unset($lineItem['fulfilled_quantity']);

        return $lineItem;
    }

    /**
     * @throws Throwable
     */
    public function updateSKUOrder(): ?SalesOrder
    {
        if (! $this->salesOrder) {
            return null;
        }
        customlog('shopifyOrdersBenchmark', $this->salesOrder->sales_order_number.': shopify prep update order');

        //Log::debug('Updating SKU order', ['order' => $this->salesOrder->sales_order_number]);
        // We bind in externally fulfilled quantity and canceled quantity.
        $lineItems = collect($this->bindExternallyFulfilledQuantity($this->getSkuOrderLines()))
            ->map(function (array $lineItem) {
                return $this->handleCancellations($lineItem);
            })->toArray();

        // Prepare data
        $data = [
            'tax_total' => (float) $this->current_total_tax,
            'is_tax_included' => $this->taxes_included == 'true',
            'shipping_method_id' => $this->getShippingMethodId(),
            'requested_shipping_method' => $this['shipping_lines'][0]['code'] ?? null,
            // We want to retain the ship by date
            //'ship_by_date' => $this->getShipByDate(),
            'shipping_address' => $this->getSkuShippingAddress(),
            'billing_address' => $this->getSkuBillingAddress(),
            'customer' => $this->getSkuCustomerAddress(),
            'financial_lines' => $this->getSkuFinancialLines(),
            'sales_order_lines' => $lineItems,
            'tags' => $this->getTags(),
            'on_hold' => $this->isOnHold(),
        ];

        $outOfSync = $this->hasSalesChannelFulfillmentsOutOfSync();
        if ($outOfSync && $this->salesOrder->isOpen()) {
            /** @var FulfillmentManager $manager */
            $manager = app(FulfillmentManager::class);
            $salesOrder = $this->salesOrder;
            if (app(WarehouseRepository::class)->hasSingleShippableWarehouse() && $salesOrder->allProductsMapped()) {
                $data['fulfillments'] = $this->getSkuFulfillments($salesOrder, $data);

                // Create Fulfillments and map Shopify Order Mappings for Fulfillments
                foreach ($data['fulfillments'] as $fulfillmentData) {
                    try {
                        // How can fulfillments be created here without a warehouse?
                        $fulfillment = $manager->fulfill($salesOrder, $fulfillmentData, false, false);
                        $shopifyFulfillment = collect($this->fulfillments)->where('id', $fulfillmentData['sales_channel_fulfillment_id'])->first();
                        $this->mapFulfillmentToSkuFulfillment($shopifyFulfillment, $fulfillment->id);
                    } catch (SalesOrderFulfillmentException $e) {
                        customlog('out-of-sync', $this->salesOrder->sales_order_number.' was determined to be out of sync (for order update).  Sales order fulfillment exception occurred, so marking out of sync');
                        $salesOrder->notes()->create(['note' => 'Out of Sync: '.$e->getMessage()]);
                        $salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC;
                        $salesOrder->save();
                    } catch (ExternallyFulfilledCantBeFulfilledException $e) {
                        customlog('out-of-sync', $this->salesOrder->sales_order_number.' was determined to be out of sync (for order update).  Not enough stock to fulfill, so marking out of sync');
                        $salesOrder->notes()->create(['note' => 'Out of Sync: '.$e->getMessage()]);
                        $salesOrder->fulfillment_status = SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC;
                        $salesOrder->save();
                    }
                }
            } else {
                customlog('out-of-sync', $this->salesOrder->sales_order_number.' was determined to be out of sync (for order update).  Not a single shippable warehouse or not all products mapped, so marking out of sync');
                $salesOrder->notes()->create(['note' => 'Out of Sync: Not a single shippable warehouse or not all products mapped']);
                $data['fulfillment_status'] = SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC;
            }

        } elseif ($this->salesOrder->fulfillment_status == SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC) {
            $data['fulfillment_status'] = SalesOrder::FULFILLMENT_STATUS_FULFILLED;
        }

        $this->createPaymentsFromCapturedTransactions($this->salesOrder);

        // Update order in sku
        /** @var SalesOrderManager $manager */
        $manager = app(SalesOrderManager::class);
        $manager->updateOrder(UpdateSalesOrderData::from([
            'salesOrder' => $this->salesOrder,
            'payload' => UpdateSalesOrderPayloadData::from($data),
        ]));

        $this->sku_order_updated_at = Carbon::now();
        $this->save();

        return $this->afterOrderProcessing();

    }

    public function findMatchingLines(Collection $orderLines, ProductListing $productListing)
    {
        return $orderLines->where('variant_id', $productListing->sales_channel_listing_id);
    }

    public function findMatchingLine(Collection $orderLines, ProductListing $productListing)
    {
        return $orderLines->firstWhere('variant_id', $productListing->sales_channel_listing_id);
    }

    public function isPreAuditTrail()
    {
        if (! $this->integrationInstance->audit_trail_start_date || $this->integrationInstance->audit_trail_start_date->lt(Carbon::parse($this->json_object['processed_at'], 'UTC'))) {
            return false;
        } else {
            return true;
        }
    }

    public function mapFulfillmentToSkuFulfillment(array $shopifyFulfillment, int $skuFulfillmentId): void
    {
        foreach ($shopifyFulfillment['line_items'] as $fulfillmentLineItem) {
            app(ShopifyOrderMappingRepository::class)
                ->upsert(
                    [
                        'shopify_order_id' => $this->id,
                        'line_id' => $fulfillmentLineItem['id'],
                        'line_type' => ShopifyOrderMapping::LINE_TYPE_LINE_ITEM,
                        'link_id' => $shopifyFulfillment['id'],
                        'link_type' => ShopifyOrderMapping::LINK_TYPE_FULFILLMENTS,
                    ],
                    [
                        'quantity' => $fulfillmentLineItem['quantity'],
                        'processed_at' => $shopifyFulfillment['created_at'],
                        'no_restock_fulfillments_applied' => null,
                        'sku_link_id' => $skuFulfillmentId,
                        'sku_link_type' => SalesOrderFulfillment::class,
                    ]);
        }
    }

    public function shouldSkip(): bool
    {
        /** @see SKU-4130 exclude archived orders when running process orders */
        if ($this->isArchived()) {
            return true;
        }

        $orderStatus = static::getOrderStatus($this->toArray());
        /*
        * If open orders aren't set to be processed or the order date is before the open start date, skip processing
        */
        if ($orderStatus == 'open') {
            return ! $this->integrationInstance->open_start_date || ($this->integrationInstance->open_start_date && $this->integrationInstance->open_start_date > $this->getOrderCreatedAtDate());
        }

        /*
        * If closed orders aren't set to be processed or the order date is before the closed start date, skip processing
        */
        if ($orderStatus == 'closed' || $orderStatus == 'refunded') {
            return ! $this->integrationInstance->closed_start_date || ($this->integrationInstance->closed_start_date && $this->integrationInstance->closed_start_date > $this->getOrderCreatedAtDate());
        }

        return false; // We shouldn't skip the order.
    }

    private function getOrderCreatedAtDate(): Carbon
    {
        return Carbon::parse($this->json_object['processed_at'])->timezone('UTC');
    }

    public function scopeLatestOrder(Builder $builder, int $integrationInstanceId, ?string $downloadedBy = null, string $column = 'updated_at')
    {
        return $builder->where('integration_instance_id', $integrationInstanceId)
            ->when($downloadedBy, function (Builder $builder) use ($downloadedBy) {
                $builder->where(function (Builder $builder) use ($downloadedBy) {
                    $builder->where('downloaded_by', $downloadedBy);
                    $builder->orWhere('updated_by', $downloadedBy);
                });
            })
            ->latest($column);
    }

    public function scopeWhereOrderDate(Builder $builder, string $operator, $value)
    {
        $orderDateAttribute = static::getOrderDateAttributeName();
        $builder->whereRaw("CAST({$orderDateAttribute} AS DATETIME) {$operator} CAST('".Carbon::parse($value)->toIso8601String()."' AS DATETIME)");
    }

    public function getAttribute($key)
    {
        $value = parent::getAttribute($key);
        if (is_null($value)) {
            $value = $this->json_object[$key] ?? null;
        }

        return $value;
    }

    public static function getNeedsUpdatingOrderNames(IntegrationInstance $integrationInstance): array
    {
        $minOrderDate = min($integrationInstance->open_start_date ?: now(), $integrationInstance->closed_start_date ?: now());

        return self::query()
            ->selectRaw('shopify_orders.name')
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereNull('archived_at') //@see SKU-4130 exclude archived orders when running process orders
            ->whereNotNull('sku_sales_order_id') // sku order exists
            // orders that need to update
            ->where(function (Builder $query) {
                $query->whereNull('sku_order_updated_at')
                    ->orWhereColumn('sku_order_updated_at', '<', 'sku_updated_at');
                // Not needed as per SKU-6044
                //->orWhereHas('orderMappings', fn ($q) => $q->where('link_type', ShopifyOrderMapping::LINK_TYPE_REFUNDS), '<', DB::raw('`refund_lines_count`'));
            })
            // order date (after converted to UTC timezone) greater than or equals to minOrderDate
            // ->whereRaw("CONVERT_TZ(STR_TO_DATE(SUBSTRING_INDEX(JSON_UNQUOTE(JSON_EXTRACT(`json_object`, '$.processed_at')), '+', 1), '%Y-%m-%dT%H:%i:%s'), CONCAT(IF(LOCATE('+', JSON_UNQUOTE(JSON_EXTRACT(`json_object`, '$.processed_at'))), '+', '-'), SUBSTRING_INDEX(JSON_UNQUOTE(JSON_EXTRACT(`json_object`, '$.processed_at')), IF(LOCATE('+', JSON_UNQUOTE(JSON_EXTRACT(`json_object`, '$.processed_at'))), '+', '-'), -1)), 'UTC') >= '{$minOrderDate->toDateTimeString()}'")
            ->whereRaw("processedAtUtc >= '{$minOrderDate->toDateTimeString()}'")
            ->pluck('shopify_orders.name')->toArray();
    }

    public static function getOrderNamesFromIds(IntegrationInstance $integrationInstance, array $ids): ?array
    {
        return self::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereIn('id', $ids)
            ->pluck('name')
            ->toArray();
    }

    public static function getUnprocessedOrderNames(IntegrationInstance $integrationInstance): array
    {
        /*
         * The intent is to only process orders that got the latest transaction data
         * Transaction data is latest if the updated at date is the same as sku_updated_at since transaction update
         * triggers sku_updated_at update
         */
        $unprocessed_line_items = self::query()
            ->from('shopify_orders', 'so')
            ->select([
                'so.id',
                'so.processed_at',
            ])
            ->selectRaw('sol.line_item as json_object')
            ->leftJoin('shopify_order_line_items as sol', 'sol.shopify_order_id', 'so.id')
            ->whereNull('so.sku_sales_order_id')
            ->whereNull('so.archived_at')
            // TODO: processed_at is in UTC?
            ->whereDate('so.processed_at', '>=', Helpers::setting(Setting::KEY_INVENTORY_START_DATE))
            ->where('so.integration_instance_id', $integrationInstance->id);

        $dataFileResource = Helpers::dto2csvFile(
            $unprocessed_line_items->get(),
            function ($data) {
                return [
                    $data['id'],
                    $data['processed_at'],
                    '',
                    '',
                    json_encode($data['json_object']),
                ];
            }
        );

        $collate = config('database.connections.mysql.collation');

        $createTemporaryTableQuery = <<<SQL
            CREATE TEMPORARY TABLE IF NOT EXISTS temporary_shopify_order_line_items (
                `shopify_order_id` BIGINT,
                `processed_at` DATETIME,
                `shopify_order_line_id` BIGINT AS (JSON_UNQUOTE(json_extract(json_object,'$.id'))) STORED,
                `fulfillable_quantity` int(11) AS (JSON_UNQUOTE(json_extract(json_object,'$.fulfillable_quantity'))) STORED,
                `json_object` json,
                INDEX (`shopify_order_id`),
                UNIQUE INDEX (`shopify_order_line_id`)
            ) ENGINE=INNODB DEFAULT COLLATE=$collate;
        SQL;

        DB::statement($createTemporaryTableQuery);

        $insertTemporaryTableQuery = "
            LOAD DATA LOCAL INFILE '".stream_get_meta_data($dataFileResource)['uri']."'
                INTO TABLE temporary_shopify_order_line_items
                FIELDS TERMINATED BY ','
                ENCLOSED BY '\"'
                ESCAPED BY '\\\\'
                LINES TERMINATED BY '\n'
        ";

        DB::statement($insertTemporaryTableQuery);

        $results = DB::select('
            SELECT so.`name`, SUM(tsoli.fulfillable_quantity), tsoli.processed_at
            FROM shopify_orders AS so
            INNER JOIN temporary_shopify_order_line_items AS tsoli
                ON tsoli.shopify_order_id = so.id
            WHERE
                so.sku_sales_order_id IS NULL
                AND so.archived_at IS NULL
                AND so.integration_instance_id = "'.$integrationInstance->id.'"
            GROUP BY so.shopify_order_id, so.name, tsoli.processed_at
            HAVING
                /*
                 * Make sure to only get orders passing the rules of
                 *
                 * open orders on or after open start date
                 * OR
                 * closed orders on or after closed start date
                 *
                 */
                (tsoli.processed_at >= "'.$integrationInstance->open_start_date.'"
                AND SUM(tsoli.fulfillable_quantity) > 0)
                OR
                (tsoli.processed_at >= "'.$integrationInstance->closed_start_date.'"
                AND SUM(tsoli.fulfillable_quantity) = 0)
        ');

        return array_column($results, 'name');
    }

    public static function getOrderFromOrderName(string $name, IntegrationInstance $integrationInstance): self
    {
        return self::query()->where('integration_instance_id', $integrationInstance->id)->where('name', $name)->first();
    }

    public function hasSalesChannelFulfillmentsOutOfSync(): bool
    {
        $salesChannelFulfillmentDto = $this->compareFulfillments();

        return
            count($salesChannelFulfillmentDto->fulfillmentsToSubmit) ||
            count($salesChannelFulfillmentDto->fulfillmentToCreate) ||
            count($salesChannelFulfillmentDto->differences);
    }

    public function compareFulfillments(): SalesChannelFulfillmentDto
    {
        // Step 1: Retrieve payload for eligible Shopify fulfillments
        $shopifyFulfillments = collect($this->fulfillments)
            ->whereIn('status', ['open', 'success'])
            ->where('service', '!=', 'gift_card')
            ->toArray();

        // Step 2: Retrieve sku fulfillment data
        $skuFulfillments = SalesOrderFulfillment::query()
            ->with('salesOrderFulfillmentLines.salesOrderLine')
            ->where('sales_order_id', $this->sku_sales_order_id)
            ->get();

        // Step 3-6: Compare fulfillments
        $fulfillmentsToCreate = new SalesOrderFulfillmentCollection();
        $fulfillmentsToSubmit = new SalesOrderFulfillmentCollection();
        $differences = [];

        // Fulfillments in Shopify but not in SKU
        foreach ($shopifyFulfillments as $shopifyFulfillment) {
            $found = false;
            if (Carbon::parse($shopifyFulfillment['created_at'])->lt(Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE)))) {
                continue;
            }

            $shopifyFulfillmentLineItems = $this->sanitizeShopifyFulfillmentLineItemsFromLineItemsPayload($shopifyFulfillment['line_items']);

            // Don't include fulfillments from shopify that only contain lines not requiring shipping (i.e. Navidium)
            if (empty($shopifyFulfillmentLineItems)) {
                continue;
            }

            /** @var SalesOrderFulfillment $skuFulfillment */
            foreach ($skuFulfillments as $skuFulfillment) {
                // TODO: This comparison does not work if there are multiple shopify fulfillments with the same tracking number
                if (@$shopifyFulfillment['tracking_number'] == $skuFulfillment->tracking_number) {
                    $found = true;

                    // Compare line items
                    $skuFulfillmentLineItems = collect($skuFulfillment->salesOrderFulfillmentLines)->map(function ($fulfillmentLine) {
                        return [
                            'sales_channel_line_id' => (string) $fulfillmentLine->salesOrderLine->sales_channel_line_id,
                            'quantity' => (int) $fulfillmentLine->quantity,
                            'product_id' => $fulfillmentLine->salesOrderLine->product_id,
                        ];
                    })->toArray();

                    $difference = Helpers::arrayDiffAssoc2($skuFulfillmentLineItems, $shopifyFulfillmentLineItems);
                    if ($difference) {
                        customlog('out-of-sync', $this->salesOrder->sales_order_number.': there is a difference between sku fulfillment lines items and shopify fulfillment line items', [
                            'skuFulfillmentLineItems' => $skuFulfillmentLineItems,
                            'shopifyFulfillmentLineItems' => $shopifyFulfillmentLineItems,
                        ]);
                        $differences[] = [
                            'sku_fulfillment' => $skuFulfillment->id,
                            'shopify_fulfillment' => $shopifyFulfillment['id'],
                            'type' => 'sku_fulfillment',
                            'difference' => $difference,
                        ];
                    }
                    break;
                }
            }

            if (! $found) {
                $fulfillmentsToCreate->add(SalesOrderFulfillmentDto::from([
                    'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
                    'fulfilled_at' => Carbon::parse($shopifyFulfillment['created_at'])->setTimezone('UTC'),
                    'shipping_method_id' => $this->getShippingMethodId(),
                    'fulfilled_shipping_method' => $shopifyFulfillment['tracking_company'] ?? SalesChannel::UNSPECIFIED_SHIPPING_METHOD,
                    'requested_shipping_method' => @$this['shipping_lines'][0]['code'] ?? SalesChannel::UNSPECIFIED_SHIPPING_METHOD,
                    'tracking_number' => count($shopifyFulfillment['tracking_numbers']) ? implode(', ', $shopifyFulfillment['tracking_numbers']) : SalesChannel::UNSPECIFIED_SHIPPING_METHOD,
                    'warehouse_id' => null,
                    'fulfillment_lines' => $shopifyFulfillmentLineItems,
                    'submit_to_shipping_provider' => false,
                    'submit_to_sales_channel' => false,
                ]));
            }
        }

        // Fulfillments in SKU but not in Shopify
        /** @var SalesOrderFulfillment $skuFulfillment */
        foreach ($skuFulfillments->where('status', SalesOrderFulfillment::STATUS_FULFILLED) as $skuFulfillment) {
            $found = false;

            foreach ($shopifyFulfillments as $shopifyFulfillment) {
                // TODO: This comparison does not work if there are multiple shopify fulfillments with the same tracking number
                if (@$shopifyFulfillment['tracking_number'] == $skuFulfillment->tracking_number) {
                    $found = true;

                    $shopifyFulfillmentLineItems = $this->sanitizeShopifyFulfillmentLineItemsFromLineItemsPayload($shopifyFulfillment['line_items']);

                    // Compare line items
                    $skuFulfillmentLineItems = collect($skuFulfillment->salesOrderFulfillmentLines)->map(function (SalesOrderFulfillmentLine $fulfillmentLine) {
                        return [
                            'sales_channel_line_id' => $fulfillmentLine->salesOrderLine->sales_channel_line_id,
                            'quantity' => (int) $fulfillmentLine->quantity,
                            'product_id' => $fulfillmentLine->salesOrderLine->product_id,
                        ];
                    })->toArray();

                    $difference = Helpers::arrayDiffAssoc2($shopifyFulfillmentLineItems, $skuFulfillmentLineItems);

                    if ($difference) {
                        customlog('out-of-sync', $this->salesOrder->sales_order_number.': there is a difference between shopify fulfillment lines items and sku fulfillment line items', [
                            'shopifyFulfillmentLineItems' => $shopifyFulfillmentLineItems,
                            'skuFulfillmentLineItems' => $skuFulfillmentLineItems,
                        ]);
                        $differences[] = [
                            'sku_fulfillment' => $skuFulfillment->id,
                            'shopify_fulfillment' => $shopifyFulfillment['id'],
                            'type' => 'shopify_fulfillment',
                            'difference' => $difference,
                        ];
                    }
                    break;
                }
            }

            if (! $found) {
                $fulfillmentsToSubmit->add($skuFulfillment);
            }
        }

        return SalesChannelFulfillmentDto::from([
            'fulfillmentToCreate' => $fulfillmentsToCreate,
            'fulfillmentsToSubmit' => $fulfillmentsToSubmit,
            'differences' => $differences,
        ]);

    }

    public function delete(): ?bool
    {
        app(ShopifyOrderMappingRepository::class)->deleteMappingsForShopifyOrder($this);

        ShopifyOrderLineItem::query()
            ->where('shopify_order_id', $this->id)
            ->delete();

        return parent::delete();
    }

    private function sanitizeShopifyFulfillmentLineItemsFromLineItemsPayload($line_items): array
    {
        $shopifyFulfillmentLineItems = collect($line_items);

        $shopifyFulfillmentLineItems = $shopifyFulfillmentLineItems->filter(function ($fulfillmentLine) {
            if (isset($fulfillmentLine['requires_shipping'])) {
                return $fulfillmentLine['requires_shipping'] == true;
            }

            return true;
        });

        return $shopifyFulfillmentLineItems->where('product_exists', true)->flatMap(function ($fulfillmentLine) {
            $shopifyProduct = ShopifyProduct::with('productListing')->where('sku', $fulfillmentLine['sku'])
                ->where('integration_instance_id', $this->integration_instance_id)
                ->first();

            if (! $shopifyProduct) {
                // Product not found, possibly handle this scenario or return a placeholder.
                return [[
                    'sales_channel_line_id' => (string) $fulfillmentLine['id'],
                    'quantity' => (int) $fulfillmentLine['quantity'],
                    'product_id' => null,
                ]];
            }

            $product = $shopifyProduct->productListing?->product;

            if ($product && $product->type == \App\Models\Product::TYPE_BUNDLE) {
                // The product is a bundle, get the components.
                return $product->components->map(function ($component) use ($fulfillmentLine) {
                    return [
                        'sales_channel_line_id' => (string) $fulfillmentLine['id'],
                        'quantity' => (int) $fulfillmentLine['quantity'] * $component->pivot->quantity, // Adjust if component quantity differs.
                        'product_id' => $component->id, // Use the component's product_id
                    ];
                })->all(); // Convert the collection to an array.
            }

            // Product is not a bundle or no components, return the single product.
            return [[
                'sales_channel_line_id' => (string) $fulfillmentLine['id'],
                'quantity' => (int) $fulfillmentLine['quantity'],
                'product_id' => $product ? $product->id : null,
            ]];
        })->toArray();
    }
}
