<?php

namespace App\Models;

use App\Exporters\MapsExportableFields;
use App\Importers\DataImporter;
use App\Importers\DataImporters\ShippingMethodDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * Class ShippingMethod.
 *
 *
 * @property int $id
 * @property int $shipping_carrier_id
 * @property string $name
 * @property int|null $delivery_min
 * @property int|null $delivery_max
 * @property array|null $included_locations
 * @property array|null $excluded_locations
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read string $full_name
 * @property-read string|null $type
 * @property-read ShippingCarrier $shippingCarrier
 */
class ShippingMethod extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable
{
    use Archive, HasFilters, HasSort;
    use HasFactory;

    const TYPE_INTERNATIONAL = 'international';

    const TYPE_DOMESTIC = 'domestic';

    const TYPE_BOTH = 'both';

    const TYPES = [
        'International' => self::TYPE_INTERNATIONAL,
        'Domestic' => self::TYPE_DOMESTIC,
        'Both' => self::TYPE_BOTH,
    ];

    protected $fillable = [
        'name',
        'shipping_carrier_id',
        'delivery_min',
        'delivery_max',
        'included_locations',
        'excluded_locations',
        'type',
    ];

    protected $casts = [
        'delivery_min' => 'integer',
        'delivery_max' => 'integer',
        'included_locations' => 'array',
        'excluded_locations' => 'array',
    ];

    protected $hidden = ['created_at', 'updated_at'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function shippingCarrier()
    {
        return $this->belongsTo(ShippingCarrier::class);
    }

    public function salesOrders()
    {
        return $this->hasMany(SalesOrder::class);
    }

    public function salesOrderFulfillments()
    {
        return $this->hasMany(SalesOrderFulfillment::class);
    }

    public function purchaseOrders()
    {
        return $this->hasMany(PurchaseOrder::class);
    }

    public function domesticCustomers()
    {
        return $this->hasMany(Customer::class, 'shipping_method_domestic_id');
    }

    public function internationalCustomers()
    {
        return $this->hasMany(Customer::class, 'shipping_method_international_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

//      public function getFullNameAttribute()
//      {
//        return $this->shippingCarrier ? $this->shippingCarrier->name.' '.$this->name : $this->name;
//      }

    //  public function getTypeAttribute()
    //  {
    //    $sellerCountry     = Helpers::setting( Setting::KEY_SELLER_COUNTRY, 'US' );
    //    $includedLocations = $this->included_locations ?? [];
    //
    //    if ( in_array( 'Worldwide', $includedLocations ) || ( in_array( $sellerCountry, $includedLocations ) && count( $includedLocations ) > 1 ) )
    //    {
    //      return 'international';
    //    }
    //
    //    if ( in_array( $sellerCountry, $includedLocations ) )
    //    {
    //      return 'domestic';
    //    }
    //
    //    return null;
    //  }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function save(array $options = [])
    {
        if ($this->shippingCarrier) {
            $this->full_name = $this->shippingCarrier->name.' '.$this->name;
        }

        return parent::save($options);
    }

    public function delete()
    {
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return parent::delete();
    }

    /**
     * Determine if the nominal code is used.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        $usage = [];
        // used in settings
        $settings = Setting::with([])->whereIn('key', [
            Setting::KEY_SO_DEFAULT_SM_DOMESTIC,
            Setting::KEY_SO_DEFAULT_SM_INTERNATIONAL,
        ])->get();
        if ($settings->firstWhere('value', $this->id) || $settings->where('value', null)->firstWhere('default_value', $this->id)) {
            $usage['settings'] = __('messages.shipping_methods.used_in_settings', ['id' => $this->name]);
        }

        // used in sales orders
        $this->loadCount('salesOrders');
        if ($this->sales_orders_count) {
            $usage['salesOrders'] = trans_choice('messages.currently_used', $this->sales_orders_count, [
                'resource' => 'sales order',
                'model' => 'shipping method('.$this->name.')',
            ]);
        }

        // used in customers
        $this->loadCount('domesticCustomers', 'internationalCustomers');
        if ($this->domestic_customers_count || $this->international_customers_count) {
            $usage['customers'] = trans_choice('messages.currently_used', $this->domestic_customers_count + $this->international_customers_count, [
                'resource' => 'customer',
                'model' => 'shipping method('.$this->name.')',
            ]);
        }

        return count($usage) ? $usage : false;
    }

    public function availableColumns()
    {
        return config('data_table.shipping_method.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return ['name', 'full_name'];
    }

    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new ShippingMethodDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return ShippingMethodDataImporter::getExportableFields();
    }
}
