<?php

namespace App\Models;

use App\Helpers;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\DB;

/**
 * Class ShipBySchedule
 *
 * @property-read Carbon $ship_by_date
 * @property-read int $product_id
 * @property-read string $product_name
 * @property-read string $product_sku
 * @property-read int $total_quantity
 * @property-read int $shortage_quantity
 * @property-read int $covered_quantity
 * @property-read int $uncovered_quantity
 * @property-read int $total_orders
 *
 * @property-read Product $product
 */
class ShipBySchedule extends Model implements Filterable, Sortable
{
    use HasFilters;
    use HasSort;

    public $timestamps = false;

    public $table = 'sales_order_lines';

    protected static function boot() {
        parent::boot();

        static::addGlobalScope('ship_by_schedule', function (Builder $builder) {

            $ordersPerDateQuery = SalesOrder::query()
                ->selectRaw("DATE_FORMAT(CONVERT_TZ(sales_orders.ship_by_date, 'UTC', '" . Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE) . "'), '%Y-%m-%d') AS ship_by_date")
                ->selectRaw("COUNT(DISTINCT id) AS total_orders")
                ->whereNotNull('sales_orders.ship_by_date')
                ->filter()
                ->groupBy(DB::raw("DATE_FORMAT(CONVERT_TZ(sales_orders.ship_by_date, 'UTC', '" . Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE) . "'), '%Y-%m-%d')"));

            $builder->selectRaw("DATE_FORMAT(CONVERT_TZ(sales_orders.ship_by_date, 'UTC', '" . Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE) . "'), '%Y-%m-%d') as ship_by_date")
                ->selectRaw('products.id as product_id')
                ->selectRaw('products.name as product_name')
                ->selectRaw('products.sku as product_sku')
                ->selectRaw('sales_orders.order_status as order_status')
                ->selectRaw("SUM(quantity) as total_quantity")
                ->selectRaw('SUM(backorder_queues.shortage_quantity) as shortage_quantity')
                ->selectRaw('COALESCE(SUM(unreleased_quantity), 0) as covered_quantity')
                ->selectRaw('(SUM(shortage_quantity) - COALESCE(SUM(unreleased_quantity), 0)) as uncovered_quantity')
                ->selectRaw('ordersPerDate.total_orders')
                ->join('products', 'products.id', 'sales_order_lines.product_id')
                ->join('sales_orders', 'sales_orders.id', 'sales_order_lines.sales_order_id')
                ->leftJoin('backorder_queues', 'backorder_queues.sales_order_line_id', '=', 'sales_order_lines.id')
                ->leftJoin('backorder_queue_coverages', 'backorder_queue_coverages.backorder_queue_id', '=', 'backorder_queues.id')
                ->leftJoinSub($ordersPerDateQuery, 'ordersPerDate', function ($join) {
                    $join->on(DB::raw("DATE_FORMAT(CONVERT_TZ(sales_orders.ship_by_date, 'UTC', '" . Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE) . "'), '%Y-%m-%d')"), '=', 'ordersPerDate.ship_by_date');
                })
                ->whereNotNull('sales_orders.ship_by_date')
                ->groupBy(DB::raw("DATE_FORMAT(CONVERT_TZ(sales_orders.ship_by_date, 'UTC', '" . Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE) . "'), '%Y-%m-%d'), products.id"));
        });
    }

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function salesOrder(): BelongsTo
    {
        return $this->belongsTo(SalesOrder::class);
    }

    public function availableColumns()
    {
        return config('data_table.ship_by_schedule.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return [];
    }

    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function scopeFilterWarehouse(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        return $builder->{$function . 'Has'}('warehouse', function (Builder $builder) use ($value, $operator) {
            $builder->where('warehouses.name', $operator, $value);
        });
    }

    public function scopeSortSalesOrder($builder, $relation, $ascending)
    {
        return $builder->orderBy($relation['key'], $ascending ? 'asc' : 'desc');
    }
}
