<?php

namespace App\Models;

use App\Contracts\HasReference;
use App\Exceptions\InsufficientStockException;
use App\Managers\ProductInventoryManager;
use App\Models\Concerns\HasAccountingTransactionLine;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\LogsActivity;
use App\Models\Contracts\Filterable;
use App\Services\InventoryManagement\InventoryDeficiencyActionType;
use App\Services\InventoryManagement\InventoryManager;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Collection;
use Spatie\Activitylog\LogOptions;

/**
 * Class SalesOrderFulfillmentLine.
 *
 *
 * @property int $id
 * @property int $sales_order_fulfillment_id
 * @property int $sales_order_line_id
 * @property float $quantity
 * @property array $metadata
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read float $prorationOfSalesOrderLine
 * @property SalesOrderLine $salesOrderLine
 * @property-read Collection|InventoryMovement[] $inventoryMovements
 * @property-read SalesOrderFulfillment $salesOrderFulfillment
 *
 * @method proration(string $proration_method)
 *
 * @property-read float $subtotal
 * @property-read float $shipping_cost
 */
class SalesOrderFulfillmentLine extends Model implements Filterable, HasReference
{
    use HasAccountingTransactionLine, HasFactory, HasFilters, LogsActivity;

    const PRORATION_WEIGHT = 'weight';

    const PRORATION_VALUE = 'value';

    const PRORATION_HYBRID = 'hybrid';

    protected $casts = [
        'quantity' => 'float',
        'metadata' => 'array',
    ];

    protected $guarded = [];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesOrderFulfillment()
    {
        return $this->belongsTo(SalesOrderFulfillment::class);
    }

    public function salesOrderLine()
    {
        return $this->belongsTo(SalesOrderLine::class);
    }

    public function inventoryMovements()
    {
        $relation = $this->morphMany(InventoryMovement::class, 'link');

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (InventoryMovement $inventoryMovement) {
                $inventoryMovement->delete();
            });
        });

        return $relation;
    }

    public function accountingTransactionLine(): MorphOne
    {
        return $this->morphOne(AccountingTransactionLine::class, 'link');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors and Mutators
    |--------------------------------------------------------------------------
    */

    public function prorationOfSalesOrderLine(): Attribute
    {
        return Attribute::get(
            fn () => $this->salesOrderLine->quantity == 0 ?
                0.00 : $this->quantity / $this->salesOrderLine->quantity
        );
    }

    public function brothers()
    {
        return $this->hasMany(self::class, 'sales_order_fulfillment_id', 'sales_order_fulfillment_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function delete(): ?bool
    {
        // For some reason returning the eloquent collection with $this->salesOrderLine calls an endless loop.
        $this->salesOrderLine()->each(/**
         * @throws Exception
         */ function (SalesOrderLine $salesOrderLine) {
            $salesOrderLine->fulfilled_quantity = max(0, $salesOrderLine->fulfilled_quantity - $this->quantity);
            $salesOrderLine->save();
        });
        $this->inventoryMovements()->delete();

        // cache inventory
        (new ProductInventoryManager($this->salesOrderLine->product_id))->updateProductInventoryAndAvgCost();

        if ($this->salesOrderFulfillment->salesOrderFulfillmentLines->count() == 1) {
            parent::delete();
            $this->salesOrderFulfillment->delete();
        } else {
            return parent::delete();
        }
        // This model triggers observers in Modules

        return true;
    }

    public function deleteWithFulfillmentIfLast($ignoreShippingProviderExceptions = false)
    {
        $fulfillment = $this->salesOrderFulfillment;
        $this->delete();

        if ($fulfillment->salesOrderFulfillmentLines()->count() == 0) {
            $fulfillment->delete($ignoreShippingProviderExceptions);
        }
    }

    /*
     * If even one fulfillment line gets deleted, delete the whole fulfillment
     */
    public function deleteWithFulfillment($ignoreShippingProviderExceptions = false)
    {
        $fulfillment = $this->salesOrderFulfillment;
        $this->delete();
        $fulfillment->delete($ignoreShippingProviderExceptions);
    }

    public function getReference(): ?string
    {
        return $this->getDrawer()->getReference();
    }

    public function getDrawer(): Model|SalesOrderFulfillment
    {
        return $this->salesOrderFulfillment;
    }

    /**
     * Reverse inventory movements reservations that reserve when approve sales order.
     *
     *
     * @throws InsufficientStockException
     */
    public function negateReservationMovements(?int $quantity = null, bool $ignoreExistingQty = false): bool
    {
        if ($this->salesOrderLine->inventoryMovements->isEmpty()) {
            // The order line doesn't have inventory movements,
            // we create them on the fly for the non-canceled quantity.
            //            InventoryManager::with(
            //                $this->salesOrderLine->warehouse_id,
            //                $this->salesOrderLine->product
            //            )->takeFromStock(
            //                $this->quantity,
            //                $this->salesOrderLine,
            //                true,
            //                null,
            //                InventoryDeficiencyActionType::ACTION_TYPE_POSITIVE_ADJUSTMENT
            //            );

            /*
             * Note we are disabling the ability for fulfillment to cause inventory deficiency action positive adjustments.
             * This led to a lot of confusion for the user.  Instead, if originated from a sales channel update, it should go into
             * fulfillment status out of sync with a note added.
             */
            InventoryManager::with(
                $this->salesOrderLine->warehouse_id,
                $this->salesOrderLine->product
            )->takeFromStock(
                $this->quantity,
                $this->salesOrderLine,
                true,
                null,
            );
        }

        $this->salesOrderLine->refresh();

        // inverse reservation movements
        if ($this->salesOrderLine->inventoryMovements->isNotEmpty()) {
            // reservation movements grouped by fifo layer id
            $reservationMovements = $this->salesOrderLine->inventoryMovements->where('layer_type', FifoLayer::class)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                ->groupBy('layer_id');

            $quantity = $quantity ?? $this->quantity;

            if (! count($reservationMovements)) {
                // There are no fifo movements for reservations
                return false;
            }

            foreach ($reservationMovements as $fifoLayerId => $movements) {
                $existingFulfillmentQty = 0;
                if (! $ignoreExistingQty) { // For cases such as updates
                    foreach ($this->salesOrderLine->salesOrderFulfillmentLines as $salesOrderFulfillmentLine) {
                        $existingFulfillmentQty += abs($salesOrderFulfillmentLine->inventoryMovements->where('layer_type', FifoLayer::class)
                            ->where('layer_id', $fifoLayerId)->sum('quantity'));
                    }
                }
                // movements summation on the fifo layer
                $sum = max($movements->sum('quantity') - $existingFulfillmentQty, 0);
                if ($sum > 0) { // we have reservation on this layer
                    /** @var InventoryMovement $reversReservationMovement */
                    $reversReservationMovement = $movements->first()->replicate();
                    // set movement date from fulfilled_at of the sales order fulfillment
                    $reversReservationMovement->inventory_movement_date = $this->salesOrderFulfillment->fulfilled_at ?? $this->salesOrderFulfillment->created_at;

                    if ($sum >= $quantity) { // we can fulfill whole quantity from this fifo layer
                        $reversReservationMovement->quantity = -$quantity;
                        $quantity = 0; // no remaining quantity
                    } else { // fulfill the remaining quantity on this layer
                        $reversReservationMovement->quantity = -$sum;
                        $quantity = $quantity - $sum; // remaining quantity
                    }
                    // Change link reference to the fulfillment line
                    $reversReservationMovement->link_type = self::class;
                    $reversReservationMovement->link_id = $this->id;

                    $this->inventoryMovements()->save($reversReservationMovement);
                }

                if ($quantity == 0) { // no remaining quantity, whole quantity fulfilled
                    break;
                }
            }

            if ($quantity != 0) { // the reservation quantity is less than required fulfillment quantity
                return false;
            }
        } else { // sales order lines does not have any reservation inventory movements
            return false;
        }

        return true;
    }

    public function getSubtotalAttribute()
    {
        return $this->quantity * $this->salesOrderLine->amount;
    }

    public function getShippingCostAttribute()
    {
        if (($sumSubtotal = $this->brothers->sum('subtotal')) == 0) {
            return 0;
        }

        $proration = $this->subtotal / $sumSubtotal;

        return $proration * $this->salesOrderFulfillment->cost;
    }

    public function toShipStationOrderItem()
    {
        $orderItem = $this->salesOrderLine->toShipStationOrderItem();
        $orderItem->quantity = $this->quantity;

        return $orderItem;
    }

    public function toStarshipitOrderItem(?Starshipit\StarshipitOrder $starshipOrder = null)
    {
        $orderItem = $this->salesOrderLine->toStarshipitOrderItem();
        $orderItem->quantity = $this->quantity;
        if ($starshipOrder && ! empty($starshipOrder->items)) {
            if ($starshipOrderItem = collect($starshipOrder->items)->firstWhere('description', $orderItem->description)) {
                $orderItem->item_id = $starshipOrderItem['item_id'];
            }
        }

        return $orderItem;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return ['quantity'];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * TODO: Does this belong in model?  If not we'd pass an extra parameter with the numerator fulfillment line
     *  and put the method in the SalesOrderFulfillmentRepository
     */
    public function getProration(string $proration_method): float
    {
        $is_weight_data_complete = true;

        if ($proration_method == 'weight' || $proration_method == 'hybrid') {
            $weight_tally = 0;
            /** @var SalesOrderFulfillmentLine $salesOrderFulfillmentLine */
            foreach ($this->salesOrderFulfillment->salesOrderFulfillmentLines as $salesOrderFulfillmentLine) {
                $weight = $salesOrderFulfillmentLine->salesOrderLine->product?->weight;
                if ($weight == 0) {
                    $is_weight_data_complete = false;
                }
                $weight_tally += $weight;
            }
        }

        switch ($proration_method) {
            case 'weight':
                if ($weight_tally == 0) {
                    return 0.00;
                }

                return $this->salesOrderLine->product?->weight / $weight_tally;
            case 'hybrid':
                if ($is_weight_data_complete) {
                    return $this->salesOrderLine->product?->weight / $weight_tally;
                } else {
                    $proration_method = 'value';
                }
            case 'value':
                $value_tally = 0;
                /** @var SalesOrderFulfillmentLine $salesOrderFulfillmentLine */
                foreach ($this->salesOrderFulfillment->salesOrderFulfillmentLines as $salesOrderFulfillmentLine) {
                    $value = $salesOrderFulfillmentLine->salesOrderLine->amount * $salesOrderFulfillmentLine->quantity;
                    $value_tally += $value;
                }
                if ($value_tally == 0) {
                    return 0.00;
                }

                return ($this->salesOrderLine->amount * $this->quantity) / $value_tally;
        }
    }

    /**
     * Reverse inventory movements reservations that reserve when approve sales order.
     *
     *
     * @throws InsufficientStockException
     */
    public function reverseReservedInventoryMovements(?int $quantity = null): bool
    {
        if ($this->salesOrderLine->inventoryMovements->isEmpty()) {
            // The order line doesn't have inventory movements,
            // we create them on the fly for the non-canceled quantity.
            InventoryManager::with(
                $this->salesOrderLine->warehouse_id,
                $this->salesOrderLine->product
            )->takeFromStock(
                $this->quantity,
                $this->salesOrderLine,
                true,
                null,
                InventoryDeficiencyActionType::ACTION_TYPE_POSITIVE_ADJUSTMENT
            );
        }

        $this->salesOrderLine->refresh();

        // inverse reservation movements
        if ($this->salesOrderLine->inventoryMovements->isNotEmpty()) {
            // reservation movements grouped by fifo layer id
            $reservationMovements = $this->salesOrderLine->inventoryMovements->where('layer_type', FifoLayer::class)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                ->groupBy('layer_id');

            $quantity = $quantity ?? $this->quantity;

            if (! count($reservationMovements)) {
                return true;
            }

            foreach ($reservationMovements as $fifoLayerId => $movements) {
                $existingFulfillmentQty = 0;
                foreach ($this->salesOrderLine->salesOrderFulfillmentLines as $salesOrderFulfillmentLine) {
                    $existingFulfillmentQty += abs($salesOrderFulfillmentLine->inventoryMovements->where('layer_type', FifoLayer::class)
                        ->where('layer_id', $fifoLayerId)->sum('quantity'));
                }
                /*$existingFulfillmentMovementsQuery = InventoryMovement::with([])->where('link_type', self::class)
                    ->where('layer_type', FifoLayer::class)
                    ->where('reference', $this->salesOrderLine->salesOrder->sales_order_number)
                    ->where('layer_id', $fifoLayerId);

                $existingFulfillmentQty = abs($existingFulfillmentMovementsQuery->sum('quantity'));
                */
                // movements summation on the fifo layer
                $sum = max($movements->sum('quantity') - $existingFulfillmentQty, 0);
                if ($sum > 0) { // we have reservation on this layer
                    /** @var InventoryMovement $reversReservationMovement */
                    $reversReservationMovement = $movements->first()->replicate();
                    // set movement date from fulfilled_at of the sales order fulfillment
                    $reversReservationMovement->inventory_movement_date = $this->salesOrderFulfillment->fulfilled_at ?? $this->salesOrderFulfillment->created_at;

                    if ($sum >= $quantity) { // we can fulfill whole quantity from this fifo layer
                        $reversReservationMovement->quantity = -$quantity;

                        $quantity = 0; // no remaining quantity
                    } else { // fulfill the remaining quantity on this layer
                        $reversReservationMovement->quantity = -$sum;

                        $quantity = $quantity - $sum; // remaining quantity
                    }

                    // Change link reference to the fulfillment line
                    $reversReservationMovement->link_type = self::class;
                    $reversReservationMovement->link_id = $this->id;

                    $this->inventoryMovements()->save($reversReservationMovement);
                } else {
                    throw_if(
                        ($movementQty = $movements->sum('quantity')) < $existingFulfillmentQty,
                        new Exception(
                            "Fulfillment movement qty ($existingFulfillmentQty) exceeds original 
                            movement qty ($movementQty). Fifo Layer: $fifoLayerId, 
                            Sales Order: {$this->salesOrderLine->salesOrder->sales_order_number} 
                            (sku: {$this->salesOrderLine->product->sku})"
                        )
                    );
                }

                if ($quantity == 0) { // no remaining quantity, whole quantity fulfilled
                    break;
                }
            }

            if ($quantity != 0) { // the reservation quantity is less than required fulfillment quantity
                return false;
            }
        } else { // sales order lines does not have any reservation inventory movements
            return false;
        }

        return true;
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->sales_order_fulfillment_id;
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->dontSubmitEmptyLogs();
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [
            'id' => $this->id,
            'sku' => $this->salesOrderLine->product?->sku,
            'description' => $this->salesOrderLine->description,
        ];
    }
}
