<?php

namespace App\Models;

use App\Abstractions\IsLineInterface;
use App\Contracts\HasReference;
use App\Helpers;
use App\Models\Concerns\HasAccountingTransactionLine;
use App\Models\Concerns\HasFilters;
use App\Repositories\SalesOrderLineFinancialsRepository;
use App\Services\InventoryManagement\InventoryDeficiencyActionType;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\InventoryManagement\PositiveInventoryEvent;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Collection;

/**
 * Class SalesCreditReturnLines.
 *
 *
 * @property int $id
 * @property int $sales_credit_return_id
 * @property int $sales_credit_line_id
 * @property int $quantity
 * @property string $action
 * @property int $reason_id
 * @property string $notes
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read SalesCreditLine $salesCreditLine
 * @property-read SalesCreditReturn $salesCreditReturn
 * @property-read ReturnReason $returnReason
 * @property-read Collection|FifoLayer[] $fifoLayers
 * @property-read FifoLayer $creationFifoLayer
 * @property-read Collection|InventoryMovement[] $inventoryMovements
 */
class SalesCreditReturnLine extends Model implements HasReference, PositiveInventoryEvent, IsLineInterface
{
    use HasAccountingTransactionLine,HasFactory,HasFilters;

    const ACTION_ADD_TO_STOCK = 'added_to_stock';

    const ACTION_DISCARD = 'discarded';

    const ACTION_AS_BLEMISHED = 'new_blemished_sku';

    const ACTION_NO_RESTOCK = 'no_restock';

    const ACTIONS = [
        self::ACTION_ADD_TO_STOCK,
        self::ACTION_DISCARD,
        self::ACTION_AS_BLEMISHED,
        self::ACTION_NO_RESTOCK,
    ];

    const ACTION_INVENTORY_IMPACT = [
        self::ACTION_ADD_TO_STOCK,
        self::ACTION_AS_BLEMISHED,
    ];

    protected $casts = [
        'quantity' => 'integer',
    ];

    protected $fillable = [
        'sales_credit_line_id',
        'quantity',
        'action',
        'reason_id',
        'notes',
    ];

    public function getLineParent(): Model
    {
        return $this->salesCreditReturn;
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesCreditReturn()
    {
        return $this->belongsTo(SalesCreditReturn::class);
    }

    public function salesCreditLine()
    {
        return $this->belongsTo(SalesCreditLine::class);
    }

    public function inventoryMovements()
    {
        return $this->morphMany(InventoryMovement::class, 'link');
    }

    public function returnReason()
    {
        return $this->belongsTo(ReturnReason::class, 'reason_id');
    }

    public function fifoLayers()
    {
        return $this->morphMany(FifoLayer::class, 'link');
    }

    public function creationFifoLayer()
    {
        return $this->morphOne(FifoLayer::class, 'link');
    }

    public function accountingTransactionLine()
    {
        return $this->morphOne(AccountingTransactionLine::class, 'link');
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function save(array $options = [])
    {
        if ($this->salesCreditLine->salesOrderLine) {
            app(SalesOrderLineFinancialsRepository::class)->invalidateForSalesOrderLineIds($this->salesCreditLine->salesOrderLine->id);
        }

        return parent::save($options);
    }

    public function delete()
    {
        if ($this->action == self::ACTION_ADD_TO_STOCK || $this->action == self::ACTION_AS_BLEMISHED) {
            $product = $this->salesCreditLine->product;
            if ($this->action == self::ACTION_AS_BLEMISHED) {
                $product = $this->getOriginatingMovement()?->product ?? $product;
            }

            if ($product) {
                InventoryManager::with(
                    $this->salesCreditReturn->warehouse_id,
                    $product
                )->removeAllStockFrom($this, InventoryDeficiencyActionType::ACTION_TYPE_POSITIVE_ADJUSTMENT);
            }
        }

        if ($this->salesCreditLine->salesOrderLine) {
            app(SalesOrderLineFinancialsRepository::class)->invalidateForSalesOrderLineIds($this->salesCreditLine->salesOrderLine->id);
        }

        // This model triggers observers in Modules

        return parent::delete();
    }

    /**
     * Add return line to inventory based on the action.
     */
    public function addToInventory(?string $blemishedCondition = null, ?string $blemishedReference = null)
    {
        if (! $this->salesCreditLine->product_id) {
            return;
        }

        if ($this->action == self::ACTION_ADD_TO_STOCK) {
            $this->load('salesCreditLine', 'salesCreditReturn');

            $this->addInventory($this->salesCreditLine->product_id);
        } elseif ($this->action == self::ACTION_AS_BLEMISHED) {
            $this->load('salesCreditLine', 'salesCreditReturn');
            // add/get blemished product
            $productBlemished = $this->salesCreditLine->product->addBlemished($blemishedCondition, $blemishedReference);

            $this->addInventory($productBlemished->product_id);
        }
    }

    private function addInventory($productId)
    {
        /** @var Product $product */
        $product = Product::with([])->findOrFail($productId);
        InventoryManager::with(
            $this->salesCreditReturn->warehouse_id,
            $product
        )->addToStock(
            $this->quantity,
            $this,
            true,
            true,
            $product->getUnitCostAtWarehouse($this->salesCreditReturn->warehouse_id)
        );
    }

    public function createFifoLayer(int $quantity, ?float $unitCost = null, ?int $productId = null): FifoLayer
    {
        // add a new fifo-layer linked with sales credit line
        $fifoLayer = new FifoLayer();
        $fifoLayer->fifo_layer_date = $this->getEventDate();
        // use productId because it maybe the id of the blemished product
        $fifoLayer->product_id = $productId ?: $this->salesCreditLine->product_id;
        $fifoLayer->original_quantity = $this->quantity;
        $fifoLayer->fulfilled_quantity = 0;
        $fifoLayer->warehouse_id = $this->salesCreditReturn->warehouse_id;
        $fifoLayer->total_cost = ($this->salesCreditLine->unit_cost ?? $unitCost) * $this->quantity;
        $this->fifoLayers()->save($fifoLayer);

        return $fifoLayer;
    }

    public function getUnitCost(): float
    {
        return
            $this->salesCreditLine->unit_cost > 0 ? $this->salesCreditLine->unit_cost :
            $this->salesCreditLine->product->getUnitCostAtWarehouse($this->salesCreditReturn->warehouse_id);
    }

    public function createInventoryMovements(int $quantity, FifoLayer $fifoLayer): void
    {
        // add a new inventory movement linked with return line and with the new fifo-layer
        $inventoryMovement = new InventoryMovement();
        $inventoryMovement->product_id = $fifoLayer->product_id;
        $inventoryMovement->inventory_movement_date = $this->getEventDate();
        $inventoryMovement->quantity = min($quantity, $this->quantity);
        $inventoryMovement->type = InventoryMovement::TYPE_RETURN;
        $inventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $inventoryMovement->warehouse_id = $this->salesCreditReturn->warehouse_id;
        $inventoryMovement->fifo_layer = $fifoLayer->id;
        $this->inventoryMovements()->save($inventoryMovement);
    }

    public function getOriginatingMovement(): ?InventoryMovement
    {
        return $this->inventoryMovements()->first();
    }

    public function getFifoLayer(): ?FifoLayer
    {
        // return $this->fifoLayers()->first();

        if ($movement = $this->getOriginatingMovement()) {
            return $movement->fifo_layer;
        }

        return null;
    }

    public function getReference(): string
    {
        return $this->getDrawer()->getReference();
    }

    public function getDrawer(): Model|SalesCredit
    {
        return $this->salesCreditReturn->salesCredit;
    }

    public function getLinkType(): string
    {
        return self::class;
    }

    public function getId(): int
    {
        return $this->id;
    }

    public function getProductId(): int
    {
        return $this->salesCreditLine->product_id;
    }

    public function getWarehouseId(): int
    {
        return $this->salesCreditReturn->warehouse_id;
    }

    public function getEventDate(): Carbon
    {
        return $this->salesCreditReturn->received_at->max(Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE), 'UTC'));
    }

    public function getQuantity(): int
    {
        return $this->quantity;
    }

    public function getType(): string
    {
        return InventoryMovement::TYPE_RETURN;
    }

    public function backorderQueueReleases()
    {
        return $this->morphMany(BackorderQueueRelease::class, 'link');
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeCreatesFifoLayer(Builder $query): Builder
    {
        return $query->whereIn('action', self::ACTION_INVENTORY_IMPACT);
    }
}
