<?php

namespace App\Models;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasAccountingTransaction;
use App\Models\Concerns\HasFilters;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionDataFromSalesCreditReturn;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Support\Collection;
use Throwable;

/**
 * Class SalesCreditReturns.
 *
 *
 * @property int $id
 * @property string $status
 * @property int $sales_credit_id
 * @property int $warehouse_id
 * @property int|null $shipping_method_id
 * @property string|null $tracking_number
 * @property Carbon|null $shipped_at
 * @property Carbon $received_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read SalesCredit $salesCredit
 * @property-read Collection|SalesCreditReturnLine[] $salesCreditReturnLines
 */
class SalesCreditReturn extends Model implements FinancialDocumentInterface
{
    use HandleDateTimeAttributes, HasAccountingTransaction, HasFactory, HasFilters;

    const STATUS_OPEN = 'open';

    const STATUS_CLOSED = 'closed';

    const STATUSES = [
        self::STATUS_OPEN,
        self::STATUS_CLOSED,
    ];

    protected $casts = [
        'shipped_at' => 'datetime',
        'received_at' => 'datetime',
    ];

    protected $fillable = [
        'warehouse_id',
        'shipping_method_id',
        'tracking_number',
        'shipped_at',
        'received_at',
        'updated_at'
    ];

    protected $attributes = [
        'status' => self::STATUS_CLOSED,
    ];

    /**
     * @throws Throwable
     */
    public function getAccountingTransactionData(): AccountingTransactionData
    {
        return (new BuildAccountingTransactionDataFromSalesCreditReturn($this))->handle();
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return $this->salesCredit->accountingTransaction;
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesCredit()
    {
        return $this->belongsTo(SalesCredit::class);
    }

    public function salesCreditReturnLines()
    {
        $relation = $this->hasMany(SalesCreditReturnLine::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (SalesCreditReturnLine $returnLine) {
                $returnLine->delete();
            });
        });

        return $relation;
    }

    public function fifoLayers(): HasManyThrough
    {
        return $this->hasManyThrough(
            FifoLayer::class,
            SalesCreditReturnLine::class,
            'sales_credit_return_id',
            'link_id',
            'id',
            'id'
        );
    }

    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function shippingMethod()
    {
        return $this->belongsTo(ShippingMethod::class);
    }

    public function accountingTransaction()
    {
        return $this->morphOne(AccountingTransaction::class, 'link');
    }

    public function getAccountingDateFieldName(): string
    {
        return 'received_at';
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function save(array $options = [])
    {
        // default warehouse from sales credit warehouse
        if (empty($this->warehouse_id)) {
            $this->warehouse_id = $this->salesCredit->to_warehouse_id;
        }

        // default received date
        if (empty($this->received_at)) {
            $this->received_at = now();
        }

        return parent::save($options);
    }

    public function delete()
    {
        $this->salesCreditReturnLines()->each(function (SalesCreditReturnLine $line) {
            $line->delete();
        });

        return parent::delete();
    }

    public function setReturnLines(array $returnLines)
    {
        foreach ($returnLines as $returnLine) {
            /** @var SalesCreditReturnLine $salesCreditReturnLine */
            $salesCreditReturnLine = $this->salesCreditReturnLines()->create($returnLine);
            $salesCreditReturnLine->addToInventory($returnLine['blemished_condition'] ?? null, $returnLine['blemished_reference'] ?? null);
        }
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeAccountingReady(Builder $builder): Builder
    {
        return $builder->where('status', SalesCreditReturn::STATUS_CLOSED)
            ->whereHas('salesCreditReturnLines', function (Builder $builder) {
                $builder->where('action', SalesCreditReturnLine::ACTION_INVENTORY_IMPACT);
            });
    }
}
