<?php

namespace App\Models;

use App\Contracts\HasReference;
use App\Helpers;
use App\Models\Concerns\HasAccountingTransactionLine;
use App\Repositories\SalesOrderLineFinancialsRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Casts\Attribute as Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;

/**
 * Class SalesCreditLine.
 *
 *
 * @property int $id
 * @property int $sales_credit_id
 * @property int $sales_order_line_id
 * @property string $description
 * @property int $product_id
 * @property float $amount
 * @property float $unit_cost
 * @property int $quantity
 * @property int $nominal_code_id
 * @property float $tax_allocation
 * @property int $tax_rate_id
 * @property float $tax_rate
 * @property float $proration
 * @property bool $is_product
 * @property bool $is_taxable
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read float $amount_in_tenant_currency
 * @property-read float $tax_allocation_in_tenant_currency
 * @property-read SalesOrderLine $salesOrderLine
 * @property-read bool $fully_received
 * @property-read int $received_quantity
 * @property-read int $unreceived_quantity
 * @property-read Product $product
 * @property-read NominalCode $nominalCode
 * @property-read Collection|SalesCreditReturnLine[] $salesCreditReturnLines
 * @property-read float $subtotal
 * @property-read float $subtotal_in_tenant_currency
 * @property-read float $tax_value
 * @property-read float $unitCostExtended
 * @property-read SalesCredit $salesCredit
 * @property-read TaxRate $taxRate
 */
class SalesCreditLine extends Model implements HasReference
{
    use HasAccountingTransactionLine, HasFactory;

    protected $casts = [
        'amount' => 'float',
        'unit_cost' => 'float',
        'quantity' => 'integer',
        'tax_allocation' => 'float',
        'is_product' => 'boolean',
        'proration' => 'float',
    ];

    protected $fillable = [
        'sales_order_line_id',
        'sales_credit_id',
        'quantity',
        'description',
        'product_id',
        'amount',
        'unit_cost',
        'nominal_code_id',
        'tax_allocation',
        'tax_rate_id',
        'is_product',
        'updated_at',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesCredit()
    {
        return $this->belongsTo(SalesCredit::class);
    }

    public function inventoryMovements()
    {
        return $this->morphMany(InventoryMovement::class, 'link');
    }

    public function salesOrderLine()
    {
        return $this->belongsTo(SalesOrderLine::class);
    }

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function nominalCode()
    {
        return $this->belongsTo(NominalCode::class);
    }

    public function salesCreditReturnLines()
    {
        return $this->hasMany(SalesCreditReturnLine::class);
    }

    public function taxRate()
    {
        return $this->belongsTo(TaxRate::class);
    }

    public function accountingTransactionLine()
    {
        return $this->morphOne(AccountingTransactionLine::class, 'link');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function amountInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->amount * $this->salesCredit->currency_rate);
    }

    public function unitCostExtended(): Attribute
    {
        return Attribute::get(fn () => $this->unit_cost * $this->quantity);
    }

    public function getFullyReceivedAttribute()
    {
        if (! $this->is_product) {
            return true;
        }

        return $this->received_quantity >= $this->quantity;
    }

    public function getReceivedQuantityAttribute()
    {
        if (! $this->is_product) {
            return 0;
        }

        return $this->salesCreditReturnLines->sum('quantity');
    }

    public function getUnreceivedQuantityAttribute()
    {
        if (! $this->is_product) {
            return 0;
        }

        return $this->quantity - $this->received_quantity;
    }

    public function getSubtotalAttribute()
    {
        return $this->quantity * $this->amount;
    }

    public function subtotalInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->subtotal * $this->salesCredit->currency_rate);
    }

    public function taxAllocationInTenantCurrency(): Attribute
    {
        return Attribute::get(fn () => $this->tax_allocation * $this->salesCredit->currency_rate);
    }

    public function getTaxValueAttribute()
    {
        return $this->quantity * $this->amount * $this->tax_allocation / 100;
    }

    /*
    |--------------------------------------------------------------------------
    | Function
    |--------------------------------------------------------------------------
    */

    public function delete()
    {
        $this->salesCreditReturnLines()->each(function (SalesCreditReturnLine $line) {
            $line->delete();
        });

        if ($this->salesOrderLine) {
            app(SalesOrderLineFinancialsRepository::class)->invalidateForSalesOrderLineIds($this->salesOrderLine->id);
        }

        return parent::delete();
    }

    public function save(array $options = [])
    {
        // fill data from sales order line
        if (! $this->exists && empty($this->description) && $this->sales_order_line_id) {
            $this->fill($this->salesOrderLine->only([
                'description',
                'product_id',
                'tax_allocation',
                'is_product',
            ]));
            $this->unit_cost = $this->salesOrderLine->salesOrderLineFinancial->perUnit('cogs');
            app(SalesOrderLineFinancialsRepository::class)->invalidateForSalesOrderLineIds($this->salesOrderLine->id);
        }

        if (empty($this->amount)) {
            $this->amount = $this->salesOrderLine->amount;
        }
        if (empty($this->nominal_code_id)) {
            $this->nominal_code_id = Helpers::setting(Setting::KEY_NC_MAPPING_COGS);
        }

        if (empty($this->is_product)) {
            $this->is_product = (bool) $this->product_id;
        }

        return parent::save($options);
    }

    public function getReference(): ?string
    {
        return $this->salesCredit->getReference();
    }
}
