<?php

namespace App\Models;

use App\Integrations\Amazon;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * @deprecated We will remove it after finished from refactoring {@see Integration}
 *
 * Class SalesChannelType
 *
 * @property int $id
 * @property string $name
 * @property string $description
 * @property string $category
 * @property string|null $country
 * @property string|null $image_url
 * @property array|null $integration_settings_template
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 */
class SalesChannelType extends Model implements Filterable
{
    use HasFactory;
    use HasFilters;

    /**
     * Sales Channel types (Class name in SalesChannels).
     */
    const TYPE_AMAZON = 'Amazon';

    const TYPE_EBAY = 'Ebay';

    const TYPE_SHOPIFY = 'Shopify';

    const TYPE_WOOCOMMERCE = 'WooCommerce';

    const TYPE_WALMART = 'Walmart';

    const TYPE_Google_Express = 'GoogleExpress';

    const TYPE_Magento = 'Magento';

    const TYPE_SKU = 'SKU';

    //  All sales channels types
    const TYPES = [
        self::TYPE_AMAZON,
        self::TYPE_EBAY,
        self::TYPE_SHOPIFY,
        self::TYPE_WOOCOMMERCE,
        self::TYPE_WALMART,
        self::TYPE_Google_Express,
        self::TYPE_Magento,
    ];

    /**
     * Array of sales channels types as in DB.
     */
    const SALES_CHANNEL_TYPES = [
        self::TYPE_SKU => [
            'id' => 1,
            'name' => 'sku.io',
            'description' => 'Manual Orders',
            'type' => self::TYPE_SKU,
        ],
        self::TYPE_AMAZON => [
            'id' => 2,
            'name' => 'amazon',
            'description' => 'Amazon.com',
            'type' => self::TYPE_AMAZON,
        ],
        self::TYPE_EBAY => [
            'id' => 3,
            'name' => 'ebay',
            'description' => 'eBay.com',
            'type' => self::TYPE_EBAY,
        ],
        self::TYPE_SHOPIFY => [
            'id' => 4,
            'name' => 'shopify',
            'description' => 'Shopify',
            'type' => self::TYPE_SHOPIFY,
        ],
        self::TYPE_WOOCOMMERCE => [
            'id' => 5,
            'name' => 'woocommerce',
            'description' => 'WooCommerce',
            'type' => self::TYPE_WOOCOMMERCE,
        ],
        self::TYPE_WALMART => [
            'id' => 6,
            'name' => 'walmart',
            'description' => 'Walmart',
            'type' => self::TYPE_WALMART,
        ],
        self::TYPE_Google_Express => [
            'id' => 7,
            'name' => 'google_express',
            'description' => 'Google Express',
            'type' => self::TYPE_Google_Express,
        ],
        self::TYPE_Magento => [
            'id' => 8,
            'name' => 'magento',
            'description' => 'Magento',
            'type' => self::TYPE_Magento,
        ],
    ];

    /**
     * Categories.
     */
    const CATEGORY_MARKETPLACE = 'marketplace';

    const CATEGORY_STORE = 'store';

    const CATEGORY_WEB_STORE = 'web_store';

    const CATEGORY_SOCIAL_MEDIA = 'social_media';

    const CATEGORIES = [
        self::CATEGORY_MARKETPLACE,
        self::CATEGORY_STORE,
        self::CATEGORY_WEB_STORE,
        self::CATEGORY_SOCIAL_MEDIA,
    ];

    protected $casts = [
        'integration_settings_template' => 'array',
    ];

    protected $hidden = ['created_at', 'updated_at', 'integration_settings_template'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function sales_channels()
    {
        return $this->hasMany(SalesChannel::class);
    }

    /**
     * Get Type (class name) by id.
     *
     *
     * @return mixed
     *
     * @throws Exception
     */
    public static function getTypeByID($id)
    {
        $types = collect(array_values(self::SALES_CHANNEL_TYPES));
        $type = $types->firstWhere('id', $id);
        if (! $type) {
            throw new Exception('Channel type not found');
        }

        return $type['type'];
    }

    /**
     * Get id by type.
     *
     *
     * @return mixed
     *
     * @throws Exception
     */
    public static function getIdByType(string $type)
    {
        $types = collect(array_values(self::SALES_CHANNEL_TYPES));
        $type = $types->firstWhere('type', $type);
        if (! $type) {
            throw new Exception('Channel type not found');
        }

        return $type['id'];
    }

    /**
     * Return amazon credentials based on marketplace.
     */
    public static function getAmazonDeveloperCredentials(?string $marketplaceId = null): array
    {
        /**
         * Default US.
         */
        $developer_credentials = [
            'developerId' => env('AMZ_US_DEVELOPER_ID'),
            'keyId' => env('AMZ_US_DEVELOPER_ACCESS_KEY'),
            'secretKey' => env('AMZ_US_DEVELOPER_SECRET_KEY'),
            'marketplaceId' => Amazon::MARKETPLACE_US,
        ];

        if ($marketplaceId == Amazon::MARKETPLACE_CANADA) {
            $developer_credentials = [
                'developerId' => env('AMZ_CANADA_DEVELOPER_ID') ?? env('AMZ_US_DEVELOPER_ID'),
                'keyId' => env('AMZ_CANADA_DEVELOPER_ACCESS_KEY'),
                'secretKey' => env('AMZ_CANADA_DEVELOPER_SECRET_KEY'),
                'marketplaceId' => Amazon::MARKETPLACE_CANADA,
            ];
        } elseif ($marketplaceId == Amazon::MARKETPLACE_UK) {
            $developer_credentials = [
                'developerId' => env('AMZ_UK_DEVELOPER_ID'),
                'keyId' => env('AMZ_UK_DEVELOPER_ACCESS_KEY'),
                'secretKey' => env('AMZ_UK_DEVELOPER_SECRET_KEY'),
                'marketplaceId' => Amazon::MARKETPLACE_UK,
            ];
        }

        return $developer_credentials;
    }

    /**
     * credentials attribute that taken from .env file by type.
     *
     *
     * @throws Exception
     */
    public function getCredentialsAttribute(): array
    {
        $type = self::getTypeByID($this->id);
        switch ($type) {
            case self::TYPE_AMAZON:
                return [
                    //          'developer_id' => env( 'AMZ_US_DEVELOPER_ID' ),
                ];
            case self::TYPE_EBAY:
                $ebay_environment = env('EBAY_ENVIRONMENT', 'sandbox');
                if ($ebay_environment == 'sandbox') {
                    return [
                        'appId' => env('EBAY_SANDBOX_US_APP_ID'),
                        'devId' => env('EBAY_SANDBOX_US_DEV_ID'),
                        'certId' => env('EBAY_SANDBOX_US_CERT_ID'),
                        'ruName' => env('EBAY_SANDBOX_US_RU_NAME_ID'),
                    ];
                }

                return [
                    'appId' => env('EBAY_US_APP_ID'),
                    'devId' => env('EBAY_US_DEV_ID'),
                    'certId' => env('EBAY_US_CERT_ID'),
                    'ruName' => env('EBAY_US_RU_NAME_ID'),
                ];
            case self::TYPE_SHOPIFY:
                return [
                    'api_key' => env('SHOPIFY_API_KEY'),
                    'api_secret_key' => env('SHOPIFY_API_SECRET'),
                ];
            case self::TYPE_Google_Express:
                return [
                    'client_id' => env('GOOGLE_CLIENT_ID'),
                    'client_secret' => env('GOOGLE_CLIENT_SECRET'),
                ];
            default:
                return [];
        }
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return ['name', 'description', 'category', 'country'];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->availableColumns();
    }
}
