<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Models\Abstractions\AbstractReportable;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphTo;

/**
 * TODO: Make this a normalized model, with reportable_id, reportable_type
 *  linkable_type possibilities would be Product (product allocated), FinancialLineType (order allocated), NominalCode (overhead from cost invoices)
 *
 * @property int $id
 * @property Carbon $date
 * @property int $reportable_id
 * @property string $reportable_type
 * @property int $quantity
 * @property int $num_orders
 * @property float $revenue
 * @property float $revenue_allocated
 * @property float $revenue_credits
 * @property-read float $total_revenue
 * @property float $cost
 * @property float $cost_allocated
 * @property float $cost_invoices
 * @property float $cost_returned
 * @property-read float $total_cost
 * @property-read float $profit
 * @property-read float $profit_margin
 * @property Carbon $created_at
 * @property Carbon $updated_at
 * @property-read AbstractReportable|Product|FinancialLineType|NominalCode $reportable
 */
class ReportingDailyFinancial extends Model implements Filterable, Sortable, UniqueFieldsInterface
{
    use BulkImport,
        HasFactory,
        HasFilters,
        HasSort;

    protected $guarded = [];

    protected $casts = [
        'date' => 'datetime',
    ];

    const INVALID_FINANCIALS_KEY = 'invalid_financials_dates';

    public static function getUniqueFields(): array
    {
        return ['date', 'reportable_id', 'reportable_type'];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function reportable(): MorphTo
    {
        return $this->morphTo();
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getProductAttribute(): ?Product
    {
        if ($this->reportable_type === Product::class) {
            return $this->reportable;
        }

        return null;
    }

    /*
    |--------------------------------------------------------------------------
    | Filter/Sort
    |--------------------------------------------------------------------------
    */

    public function availableColumns()
    {
        return config('data_table.reporting_daily_financial.columns');
    }

    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    public function generalFilterableColumns(): array
    {
        return ['date', 'product'];
    }

    public function sortableColumns(): array
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeTrailingDays(Builder $query, int $trailing_days): Builder
    {
        return $query->where('date', '>=', Carbon::now()->subDays($trailing_days));
    }

    public function scopeStartDate(Builder $query, $date): Builder
    {
        return $query->where('date', '>=', Carbon::parse($date));
    }

    public function scopeEndDate(Builder $query, $date): Builder
    {
        return $query->where('date', '<', Carbon::parse($date));
    }

    public function scopeFilterReportable(Builder $query, array $relation, string $operator, $value, $conjunction): Builder
    {
        $function = $conjunction == 'and' ? 'whereHasMorph' : 'orWhereHasMorph';

        $query->{$function}('reportable', [Product::class], function ($q) use ($operator, $value) {
            $q->filterKey('sku', $operator, $value);
        })->orWhereHasMorph('reportable', [FinancialLineType::class], function ($q) use ($operator, $value) {
            $q->filterKey('name', $operator, $value);
        });
        if (in_array($operator, ['!=', 'doesNotContain', 'isEmpty'])) {
            $query->orWhereNull($this->{$relation['name']}()->getForeignKeyName());
        }

        return $query;
    }
}
