<?php

namespace App\Models;

use App\Contracts\HasReference;
use App\Exceptions\NegativeInventoryFulfilledSalesOrderLinesException;
use App\Helpers;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasAccountingTransactionLine;
use App\Services\InventoryManagement\PositiveInventoryEvent;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use InvalidArgumentException;
use Kirschbaum\PowerJoins\PowerJoins;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Throwable;

/**
 * Class PurchaseOrderShipmentReceiptLine.
 *
 *
 * @property int $id
 * @property int $purchase_order_shipment_receipt_id
 * @property int $purchase_order_shipment_line_id
 * @property int $purchase_order_line_id
 * @property int $quantity
 * @property float $total_cost
 * @property-read float $prorationOfPurchaseOrderLine
 * @property-read PurchaseOrderShipmentLine $purchaseOrderShipmentLine
 * @property-read PurchaseOrderShipmentReceipt $purchaseOrderShipmentReceipt
 * @property-read PurchaseOrderLine $purchaseOrderLine
 * @property-read InventoryMovement[]|Collection $inventoryMovements
 * @property-read BackorderQueueRelease[]|Collection $backorderQueueReleases
 * @property-read AccountingTransactionLine $accountingTransactionLine
 */
class PurchaseOrderShipmentReceiptLine extends Model implements HasReference, PositiveInventoryEvent
{
    use BulkImport, HasAccountingTransactionLine, HasFactory, PowerJoins;

    protected $guarded = [];

    protected $casts = [
        'quantity' => 'integer',
    ];

    protected $touches = ['purchaseOrderShipmentReceipt'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function purchaseOrderShipmentReceipt()
    {
        return $this->belongsTo(PurchaseOrderShipmentReceipt::class);
    }

    public function purchaseOrderShipmentLine()
    {
        return $this->belongsTo(PurchaseOrderShipmentLine::class);
    }

    public function purchaseOrderLine(): BelongsTo
    {
        return $this->belongsTo(PurchaseOrderLine::class);
    }

    public function inventoryMovements()
    {
        return $this->morphMany(InventoryMovement::class, 'link');
    }

    public function getReference(): string
    {
        return $this->getDrawer()->getReference();
    }

    public function getDrawer(): Model|PurchaseOrder
    {
        return $this->purchaseOrderShipmentReceipt->purchaseOrderShipment->purchaseOrder;
    }

    public function backorderQueueReleases()
    {
        return $this->morphMany(BackorderQueueRelease::class, 'link');
    }

    public function accountingTransactionLine()
    {
        return $this->morphOne(AccountingTransactionLine::class, 'link');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors and Mutators
    |--------------------------------------------------------------------------
    */

    public function prorationOfPurchaseOrderLine(): Attribute
    {
        return Attribute::get(
            fn () => $this->purchaseOrderShipmentLine->purchaseOrderLine->quantity == 0 ?
                0.00 : $this->quantity / $this->purchaseOrderShipmentLine->purchaseOrderLine->quantity
        );
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function fifoLayers()
    {
        return $this->morphMany(FifoLayer::class, 'link');
    }

    /**
     * Creates fifo layer for the given quantity.
     */
    public function createFifoLayer(int $quantity, ?float $unitCost = null, ?int $productId = null): FifoLayer
    {
        if ($quantity > $this->quantity) {
            throw new InvalidArgumentException("Requesting to create fifo quantity: {$quantity} which is greater than receipt quantity of: {$this->quantity}");
        }

        $fifoLayer = new FifoLayer;
        $fifoLayer->original_quantity = $quantity;
        $fifoLayer->fulfilled_quantity = 0;
        $fifoLayer->fifo_layer_date = $this->getEventDate();
        $fifoLayer->product_id = $this->purchaseOrderShipmentLine->purchaseOrderLine->product_id;
        $fifoLayer->warehouse_id = $this->purchaseOrderShipmentLine->purchaseOrderLine->purchaseOrder->destination_warehouse_id;

        $this->purchaseOrderShipmentReceipt->calculateReceiptLinesCost([$this]);
        $fifoLayer->total_cost = $this->total_cost; // Total cost calculated
        $this->fifoLayers()->save($fifoLayer);

        return $fifoLayer;
    }

    public function getUnitCost(): float
    {
        $this->purchaseOrderShipmentReceipt->calculateReceiptLinesCost([$this]);

        return $this->quantity == 0 ? 0 : $this->total_cost / $this->quantity;
    }

    public function createInventoryMovements(int $quantity, FifoLayer $fifoLayer): void
    {
        $inventoryMovement = new InventoryMovement();
        $inventoryMovement->product_id = $fifoLayer->product_id;
        $inventoryMovement->inventory_movement_date = $this->getEventDate();
        $inventoryMovement->quantity = $quantity;
        $inventoryMovement->type = InventoryMovement::TYPE_PURCHASE_RECEIPT;
        $inventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $inventoryMovement->warehouse_id = $fifoLayer->warehouse_id;
        $inventoryMovement->warehouse_location_id = Warehouse::with([])->findOrFail($inventoryMovement->warehouse_id)->defaultLocation->id ?? null;
        $inventoryMovement->fifo_layer = $fifoLayer->id;
        $this->inventoryMovements()->save($inventoryMovement);
    }

    public function reduceReceivedQuantityCacheOnOrderLine(?int $quantity = null): void
    {
        $orderLine = $this->purchaseOrderShipmentLine->purchaseOrderLine;
        $orderLine->update([
            'received_quantity' => max(0, $orderLine->received_quantity - ($quantity ?? $this->quantity)),
        ]);
    }

    /**
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     * @throws Throwable
     */
    public function delete(): ?bool
    {
        // Update quantity received cache
        $this->reduceReceivedQuantityCacheOnOrderLine();

        $this->inventoryMovements->each(function (InventoryMovement $inventoryMovement) {
            $inventoryMovement->layer?->delete();
            $inventoryMovement->delete();
        });

       // this triggers observers in modules
        if ($this->purchaseOrderShipmentReceipt->purchaseOrderShipmentReceiptLines->count() == 1) {
            parent::delete();
            $this->purchaseOrderShipmentReceipt()->delete();
        } else {
            return parent::delete();
        }

        return true;
    }

    public function getOriginatingMovement(): ?InventoryMovement
    {
        return $this->inventoryMovements->first();
    }

    public function getFifoLayer(): ?FifoLayer
    {
        return $this->fifoLayers->firstOrFail();
    }

    public function getLinkType(): string
    {
        return self::class;
    }

    public function getId(): int
    {
        return $this->id;
    }

    public function getProductId(): int
    {
        return $this->purchaseOrderShipmentLine->purchaseOrderLine->product_id;
    }

    public function getWarehouseId(): int
    {
        return $this->purchaseOrderShipment->purchaseOrder->destination_warehouse_id;
    }

    public function getEventDate(): Carbon
    {
        return $this->purchaseOrderShipmentReceipt->received_at->max(Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE), 'UTC'));
    }

    public function getQuantity(): int
    {
        return $this->quantity;
    }

    public function getType(): string
    {
        return InventoryMovement::TYPE_PURCHASE_RECEIPT;
    }
}
