<?php

namespace App\Models;

use App\Contracts\HasReference;
use App\Models\Concerns\BulkImport;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Kirschbaum\PowerJoins\PowerJoins;

/**
 * Class PurchaseOrderShipmentLine.
 *
 *
 * @property int $id
 * @property int $purchase_order_shipment_id
 * @property int $purchase_order_line_id
 * @property int $quantity
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read bool $fully_received
 * @property-read Collection|PurchaseOrderShipmentReceiptLine[] $purchaseOrderShipmentReceiptLines
 * @property-read int $unreceived_quantity
 * @property-read int $received_quantity
 * @property-read PurchaseOrderLine $purchaseOrderLine
 * @property-read PurchaseOrderShipment $purchaseOrderShipment
 */
class PurchaseOrderShipmentLine extends Model implements HasReference
{
    use BulkImport, HasFactory, PowerJoins;

    protected $fillable = ['purchase_order_shipment_id', 'purchase_order_line_id', 'quantity'];

    protected $casts = ['quantity' => 'integer'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function purchaseOrderShipment()
    {
        return $this->belongsTo(PurchaseOrderShipment::class);
    }

    public function purchaseOrderLine()
    {
        return $this->belongsTo(PurchaseOrderLine::class);
    }

    public function purchaseOrderShipmentReceiptLines()
    {
        return $this->hasMany(PurchaseOrderShipmentReceiptLine::class);
    }

    public function getReference(): ?string
    {
        return $this->purchaseOrderShipment->getReference();
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getFullyReceivedAttribute()
    {
        return $this->quantity <= $this->purchaseOrderShipmentReceiptLines->sum('quantity');
    }

    public function getUnreceivedQuantityAttribute()
    {
        return $this->quantity - $this->received_quantity;
    }

    public function getReceivedQuantityAttribute()
    {
        return $this->purchaseOrderShipmentReceiptLines->sum('quantity');
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public function delete(): ?bool
    {
        if ($this->purchaseOrderShipment->purchaseOrderShipmentLines->count() == 1)
        {
            parent::delete();
            $this->purchaseOrderShipment->purchaseOrderShipmentReceipts()->delete();
            $this->purchaseOrderShipment()->delete();
        } else {
            return parent::delete();
        }

        return true;
    }
}
