<?php

namespace App\Models;

use App\Contracts\HasReference;
use App\Exceptions\NegativeInventoryFulfilledSalesOrderLinesException;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Collection;
use Throwable;

/**
 * Class PurchaseOrderShipment.
 *
 *
 * @property int $id
 * @property int $purchase_order_id
 * @property Carbon $shipment_date
 * @property int|null $fulfilled_shipping_method_id
 * @property string|null $fulfilled_shipping_method
 * @property string $tracking
 * @property Carbon|null $fully_received_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read bool $fully_received
 * @property-read Collection|PurchaseOrderShipmentLine[] $purchaseOrderShipmentLines
 * @property-read PurchaseOrder $purchaseOrder
 * @property-read SalesOrderFulfillment $salesOrderFulfillment
 * @property-read int $lines_quantity
 * @property-read int $received_lines_quantity
 * @property-read int $sales_order_fulfillment_id
 * @property-write Carbon $fulfilled_at For dropship
 * @property-write string $tracking_number For dropship
 *
 * @method static withCostValues(?Builder $builder = null)
 */
class PurchaseOrderShipment extends Model implements Filterable, HasReference
{
    use HandleDateTimeAttributes, HasFactory, HasFilters;

    protected $fillable = [
        'purchase_order_id',
        'shipment_date',
        'fulfilled_shipping_method_id',
        'shipping_method_id',
        'fulfilled_shipping_method',
        'tracking',
        'fulfilled_at',
        'sales_order_fulfillment_id',
    ];

    protected $casts = [
        'shipment_date' => 'datetime',
        'fully_received_at' => 'datetime',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function purchaseOrder()
    {
        return $this->belongsTo(PurchaseOrder::class);
    }

    public function fulfilledShippingMethod()
    {
        return $this->belongsTo(ShippingMethod::class, 'fulfilled_shipping_method_id');
    }

    public function purchaseOrderShipmentLines()
    {
        return $this->hasMany(PurchaseOrderShipmentLine::class);
    }

    public function getReference(): ?string
    {
        return $this->purchaseOrder->getReference();
    }

    public function purchaseOrderShipmentReceipts()
    {
        $relation = $this->hasMany(PurchaseOrderShipmentReceipt::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (PurchaseOrderShipmentReceipt $shipmentReceipt) {
                $shipmentReceipt->delete();
            });
        });

        return $relation;
    }

    public function salesOrderFulfillment(): BelongsTo
    {
        return $this->belongsTo(SalesOrderFulfillment::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    /**
     * Detect if whole purchase order shipment lines received.
     */
    public function getFullyReceivedAttribute(): bool
    {
        $this->load('purchaseOrderShipmentLines', 'purchaseOrderShipmentLines.purchaseOrderShipmentReceiptLines');

        foreach ($this->purchaseOrderShipmentLines as $purchaseOrderShipmentLine) {
            if (! $purchaseOrderShipmentLine->fully_received) {
                return false;
            }
        }

        return true;
    }

    public function getLinesQuantityAttribute()
    {
        return $this->purchaseOrderShipmentLines->sum('quantity');
    }

    public function getReceivedLinesQuantityAttribute()
    {
        return $this->purchaseOrderShipmentLines->sum('received_quantity');
    }

    public function setFulfilledAtAttribute($value)
    {
        $this->shipment_date = $value;
    }

    public function setTrackingNumberAttribute($value)
    {
        $this->tracking = $value;
    }

    public function setShippingMethodIdAttribute($value)
    {
        $this->fulfilled_shipping_method_id = $value;
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * Mark as Received.
     */
    public function received(?Carbon $receiptDate = null)
    {
        if ($this->fully_received) {
            $this->fully_received_at = $receiptDate ?: Carbon::now();
            $this->save();
        }
        $this->purchaseOrder->refreshReceiptStatus($receiptDate);
    }

    /**
     * @throws NegativeInventoryFulfilledSalesOrderLinesException
     * @throws Throwable
     */
    public function delete($deleteReceipts = true): ?bool
    {
        if ($deleteReceipts) {
            $this->purchaseOrderShipmentReceipts->each(function (PurchaseOrderShipmentReceipt $receipt) {
                $receipt->delete();
            });
        }

        $this->purchaseOrderShipmentLines()->delete();

        $deleted = parent::delete();

        if ($this->purchaseOrder->dropshipping) {
            $salesOrderFulfillment = $this->salesOrderFulfillment;
            if ($salesOrderFulfillment) {
                $salesOrderFulfillment->delete(fromDropshipShipmentDeletion: true);
            }
        }

        return $deleted;
    }

    /**
     * Aggregates purchase order shipment lines by either purchase order line ids
     * or line references on purchase order lines.
     */
    public static function aggregateShipmentLines(array $shipmentLines): array
    {
        // Bind in purchase order line ids for lines with line references
        $shipmentLines = array_map(function ($line) {
            if (isset($line['purchase_order_line_reference']) && ! isset($line['purchase_order_line_id'])) {
                $line['purchase_order_line_id'] = PurchaseOrderLine::with([])->where('line_reference', $line['purchase_order_line_reference'])->firstOrFail(['id'])->id;
            }

            return $line;
        }, $shipmentLines);

        $uniqueLineIds = array_unique(array_map(function ($line) {
            return $line['purchase_order_line_id'];
        }, $shipmentLines));
        $uniqueLines = [];

        foreach ($uniqueLineIds as $lineId) {
            $matched = array_values(array_filter($shipmentLines, function ($line) use ($lineId) {
                return $line['purchase_order_line_id'] === $lineId;
            }));

            $uniqueLines[] = array_merge($matched[0], [
                'purchase_order_line_id' => $lineId,
                'quantity' => array_sum(array_values(array_map(function ($line) {
                    return $line['quantity'];
                }, $matched))),
            ]);
        }

        return $uniqueLines;
    }

    public function receiveAll(array $data): PurchaseOrderShipmentReceipt
    {
        /** @var PurchaseOrderShipmentReceipt $receipt */
        $receipt = $this->purchaseOrderShipmentReceipts()->create(array_merge($data, ['purchase_order_shipment_id' => $this->id]));

        $receiptLines = [];
        $this->purchaseOrderShipmentLines->each(function (PurchaseOrderShipmentLine $shipmentLine) use ($receipt, &$receiptLines) {
            if (! $shipmentLine->purchaseOrderLine->fully_received) {
                // Add the receipt line
                $receiptLines[] = [
                    'purchase_order_shipment_receipt_id' => $receipt->id,
                    'purchase_order_shipment_line_id' => $shipmentLine->id,
                    'quantity' => $shipmentLine->unreceived_quantity,
                ];
            }
        });

        // Create the receipt lines
        $receipt->purchaseOrderShipmentReceiptLines()->createMany($receiptLines);

        if (! empty($data['warehouse_id'])) {
            // Update inventory audit trial
            $receipt->addLinesToInventory($receipt->purchaseOrderShipmentReceiptLines, $data['warehouse_id']);
        }

        // Set the receipt status of the shipment
        $this->received($receipt->received_at);

        return $receipt;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return ['shipment_date', 'shipping_method_id', 'tracking', 'fully_received_at'];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->availableColumns();
    }

    public function scopeWithCostValues(Builder $builder): Builder
    {
        return $builder->with([
            'purchaseOrderShipmentLines.purchaseOrderLine' => fn ($q) => $q->withCostValues(),
        ]);
    }
}
