<?php

namespace App\Models;

use App\Abstractions\AdvancedShipmentNoticeInterface;
use App\Abstractions\FinancialDocumentInterface;
use App\Abstractions\HasNotesInterface;
use App\Abstractions\SendEmailContextInterface;
use App\Abstractions\UniqueFieldsInterface;
use App\Contracts\HasDateInterface;
use App\Contracts\HasReference;
use App\Data\AccountingTransactionData;
use App\Enums\AccountingTransactionTypeEnum;
use App\Events\PurchaseOrderApproved;
use App\Events\PurchaseOrderSubmitted;
use App\Exporters\BaseExporter;
use App\Exporters\Jasper\PurchaseOrderPicklistTransformer;
use App\Exporters\MapsExportableFields;
use App\Exporters\TransformsExportData;
use App\Helpers;
use App\Helpers\ExcelHelper;
use App\Importers\DataImporter;
use App\Importers\DataImporters\PurchaseOrderDataImporter;
use App\Importers\ImportableInterface;
use App\Jobs\Amazon\GetInboundShipments;
use App\Jobs\GeneratePurchaseOrderInvoice;
use App\Jobs\GeneratePurchaseOrderPicklist;
use App\Jobs\SyncBackorderQueueCoveragesJob;
use App\Managers\ProductInventoryManager;
use App\Models\Concerns\Archive;
use App\Models\Concerns\CachesOrderCurrency;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasNotesTrait;
use App\Models\Concerns\HasSort;
use App\Models\Concerns\LogsActivity;
use App\Models\Concerns\TaxRateTrait;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Notifications\SubmitPurchaseOrderToSupplierNotification;
use App\Repositories\FifoLayerRepository;
use App\Response;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionDataFromPurchaseOrder;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Modules\Amazon\Abstractions\AmazonInboundShipmentSourceInterface;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipment;
use Spatie\Activitylog\LogOptions;
use Spatie\Tags\HasTags;
use Throwable;

/**
 * Class PurchaseOrder.
 *
 *
 * @property int $id
 * @property Carbon $purchase_order_date
 * @property Carbon $other_date
 * @property string $purchase_order_number
 * @property int $sequence
 * @property string $order_status
 * @property string $submission_status
 * @property string $receipt_status
 * @property string $shipment_status
 * @property string $invoice_status
 * @property string $submission_format
 * @property string $tracking_number
 * @property float $total_cost
 * @property int $payment_term_id
 * @property int $incoterm_id
 * @property int $store_id
 * @property int|null $requested_shipping_method_id
 * @property string|null $requested_shipping_method
 * @property int $supplier_id
 * @property int $supplier_warehouse_id
 * @property int $destination_warehouse_id
 * @property int $destination_address_id
 * @property int $currency_id
 * @property float $currency_rate
 * @property int $currency_id_tenant_snapshot
 * @property bool $is_tax_included
 * @property int|null $tax_rate_id
 * @property float|null $tax_rate
 * @property float $tax_total
 * @property int|null $sales_order_id
 * @property string|null $supplier_notes
 * @property Carbon|null $estimated_delivery_date
 * @property Carbon|null $approved_at
 * @property Carbon|null $last_submitted_at
 * @property Carbon|null $last_supplier_confirmed_at
 * @property Carbon|null $finalized_at
 * @property Carbon|null $fully_shipped_at
 * @property Carbon|null $fully_received_at
 * @property Carbon|null $fully_invoiced_at
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read  bool $dropshipping
 * @property-read SalesOrder $salesOrder
 * @property-read Warehouse $destinationWarehouse
 * @property-read \Illuminate\Database\Eloquent\Collection|PurchaseOrderLine[] $purchaseOrderLines
 * @property-read \Illuminate\Database\Eloquent\Collection|PurchaseOrderLine[] $productLines
 * @property-read Supplier $supplier
 * @property-read Store $store
 * @property-read bool $fully_shipped
 * @property-read bool $fully_invoiced
 * @property-read bool $fully_received
 * @property-read bool $partially_received
 * @property-read float $product_total
 * @property-read float $additional_cost
 * @property-read float $additional_cost_in_tenant_currency
 * @property-read float $total
 * @property-read float $invoice_total
 * @property-read Collection|PurchaseOrderShipment[] $purchaseOrderShipments
 * @property-read Collection|PurchaseOrderShipmentReceipt[] $purchaseOrderShipmentReceipts
 * @property-read Currency $currency
 * @property-read Currency $currencyTenantSnapshot
 * @property-read bool $finalized
 * @property-read bool $unsubmitted
 * @property-write string $currency_code
 * @property-write string $payment_term_name
 * @property-write string $incoterm_code
 * @property-write string $supplier_name
 * @property-write string $destination_warehouse_name
 * @property-write string $shipping_method_name
 * @property-read  Address $destination
 * @property-read AccountingTransaction $receivingDiscrepancy
 * @property Collection|PurchaseInvoice[] $purchaseInvoices
 * @property-read AmazonFbaInboundShipment $inboundShipmentRelation
 * @property-read AmazonNewFbaInboundShipment $inboundNewShipmentRelation
 * @property-read ShippingMethod $shippingMethod
 * @property-read AccountingTransaction $accountingTransaction
 * @property Carbon|null $asn_last_sent_at
 *
 * @method static withCostValues(?Builder $builder = null)
 */
class PurchaseOrder extends Model
    implements
    AmazonInboundShipmentSourceInterface,
    Filterable,
    FinancialDocumentInterface,
    HasNotesInterface,
    HasReference,
    ImportableInterface,
    MapsExportableFields,
    Sortable,
    TransformsExportData,
    UniqueFieldsInterface,
    AdvancedShipmentNoticeInterface,
    SendEmailContextInterface
{
    use Archive;
    use CachesOrderCurrency;
    use HandleDateTimeAttributes;
    use HasFactory;
    use HasFilters;
    use HasNotesTrait;
    use HasSort;
    use HasTags;
    use LogsActivity;
    use TaxRateTrait;

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['taxrate.name', '*'])
            ->logExcept(['updated_at', 'tax_rate_id', 'receipt_status'])

            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    const STATUS_DRAFT = 'draft';

    const STATUS_OPEN = 'open';

    const STATUS_CLOSED = 'closed';

    const STATUS = [
        self::STATUS_DRAFT,
        self::STATUS_OPEN,
        self::STATUS_CLOSED,
    ];

    const SUBMISSION_STATUS_UNSUBMITTED = 'unsubmitted';

    const SUBMISSION_STATUS_SUBMITTED = 'submitted';

    const SUBMISSION_STATUS_CHANGE_REQUEST_SUPPLIER = 'change_supplier';

    const SUBMISSION_STATUS_CHANGE_REQUEST_BUYER = 'change_buyer';

    const SUBMISSION_STATUS_FINALIZED = 'finalized';

    const SUBMISSION_STATUS_CANCELED = 'canceled'; // by Supplier (After Submission)

    const SUBMISSION_STATUS_VOIDED = 'voided '; // by Buyer (Before Submission)

    const SUBMISSION_STATUS_QUEUED = 'queued';

    const SUBMISSION_STATUS = [
        self::SUBMISSION_STATUS_UNSUBMITTED,
        self::SUBMISSION_STATUS_SUBMITTED,
        self::SUBMISSION_STATUS_CHANGE_REQUEST_BUYER,
        self::SUBMISSION_STATUS_CHANGE_REQUEST_SUPPLIER,
        self::SUBMISSION_STATUS_FINALIZED,
        self::SUBMISSION_STATUS_CANCELED,
        self::SUBMISSION_STATUS_VOIDED,
        self::SUBMISSION_STATUS_QUEUED,
    ];

    const RECEIPT_STATUS_UNRECEIVED = 'unreceived';

    const RECEIPT_STATUS_RECEIVED = 'received';

    const RECEIPT_STATUS_PARTIALLY_RECEIVED = 'partially_received';

    const RECEIPT_STATUS_DROPSHIP = 'dropship';

    const RECEIPT_STATUS = [
        self::RECEIPT_STATUS_UNRECEIVED,
        self::RECEIPT_STATUS_RECEIVED,
        self::RECEIPT_STATUS_PARTIALLY_RECEIVED,
        self::RECEIPT_STATUS_DROPSHIP,
    ];

    const SHIPMENT_STATUS_UNSHIPPED = 'unshipped';

    const SHIPMENT_STATUS_SHIPPED_WAREHOUSE = 'shipped_to_warehouse';

    const SHIPMENT_STATUS_SHIPPED_CUSTOMER = 'shipped_to_customer'; // In the case of dropship orders

    const SHIPMENT_STATUS = [
        self::SHIPMENT_STATUS_UNSHIPPED,
        self::SHIPMENT_STATUS_SHIPPED_WAREHOUSE,
        self::SHIPMENT_STATUS_SHIPPED_CUSTOMER,
    ];

    const INVOICE_STATUS_UNINVOICED = 'uninvoiced';

    const INVOICE_STATUS_PARTIALLY_INVOICED = 'partially_invoiced';

    const INVOICE_STATUS_INVOICED = 'invoiced'; // Ready to submit to Accounting Software

    const INVOICE_STATUS_INVOICE_PAID = 'invoice_paid'; // Can be determined by pulling status from Accounting Software

    const INVOICE_STATUS = [
        self::INVOICE_STATUS_UNINVOICED,
        self::INVOICE_STATUS_PARTIALLY_INVOICED,
        self::INVOICE_STATUS_INVOICED,
        self::INVOICE_STATUS_INVOICE_PAID,
    ];

    const SUBMISSION_FORMAT_PDF_ATTACHMENT = 'email_pdf_attachment';

    const SUBMISSION_FORMAT_MANUAL = 'manual';

    const SUBMISSION_FORMAT_CSV_ATTACHMENT = 'email_csv_attachment';

    const SUBMISSION_FORMAT_EMAIL_PDF_AND_CSV_ATTACHMENTS = 'email_pdf_csv_attachments';

    const SUBMISSION_FORMATS = [
        self::SUBMISSION_FORMAT_PDF_ATTACHMENT,
        self::SUBMISSION_FORMAT_MANUAL,
        self::SUBMISSION_FORMAT_CSV_ATTACHMENT,
        self::SUBMISSION_FORMAT_EMAIL_PDF_AND_CSV_ATTACHMENTS,
    ];

    protected $casts = [
        'purchase_order_date'     => 'datetime',
        'other_date'              => 'datetime',
        'estimated_delivery_date' => 'datetime',
        'archived_at' => 'datetime',
        'tracking_number' => 'string',
        'sequence' => 'integer',
        'fully_received_at' => 'datetime:Y-m-d H:i:s',
        'fully_shipped_at' => 'datetime:Y-m-d H:i:s',
        'asn_last_sent_at' => 'datetime',
    ];

    protected $fillable = [
        'purchase_order_number',
        'purchase_order_date',
        'other_date',
        'submission_status',
        'currency_code',
        'currency_id',
        'currency_rate',
        'submission_format',
        'payment_term_id',
        'payment_term_name',
        'incoterm_id',
        'incoterm_code',
        'store_id',
        'shipping_method_id',
        'requested_shipping_method_id',
        'requested_shipping_method',
        'shipping_method_name',
        'supplier_id',
        'supplier_name',
        'supplier_warehouse_id',
        'destination_warehouse_id',
        'destination_warehouse_name',
        'estimated_delivery_date',
        'supplier_notes',
        'internal_notes',
        'approved_at',
        'last_submitted_at',
        'tracking_number',
        'is_tax_included',
        'tax_rate_id',
        'sequence',
        'order_status',
        'sales_order_id',
        'destination_address_id'
    ];

    protected $attributes = [
        'order_status' => self::STATUS_DRAFT,
        'submission_status' => self::SUBMISSION_STATUS_UNSUBMITTED,
        'receipt_status' => self::RECEIPT_STATUS_UNRECEIVED,
        'shipment_status' => self::SHIPMENT_STATUS_UNSHIPPED,
        'invoice_status' => self::INVOICE_STATUS_UNINVOICED,
    ];

    protected $appends = ['product_total', 'additional_cost', 'tax_cost', 'total'];

    protected $touches = ['purchaseOrderShipmentReceipts'];

    public static function getUniqueFields(): array
    {
        return ['purchase_order_number'];
    }

    public static function getLinesRelationName(): string
    {
        return 'purchaseOrderLines';
    }

    public static function getNumberFieldName(): string
    {
        return 'purchase_order_number';
    }

    /**
     * @throws Throwable
     */
    public function getAccountingTransactionData(): AccountingTransactionData
    {
        return (new BuildAccountingTransactionDataFromPurchaseOrder($this))->handle();
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return null;
    }

    public function getAccountingDateFieldName(): string
    {
        return 'purchase_order_date';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function paymentTerm()
    {
        return $this->belongsTo(PaymentTerm::class);
    }

    public function incoterm()
    {
        return $this->belongsTo(Incoterm::class);
    }

    public function store()
    {
        return $this->belongsTo(Store::class);
    }

    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }

    public function destinationWarehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function destinationAddress()
    {
        return $this->belongsTo(Address::class);
    }

    public function salesOrder()
    {
        return $this->belongsTo(SalesOrder::class);
    }

    public function fifoLayers()
    {
        return $this->morphMany(FifoLayer::class, 'link');
    }

    public function adjustments()
    {
        return $this->morphMany(InventoryAdjustment::class, 'link');
    }

    public function purchaseInvoices()
    {
        $relation = $this->hasMany(PurchaseInvoice::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (PurchaseInvoice $purchaseInvoice) {
                $purchaseInvoice->delete();
            });
        });

        return $relation;
    }

    public function purchaseOrderLines()
    {
        $relation = $this->hasMany(PurchaseOrderLine::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (PurchaseOrderLine $purchaseOrderLine) {
                $purchaseOrderLine->delete();
            });
        });

        return $relation;
    }

    public function orderLines()
    {
        return $this->purchaseOrderLines();
    }

    public function products()
    {
        return $this->belongsToMany(Product::class, 'purchase_order_lines');
    }

    public function requestedShippingMethod()
    {
        return $this->belongsTo(ShippingMethod::class, 'requested_shipping_method_id');
    }

    public function accountingTransaction()
    {
        return $this->morphOne(AccountingTransaction::class, 'link')
            ->where('type', AccountingTransactionTypeEnum::PURCHASE_ORDER);
    }

    public function shippingMethod()
    {
        return $this->requestedShippingMethod();
    }

    public function purchaseOrderShipments()
    {
        $relation = $this->hasMany(PurchaseOrderShipment::class);

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (PurchaseOrderShipment $purchaseOrderShipment) {
                $purchaseOrderShipment->delete();
            });
        });

        return $relation;
    }

    public function purchaseOrderShipmentReceipts(): HasManyThrough
    {
        return $this->hasManyThrough(
            PurchaseOrderShipmentReceipt::class,
            PurchaseOrderShipment::class,
            'purchase_order_id',
            'purchase_order_shipment_id',
            'id',
            'id'
        );
    }

    public function purchaseOrderShipmentLines(): HasManyThrough
    {
        return $this->hasManyThrough(PurchaseOrderShipmentLine::class, PurchaseOrderShipment::class);
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    public function currencyTenantSnapshot(): BelongsTo
    {
        return $this->belongsTo(Currency::class, 'currency_id_tenant_snapshot');
    }

    public function inboundShipmentRelation(): MorphOne
    {
        return $this->morphOne(AmazonFbaInboundShipment::class, 'sku_link');
    }

    public function inboundNewShipmentRelation(): MorphOne
    {
        return $this->morphOne(AmazonNewFbaInboundShipment::class, 'sku_link');
    }

    public function receivingDiscrepancy(): MorphOne
    {
        return $this->morphOne(AccountingTransaction::class, 'link')
            ->where('type', AccountingTransactionTypeEnum::RECEIVING_DISCREPANCY);
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function setCurrencyCodeAttribute($value)
    {
        $this->currency_id = empty($value) ? null : Currency::with([])->where('code', $value)->value('id');
    }

    public function setPaymentTermNameAttribute($value)
    {
        $this->payment_term_id = empty($value) ? null : PaymentTerm::with([])->where('name', $value)->value('id');
    }

    public function setIncotermCodeAttribute($value)
    {
        $this->incoterm_id = empty($value) ? null : Incoterm::with([])->where('code', $value)->value('id');
    }

    public function setSupplierNameAttribute($value)
    {
        $this->supplier_id = empty($value) ? null : Supplier::with([])->where('name', $value)->value('id');
    }

    public function setDestinationWarehouseNameAttribute($value)
    {
        $this->destination_warehouse_id = empty($value) ? null : Warehouse::with([])->where('name', $value)->value('id');
    }

    public function setShippingMethodIdAttribute($value)
    {
        $this->requested_shipping_method_id = $value;
    }

    public function setShippingMethodNameAttribute($value)
    {
        $this->requested_shipping_method_id = empty($value) ? null : ShippingMethod::with([])->where('name', $value)->value('id');
    }

    public function getDropshippingAttribute(): bool
    {
        return (bool) $this->sales_order_id;
    }

    public function getAdditionalCostAttribute()
    {
        return $this->purchaseOrderLines->whereNull('product_id')->sum('subtotal');
    }

    public function getAdditionalCostInTenantCurrencyAttribute()
    {
        return $this->purchaseOrderLines->whereNull('product_id')->sum('subtotal_in_tenant_currency');
    }

    public function getProductTotalAttribute()
    {
        return $this->purchaseOrderLines->whereNotNull('product_id')->sum('subtotal');
    }

    public function getProductTotalInTenantCurrencyAttribute()
    {
        return $this->purchaseOrderLines->whereNotNull('product_id')->sum('subtotal_in_tenant_currency');
    }

    public function getDiscountAttribute()
    {
        return $this->purchaseOrderLines->sum('discount_value');
    }

    public function getAdditionalCostFormattedAttribute()
    {
        return $this->currency->code.' '.number_format($this->additional_cost, 2);
    }

    public function getProductTotalFormattedAttribute()
    {
        return $this->currency->code.' '.number_format($this->product_total, 2);
    }

    public function getStoreEmailAttribute()
    {
        return "<a href=\"{$this->store->email}\">{$this->store->email}</a>";
    }

    public function getSupplierNotesFormattedAttribute()
    {
        $notes = nl2br($this->supplier_notes);

        return $notes ?: 'No notes for Supplier';
    }

    public function getProductUnitsAttribute()
    {
        return $this->purchaseOrderLines->sum('quantity');
    }

    public function getSupplierAddressAttribute()
    {
        /** @var Address $address */
        $address = $this->supplier->address;
        if ($address) {
            return $address->formatForEmail();
        }
    }

    public function getDestinationAddressAttribute()
    {
        $destinationAddress = $this->destination;
        if ($destinationAddress) {
            return $destinationAddress->formatForEmail();
        }
    }

    public function getDestinationNameAttribute()
    {
        return $this->destination ? $this->destination->name : null;
    }

    /**
     * Detect if whole purchase order lines shipped.
     */
    public function getFullyShippedAttribute(): bool
    {
        $this->load('purchaseOrderLines', 'purchaseOrderLines.purchaseOrderShipmentLines');

        foreach ($this->purchaseOrderLines->whereNotNull('product_id') as $purchaseOrderLine) {
            if (! $purchaseOrderLine->fully_shipped) {
                return false;
            }
        }

        return true;
    }

    /**
     * Detect if whole purchase order lines shipped.
     */
    public function getFullyInvoicedAttribute(): bool
    {
        $this->load('purchaseOrderLines', 'purchaseOrderLines.purchaseInvoiceLines');

        foreach ($this->purchaseOrderLines as $purchaseOrderLine) {
            if (! $purchaseOrderLine->fully_invoiced) {
                return false;
            }
        }

        return true;
    }

    /**
     * Gets the total invoice on the purchase order.
     */
    public function getInvoiceTotalAttribute(): float
    {
        $this->load('purchaseOrderLines', 'purchaseOrderLines.purchaseInvoiceLines');

        return $this->purchaseOrderLines->sum('invoice_total');
    }

    /**
     * Detect if whole purchase order lines shipped.
     */
    public function getFullyReceivedAttribute(): bool
    {
        $this->load([
            'purchaseOrderLines',
            'purchaseOrderLines.purchaseOrderShipmentLines',
            'purchaseOrderLines.purchaseOrderShipmentLines.purchaseOrderShipmentReceiptLines',
        ]);

        foreach ($this->purchaseOrderLines->whereNotNull('product_id') as $purchaseOrderLine) {
            if (! $purchaseOrderLine->fully_received) {
                return false;
            }
        }

        return true;
    }

    /**
     * Detect if whole purchase order lines shipped.
     */
    public function getPartiallyReceivedAttribute(): bool
    {
        $this->load([
            'purchaseOrderLines',
            'purchaseOrderLines.purchaseOrderShipmentLines',
            'purchaseOrderLines.purchaseOrderShipmentLines.purchaseOrderShipmentReceiptLines',
        ]);

        foreach ($this->purchaseOrderLines as $purchaseOrderLine) {
            if ($purchaseOrderLine->received_quantity > 0) {
                return true;
            }
        }

        return false;
    }

    public function getDestinationAttribute()
    {
        $address = new Address();

        if ($this->dropshipping && $customer = $this->salesOrder->customer) {
            $address->id = $customer->id;
            $address->fill($customer->shippingAddress ? $customer->shippingAddress->toArray() : []);
            $address->fill($customer->toArray());
        } else {
            $warehouse = $this->destinationWarehouse;

            if ($warehouse) {
                $address->id = $warehouse->id;
                $address->fill($warehouse->address ? $warehouse->address->toArray() : []);
                $address->name = $address->company ?? $warehouse->name;
            }
        }

        return $address;
    }

    public function getDestinationAddress1Attribute()
    {
        return $this->destination ? $this->destination->address1 : null;
    }

    public function getDestinationAddress2Attribute()
    {
        return $this->destination ? $this->destination->address2 : null;
    }

    public function getDestinationAddress3Attribute()
    {
        return $this->destination ? $this->destination->address3 : null;
    }

    public function getDestinationCityAttribute()
    {
        return $this->destination ? $this->destination->city : null;
    }

    public function getDestinationProvinceCodeAttribute()
    {
        return $this->destination ? $this->destination->province_code : null;
    }

    public function getDestinationZipAttribute()
    {
        return $this->destination ? $this->destination->zip : null;
    }

    public function getDestinationCountryAttribute()
    {
        return $this->destination ? $this->destination->country : null;
    }

    public function isDraft(): bool
    {
        return $this->order_status === self::STATUS_DRAFT;
    }

    public function isOpen(): bool
    {
        return $this->order_status === self::STATUS_OPEN;
    }

    public function finalized(): \Illuminate\Database\Eloquent\Casts\Attribute
    {
        return \Illuminate\Database\Eloquent\Casts\Attribute::get(fn () => $this->submission_status === self::SUBMISSION_STATUS_FINALIZED);
    }

    public function unsubmitted(): \Illuminate\Database\Eloquent\Casts\Attribute
    {
        return \Illuminate\Database\Eloquent\Casts\Attribute::get(fn () => $this->submission_status === self::SUBMISSION_STATUS_UNSUBMITTED);
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function save(array $options = [])
    {
        // set destination_address_id
        if (! $this->isDirty('destination_address_id') &&
         $this->isDirty(['sales_order_id', 'destination_warehouse_id'])) {
            if ($this->dropshipping) {
                $this->destination_address_id = $this->salesOrder->shipping_address_id ?? null;
            } else {
                $this->destination_address_id = $this->destinationWarehouse->address_id ?? null;
            }
        }

        // set default sequence if it was empty
        if (empty($this->sequence)) {
            $this->sequence = self::getNextSequence();
        }

        // set default purchase order number for the new purchase order
        if (! $this->exists && empty($this->purchase_order_number)) {
            $this->purchase_order_number = self::getLocalNumber($this->sequence);
        }

        // set default submission format for the new purchase order
        if (! $this->exists && empty($this->submission_format)) {
            $this->submission_format = Helpers::setting(Setting::KEY_PO_FORMAT, self::SUBMISSION_FORMAT_PDF_ATTACHMENT);
        }

        // set default supplier warehouse id for the new purchase order
        if (! $this->exists && empty($this->supplier_warehouse_id)) {
            $this->supplier_warehouse_id = $this->supplier->default_warehouse_id;
        }

        if (empty($this->store_id)) {
            $this->store_id = $this->supplier->default_store_id ?? Helpers::setting(Setting::KEY_PO_DEFAULT_STORE);
        }

        if ($this->purchaseOrderLines()->count() == 0) {
            $this->total_cost = 0;
            $this->total_quantity = 0;
        }

        if ($this->isDirty(['is_tax_included', 'tax_rate_id'])) {
            $this->updateTotalCost(false);
        }

        $this->cacheCurrencyRate($this->isDirty(['currency_id']));

        $statusChanged = $this->isDirty(['order_status']);
        $purchaseOrderNumberChanged = $this->exists && $this->isDirty(['purchase_order_number']);
        $saved = parent::save($options);

        if ($saved && $statusChanged) {
            // We synchronize backorder queue coverages
            if ($purchaseOrderLineProductIds = $this->purchaseOrderLines->whereNotNull('product_id')->pluck('id')->toArray()) {
                dispatch(new SyncBackorderQueueCoveragesJob($purchaseOrderLineProductIds));
            }
        }

        if($purchaseOrderNumberChanged){
            // Update the references of inventory movements linked to the PO receipt lines.
            $affectedMovements = [];
            $this->purchaseOrderShipmentReceipts->each(function (PurchaseOrderShipmentReceipt $receipt) use (&$affectedMovements) {
                $receipt->purchaseOrderShipmentReceiptLines->each(function (PurchaseOrderShipmentReceiptLine $receiptLine) use (&$affectedMovements) {
                    array_push($affectedMovements, ...$receiptLine->inventoryMovements->pluck('id')->toArray());
                });
            });

            if(count($affectedMovements) > 0){
                InventoryMovement::query()->whereIn('id', $affectedMovements)->update(['reference' => $this->purchase_order_number]);
            }
        }

        return $saved;
    }

    public function updateTotalCost(bool $save = true)
    {
        $this->loadMissing('purchaseOrderLines');
        $this->total_cost = $this->total;
        $save && $this->save();
    }

    /**
     * @throws Throwable
     */
    public function delete()
    {
        $this->load([
            'purchaseOrderLines',
            'purchaseOrderShipments',
            'purchaseInvoices',
            'adjustments',
            'purchaseOrderShipments.purchaseOrderShipmentLines',
            'purchaseOrderShipments.purchaseOrderShipmentReceipts',
            'purchaseOrderShipments.purchaseOrderShipmentReceipts.purchaseOrderShipmentReceiptLines.purchaseOrderShipmentLine.purchaseOrderLine',
            'purchaseOrderShipments.purchaseOrderShipmentReceipts.purchaseOrderShipmentReceiptLines.inventoryMovements',
            'purchaseOrderShipments.purchaseOrderShipmentReceipts.purchaseOrderShipmentReceiptLines.fifoLayers.inventoryMovements',
            'purchaseInvoices.purchaseInvoiceLines',
        ]);

        $this->purchaseOrderShipments->each(function (PurchaseOrderShipment $shipment) {
            $shipment->delete(); // Deletes shipment lines and receipts
        });

        $this->adjustments->each(function (InventoryAdjustment $adjustment) {
            $adjustment->delete(); // deletes adjustment lines
        });

        if ($this->inboundNewShipmentRelation) {
            $this->inboundNewShipmentRelation->sku_link_id = null;
            $this->inboundNewShipmentRelation->sku_link_type = null;
            $this->inboundNewShipmentRelation->save();
        }

        if ($this->inboundShipmentRelation) {
            $this->inboundShipmentRelation->sku_link_id = null;
            $this->inboundShipmentRelation->sku_link_type = null;
            $this->inboundShipmentRelation->save();
        }

        return DB::transaction(function () {
            $this->purchaseInvoices()->each(function (PurchaseInvoice $purchaseInvoice) {
                $purchaseInvoice->delete(); // Deletes invoice lines
            });

            $this->inboundNewShipmentRelation()->delete();
            $this->inboundShipmentRelation()->delete();
            $this->purchaseOrderLines()->delete();

            $deleted = parent::delete();

            if ($this->dropshipping) {
                $this->salesOrder->updateFulfillmentStatus();
            }

            return $deleted;
        });
    }

    /**
     * Determine if the purchase order is used.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        $relations = ['purchaseInvoices', 'purchaseOrderShipments'];

        $this->loadCount($relations);

        $usage = [];

        foreach ($relations as $relatedRelation) {
            $countKeyName = str_replace('-', '_', Str::kebab($relatedRelation)).'_count';
            if ($this->{$countKeyName}) {
                $relatedName = Str::singular(str_replace('-', ' ', Str::kebab($relatedRelation)));

                $usage[$relatedRelation] = trans_choice('messages.currently_used', $this->{$countKeyName}, [
                    'resource' => $relatedName,
                    'model' => 'purchase order('.$this->purchase_order_number.')',
                ]);
            }
        }

        if ($this->sales_order_id) {
            $usage['salesOrder'] = __('messages.purchase_order.dropship_purchase_order', ['id' => $this->purchase_order_number]);
        }

        return count($usage) ? $usage : false;
    }

    public function isDeletable(): bool
    {
        return true;
    }

    public function getReference(): ?string
    {
        return $this->purchase_order_number;
    }

    /**
     * Approve Draft PO and Submit to supplier.
     */
    public function submit(bool $notifySupplier = true, bool $approvingDropship = false)
    {
        customlog('backorderPurchasing', 'Submitting PO ' . $this->purchase_order_number . ' to supplier ' . $this->supplier->name, [], 7);
        if ($this->purchaseOrderLines->isEmpty()) {
            customlog('backorderPurchasing', 'PO ' . $this->purchase_order_number . ' does not have purchase order lines so not submitting', [], 7);
            return [__('messages.purchase_order.empty_lines'), 'PurchaseOrderLines'.Response::CODE_EMPTY, 'purchase_order_lines'];
        }

        if ($notifySupplier && empty($this->supplier->getPurchaseOrderEmail())) {
            customlog('backorderPurchasing', 'PO ' . $this->purchase_order_number . ' does not have an email address for ' . $this->supplier->name . ' so not submitting', [], 7);
            // PO does not have a supplier email, we abort.
            return [__('messages.purchase_order.email_not_exists'), 'SupplierEmail'.Response::CODE_EMPTY, 'purchase_order_email'];
        }

        /**
         * For a draft PO, we first approve before submitting
         * the PO to the supplier.
         */
        if ($this->order_status == self::STATUS_DRAFT) {
            $this->approve();
        }

        if ($approvingDropship && $this->dropshipping && ! $this->supplier->auto_submit_dropship_po) {
            // Dropship POs aren't automatically submitted to the supplier,
            // we abort.
            customlog('backorderPurchasing', 'PO ' . $this->purchase_order_number . ' is a dropship so not submitting', [], 7);
            return false;
        }

        // For manual submission po's, submit action will change from Unsubmitted to Finalized (#SKU-1527)
        $submissionStatus = self::SUBMISSION_STATUS_SUBMITTED;
        if ($this->submission_format == self::SUBMISSION_FORMAT_MANUAL) {
            $submissionStatus = self::SUBMISSION_STATUS_FINALIZED;
        }

        $this->submission_status = in_array($this->submission_status, [self::SUBMISSION_STATUS_UNSUBMITTED, self::SUBMISSION_STATUS_QUEUED]) ? $submissionStatus : $this->submission_status;
        $this->last_submitted_at = now();

        // notify the supplier
        if ($notifySupplier) {
            $this->handleSupplierNotification();
        }
        else {
            customlog('backorderPurchasing', 'PO ' . $this->purchase_order_number . ' is set to not notify supplier so not submitting', [], 7);
        }

        // fire event to log activity
        event(new PurchaseOrderSubmitted($this));

        return $this->save();
    }

    /**
     * Handles the notification of the supplier about the purchase order.
     */
    protected function handleSupplierNotification()
    {
        if ($this->supplier->receivesOrdersIndividually() || $this->supplier->canSubmitOrderNow($this)) {
            // Supplier receives purchase orders individually.
            $this->supplier->notify(new SubmitPurchaseOrderToSupplierNotification($this));
        } else {
            customlog('backorderPurchasing', 'PO ' . $this->purchase_order_number . ' does not receive orders individually so delaying submitting', [], 7);
            $this->submission_status = self::SUBMISSION_STATUS_QUEUED;
            $this->save();
        }
    }

    public function linesToCSV(): ?string
    {
        $records = $this->linesForCSV();
        $headers = array_keys($records[0] ?? []);

        return ExcelHelper::array2csvfile("{$this->purchase_order_number}.csv", $headers, $records);
    }

    public function linesForCSV(): array
    {
        return $this->purchaseOrderLines->map(function (PurchaseOrderLine $orderLine) {
            return [
                'Purchase Order' => $this->purchase_order_number,
                'SKU' => $orderLine->product ? $orderLine->product->sku : null,
                'Supplier SKU' => $orderLine->supplier_product ? $orderLine->supplier_product->supplier_sku : null,
                'Name' => $orderLine->product ? $orderLine->product->name : null,
                'Qty Ordered' => $orderLine->quantity ?? 0,
                'Unit Cost' => $orderLine->amount,
                'Extended Cost' => $orderLine->quantity * $orderLine->amount,
            ];
        })->toArray();
    }

    public function exportToPDF(?string $reportType = null): string
    {
        $this->loadMissing([
            'supplier',
            'destinationWarehouse',
            'purchaseOrderLines.product.supplierProducts',
            'purchaseOrderLines.purchaseOrderShipmentLines',
        ]);

        if ($reportType == 'picklist') {
            $path = (new GeneratePurchaseOrderPicklist($this))->handle();
        } else {
            $path = ( new GeneratePurchaseOrderInvoice($this) )->handle();
        }

        return Storage::disk('reports')->path($path);
    }

    /**
     * Finalized approve PO by buyer and supplier.
     *
     *
     * @return bool|array Array: failed
     */
    public function approve(?Carbon $approveDate = null, $applyBackorderCoverages = true)
    {
        if ($this->purchaseOrderLines->isEmpty()) {
            return [__('messages.purchase_order.empty_lines'), 'PurchaseOrderLines'.Response::CODE_EMPTY, 'purchase_order_lines'];
        }

        //    $this->submission_status = PurchaseOrder::SUBMISSION_STATUS_FINALIZED;
        $this->order_status = self::STATUS_OPEN;
        $this->approved_at = $this->approved_at ?: ($approveDate ?: now());
        $this->finalized_at = $approveDate ?: now();

        event(new PurchaseOrderApproved($this));

        $this->cacheCurrencyRate(true);

        if ($save = $this->save()) {
            $this->updateProductInventory();

            if ($this->dropshipping && $this->supplier->auto_submit_dropship_po) {
                $this->submit(true, true);
            }

            // Sync linked backorders
            $this->syncLinkedBackorders();
        }

        return $save;
    }

    public function syncLinkedBackorders(): void
    {
        $withLinkedBackorders = $this->purchaseOrderLines()->whereNotNull('linked_backorders');
        if ($withLinkedBackorders->count() == 0) {
            return;
        }

        $linked = [];
        /** @var PurchaseOrderLine $purchaseOrderLine */
        foreach ($withLinkedBackorders->get() as $purchaseOrderLine) {
            $coverages = collect($purchaseOrderLine->linked_backorders)
                ->map(function ($backorder) use ($purchaseOrderLine) {
                    return [
                        'purchase_order_line_id' => $purchaseOrderLine->id,
                        'backorder_queue_id' => $backorder['id'],
                        'covered_quantity' => min($backorder['quantity'], $purchaseOrderLine->unreceived_quantity),
                        'is_tight' => true,
                    ];
                })->toArray();

            array_push($linked, ...$coverages);
        }

        BackorderQueueCoverage::query()->upsert(
            $linked,
            ['purchase_order_line_id', 'backorder_queue_id']
        );
    }

    private function updateProductInventory(bool $loadLinesFromDB = true)
    {
        if ($loadLinesFromDB) {
            $lines = $this->purchaseOrderLines();
        } else {
            $lines = $this->purchaseOrderLines;
        }

        $productIds = $lines->whereNotNull('product_id')
            ->pluck('product_id')
            ->toArray();

        if (empty($productIds)) {
            return;
        }

        (new ProductInventoryManager($productIds))->updateProductInventoryAndAvgCost();
    }

    /**
     * Revert to draft and open purchase order.
     */
    public function revertToDraft(): bool
    {
        if ($this->receipt_status !== self::RECEIPT_STATUS_UNRECEIVED) {
            return ['Purchase orders with receipts cannot be reverted to draft.', 'PurchaseOrder'.Response::CODE_CLOSED, 'id'];
        }

        if ($this->order_status === self::STATUS_CLOSED) {
            return ['Closed purchase orders cannot be reverted to draft.', 'PurchaseOrder'.Response::CODE_CLOSED, 'id'];
        }

        $this->order_status = self::STATUS_DRAFT;
        $this->approved_at = null;

        // Delete backorder queue coverages
        $this->purchaseOrderLines->each(function (PurchaseOrderLine $orderLine) {
            $orderLine->coveredBackorderQueues()->delete();
        });

        if ($save = $this->save()) {
            $this->updateProductInventory();
        }

        return $save;
    }

    public function getReceiptBroadcastChannel(): string
    {
        return "po-receipt-{$this->purchase_order_number}";
    }

    /**
     * Mark as Shipped Purchase order.
     */
    public function shipped(?Carbon $shippedDate = null, bool $dropship = false)
    {
        $this->shipment_status = $dropship ? self::SHIPMENT_STATUS_SHIPPED_CUSTOMER : self::SHIPMENT_STATUS_SHIPPED_WAREHOUSE;

        if ($this->fully_shipped) {
            $this->fully_shipped_at = $shippedDate ?: now();
            // Submission Status should be hooked to change to finalized when shipped, received or invoiced (#SKU-1527)
            $this->submission_status = self::SUBMISSION_STATUS_FINALIZED;
        }

        $this->save();
    }

    public function refreshReceiptStatus(?Carbon $receivedDate = null)
    {
        if ($this->fully_received) {
            $this->fully_received_at = $receivedDate ?: now();
            $this->receipt_status = self::RECEIPT_STATUS_RECEIVED;
            $this->order_status = self::STATUS_CLOSED;
            // Submission Status should be hooked to change to finalized when shipped, received or invoiced (#SKU-1527)
            $this->submission_status = self::SUBMISSION_STATUS_FINALIZED;
        } elseif ($this->partially_received) {
            $this->receipt_status = self::RECEIPT_STATUS_PARTIALLY_RECEIVED;
            $this->order_status = self::STATUS_OPEN;
            $this->fully_received_at = null;
        } else {
            $this->receipt_status = $this->dropshipping ? self::RECEIPT_STATUS_DROPSHIP : self::RECEIPT_STATUS_UNRECEIVED;
            $this->order_status = $this->order_status == self::STATUS_CLOSED ? self::STATUS_OPEN : $this->order_status;
            if ($this->dropshipping) {
                $this->shipment_status = self::SHIPMENT_STATUS_UNSHIPPED;
            }
            $this->fully_received_at = null;
        }
        activity()
            ->performedOn($this)
            ->event('updated')
            ->withProperties([
                'attributes' => [
                    'receipt_status' => $this->receipt_status,
                ],
                'old' => ['receipt_status' => $this->getOriginal('receipt_status')],
                'metadata' => $this->getMetadataForActivityLog(),
            ])
            ->log('updated');

        $this->save();
    }

    public function invoiced(?Carbon $invoicedDate = null)
    {
        if ($this->fully_invoiced) {
            $this->fully_invoiced_at = $invoicedDate ?: now();
            $this->invoice_status = self::INVOICE_STATUS_INVOICED;
            // Submission Status should be hooked to change to finalized when shipped, received or invoiced (#SKU-1527)
            $this->submission_status = self::SUBMISSION_STATUS_FINALIZED;

            // dropship
            if ($this->dropshipping) {
                $this->order_status = self::STATUS_CLOSED;
            }
        } elseif ($this->invoice_total > 0) {
            $this->invoice_status = self::INVOICE_STATUS_PARTIALLY_INVOICED;
        } else {
            $this->invoice_status = self::INVOICE_STATUS_UNINVOICED;
        }

        $this->save();
    }

    public function productLines(): HasMany
    {
        return $this->purchaseOrderLines()->whereNotNull('product_id');
    }

    /**
     * Set PO lines.
     *
     *
     * @return bool|array|null array of purchase order lines that sent with new line id
     */
    public function setPurchaseOrderLines($purchaseOrderLines, bool|array $sync = true): bool|array|null
    {
        if ($purchaseOrderLines === false) {
            return null;
        }
        $removingPurchaseOrderLines = [];
        $existingPurchaseOrderLines = [];

        foreach ($purchaseOrderLines ?: [] as $index => $orderLine) {
            $purchaseOrderLine = PurchaseOrderLine::findByStoreRequest($this->id, $orderLine);

            if (! isset($purchaseOrderLine)) {
                $purchaseOrderLine = new PurchaseOrderLine();
            }

            if (isset($orderLine['is_deleted']) && $orderLine['is_deleted']) {
                $removingPurchaseOrderLines[] = $purchaseOrderLine->id;

                continue;
            }
            if (isset($orderLine['discount'])) {
                $orderLine['discount_rate'] = $orderLine['discount'] / 100;
            }
            if (empty($orderLine['description']) && isset($orderLine['product_id'])) {
                $orderLine['description'] = Product::with([])->findOrFail($orderLine['product_id'])->name;
            }

            /** @var PurchaseOrderLine $purchaseOrderLine */
            $purchaseOrderLine->fill(Arr::except($orderLine, ['purchase_order_line_id', 'tax']));

            // Kit products cannot be purchased.
            if($purchaseOrderLine->product && $purchaseOrderLine->product->type === Product::TYPE_KIT){
                continue;
            }

            if (! empty($orderLine['tax_rate_id'])) {
                $taxRate = TaxRate::with([])->findOrFail($orderLine['tax_rate_id']);
                $purchaseOrderLine->tax_rate = $taxRate->rate;
                $purchaseOrderLine->tax_allocation = $this->getTaxAmount($this->is_tax_included, $purchaseOrderLine->subtotal, $purchaseOrderLine->tax_rate);
            } elseif (isset($orderLine['tax_rate_id'])) {
                $purchaseOrderLine->tax_rate = null;
                $purchaseOrderLine->tax_allocation = 0;
            }

            $this->purchaseOrderLines()->save($purchaseOrderLine); // Sync backorder queue coverage for the purchase order line

            if (empty($purchaseOrderLine->description)) {
                $purchaseOrderLine->description = Product::with([])->findOrFail($purchaseOrderLine->product_id)->name;
                $this->purchaseOrderLines()->save($purchaseOrderLine);
            }

            // add to existing purchase order lines to remove other lines
            $existingPurchaseOrderLines[] = $purchaseOrderLine->id;
            // add purchase order lines to lines that sent
            $purchaseOrderLines[$index]['purchase_order_line_id'] = $purchaseOrderLine->id;
            // unset $purchaseOrderLine to prevent using in next loop
            unset($purchaseOrderLine);
        }

        // sync purchase order lines
        if (! empty($sync)) {
            if (request()->isMethod('patch')) {
                $removedLines = $this->purchaseOrderLines()->when(is_array($sync), fn ($q) => $q->where($sync))->whereIn('id', $removingPurchaseOrderLines);
            } else {
                $removedLines = $this->purchaseOrderLines()->when(is_array($sync), fn ($q) => $q->where($sync))->whereNotIn('id', $existingPurchaseOrderLines);
            }
            $product_ids_needing_cache_update = $removedLines->pluck('product_id')->toArray();
            $removedLines->delete();
            // Sync product inventory for removed lines.
            (new ProductInventoryManager(
                $product_ids_needing_cache_update
            ))->setUpdateAverageCost(false)->updateProductInventoryAndAvgCost();
        }

        // We update the total cost after setting the purchase order lines
        $this->refresh();
        $this->updateTotalCost();
        $this->refreshReceiptStatus(); // update receipt statuses
        $this->handleNonProductLineCosts();
        $this->updateProductInventory();

        return $purchaseOrderLines;
    }

    public function handleNonProductLineCosts(): void
    {
        if ($this->purchaseOrderLines()->whereHas('purchaseOrderShipmentLines.purchaseOrderShipmentReceiptLines')->count() == 0)
        {
            return;
        }

        $receiptFifoLayerIds = $this->purchaseOrderLines
            ->pluck('purchaseOrderShipmentLines')
            ->flatten()
            ->pluck('purchaseOrderShipmentReceiptLines')
            ->flatten()
            ->pluck('fifoLayers')
            ->flatten()
            ->pluck('id')
            ->toArray();

        if (!empty($receiptFifoLayerIds))
        {
            app(FifoLayerRepository::class)->recalculateTotalCosts($receiptFifoLayerIds);
        }
    }

    public function otherLines(): Collection
    {
        return $this->purchaseOrderLines->whereNull('product_id');
    }

    /**
     * Get next sequence for purchase order number.
     */
    public static function getNextSequence(): int
    {
        $lastSequence = static::query()->latest('sequence')->value('sequence');

        if ($lastSequence) {
            $nextSequence = $lastSequence + 1;
        } else {
            $startFrom = Helpers::setting(Setting::KEY_PO_START_NUMBER, 1);
            $nextSequence = $startFrom;
        }

        while (static::query()->where('purchase_order_number', static::getLocalNumber($nextSequence))->exists()) {
            $nextSequence++;
        }

        return $nextSequence;
    }

    /**
     * Get next purchase order number.
     */
    public static function getNextLocalNumber(): string
    {
        return static::getLocalNumber(self::getNextSequence());
    }

    /**
     * Get the purchase order number by the sequence
     */
    public static function getLocalNumber(int $sequence): string
    {
        $prefix = Helpers::setting(Setting::KEY_PO_PREFIX, '');
        $numOfDigits = Helpers::setting(Setting::KEY_PO_NUM_DIGITS, 5);

        return static::makePurchaseOrderNumber($prefix, $sequence, $numOfDigits);
    }

    public static function makePurchaseOrderNumber(string $prefix, int $sequence, int $numOfDigits): string
    {
        return sprintf("$prefix%0{$numOfDigits}d", $sequence);
    }

    /**
     * Ships all unshipped lines on the purchase order.
     */
    public function shipAllLines(array $data): PurchaseOrderShipment
    {
        /** @var PurchaseOrderShipment $shipment */
        $shipment = null;

        // We fully ship partially shipped lines
        $this->purchaseOrderShipments()->with(['purchaseOrderShipmentLines'])->each(function (PurchaseOrderShipment $purchaseOrderShipment) use (&$shipment, $data) {
            // Get lines and complete the shipment for each line
            $purchaseOrderShipment->purchaseOrderShipmentLines()->with(['purchaseOrderLine'])->each(function (PurchaseOrderShipmentLine $shipmentLine) use (&$shipment, $data) {
                if ($shipmentLine->quantity !== $shipmentLine->purchaseOrderLine->quantity) {
                    if ($shipment === null) {
                        // Create shipment for new lines if not created already
                        $shipment = $this->purchaseOrderShipments()->create($data);
                    }

                    // The purchase order line was not fully shipped, we ship the difference in this shipment
                    $difference = $shipmentLine->purchaseOrderLine->quantity - $shipmentLine->quantity;
                    $shipment->purchaseOrderShipmentLines()->create([
                        'purchase_order_line_id' => $shipmentLine->purchaseOrderLine->id,
                        'quantity' => $difference,
                    ]);
                }
            });
        });

        // If there're more lines to ship, we create a shipment and ship those lines.
        $unshippedLines = $this->purchaseOrderLines()
            ->whereNotIn('id', $this->purchaseOrderShipmentLines()->pluck('purchase_order_line_id')->toArray())
            ->get();

        if ($unshippedLines->count() > 0) {
            if ($shipment === null) {
                $shipment = $this->purchaseOrderShipments()->create($data);
            }

            // Add unshipped lines to shipment
            $shipment->purchaseOrderShipmentLines()->createMany($unshippedLines->map(function (PurchaseOrderLine $purchaseOrderLine) {
                return ['purchase_order_line_id' => $purchaseOrderLine->id, 'quantity' => $purchaseOrderLine->quantity];
            })->toArray());
        }

        return $shipment;
    }

    public function linkInboundShipmentRelation(string $shipmentId)
    {
        if (! $this->destinationWarehouse || ! $this->destinationWarehouse->integrationInstance) {
            return;
        }

        InboundShipmentRelation::with([])->create(
            [
                'integration_instance_id' => $this->destinationWarehouse->integrationInstance->id,
                'shipment_id' => $shipmentId,
                'link_type' => self::class,
                'link_id' => $this->id,
            ]
        );

        $this->purchase_order_number = $shipmentId;
        $job = new GetInboundShipments($this->destinationWarehouse->integrationInstance);
        $job->setIds([$shipmentId]);
        dispatch_sync($job);
    }

    public function scopeAccountingReady(Builder $builder): Builder
    {
        return $builder->where('order_status', '!=', PurchaseOrder::STATUS_DRAFT);
    }

    public function scopeFilterPurchaseOrderLines(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        if (count($keyExploded = explode('.', $relation['combined_key'])) > 2) {
            $relation = implode('.', array_slice($keyExploded, 0, 2));
            $lastKey = array_slice($keyExploded, -1)[0];

            return $builder->{$function.'Has'}($relation, function (Builder $builder) use ($lastKey, $value, $operator) {
                $builder->filterKey([
                    'key' => $builder->qualifyColumn($lastKey),
                    'is_relation' => false,
                ], $operator, $value);
            });
        } elseif ($relation['key'] === 'subtotal') {
            // Product subtotal
            return $builder->{$function.'Has'}('purchaseOrderLines', function (Builder $builder) use ($operator, $value) {
                if ($operator === 'isEmpty') {
                    $operator = '=';
                    $value = 0;
                } elseif ($operator === 'isNotEmpty') {
                    $operator = '>';
                    $value = 0;
                }

                return $builder->whereRaw(" purchase_order_lines.quantity * purchase_order_lines.amount {$operator} ?", [$value]);
            });
        } else {
            // default behavior
            return false;
        }
    }

    public function scopeFilterSupplier(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        return $this->scopeFilterPurchaseOrderLines($builder, $relation, $operator, $value, $conjunction);
    }

    public function scopeFilterPurchaseOrderShipments(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        // filter by shipment_item_quantity
        if ($relation['key'] == 'lines_quantity') {
            if ($operator == 'isEmpty' || $operator == 'isNotEmpty') {
                if ($conjunction == 'and') {
                    $function = $operator == 'isEmpty' ? 'whereDoesntHave' : 'whereHas';
                } else {
                    $function = $operator == 'isEmpty' ? 'orWhereDoesntHave' : 'orWhereHas';
                }

                return $builder->{$function}('purchaseOrderShipments');
            }

            $function = $conjunction == 'and' ? 'whereHas' : 'orWhereHas';

            return $builder->{$function}('purchaseOrderShipments', function (Builder $builder) use ($value, $operator) {
                $key = function (\Illuminate\Database\Query\Builder $q) {
                    return $q->from('purchase_order_shipment_lines')
                        ->select(DB::raw('SUM(quantity)'))
                        ->whereColumn('purchase_order_shipment_id', 'purchase_order_shipments.id');
                };

                if ($operator == 'isAnyOf') {
                    $values = collect(Arr::wrap($value))->map(fn ($v) => "'$v'")->implode(',');

                    return $builder->whereRaw('('.$key($builder->clone()->newQuery()->getQuery())->toSql().") IN ($values)");
                }

                return $builder->filterKey($key, $operator, $value);
            });
        }

        return $this->scopeFilterPurchaseOrderLines($builder, $relation, $operator, $value, $conjunction);
    }

    public function scopeWithCostValues(Builder $builder): Builder
    {
        return $builder->with(['purchaseOrderLines' => fn ($q) => $q->withCostValues()]);
    }

    public function scopeFilterDestinationAddress(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        // filter on destination warehouse OR customer (if dropshipping)
        $builder->where(function (Builder $builder) use ($value, $operator, $relation) {
            // filter on destination warehouse
            if ($relation['key'] == 'id') {
                $builder->filterKey('destination_warehouse_id', $operator, $value);
            } elseif ($relation['key'] == 'name') {
                $builder->whereHas('destinationWarehouse', function (Builder $builder) use ($value, $operator) {
                    $builder->filterKey('name', $operator, $value);
                });
            } else {
                $builder->whereHas('destinationWarehouse.address', function (Builder $builder) use ($relation, $value, $operator) {
                    $builder->filterKey($relation['key'], $operator, $value);
                });
            }

            // filter on customer in dropship
            if ($relation['key'] == 'id') {
                $builder->orWhereHas('salesOrder', function (Builder $builder) use ($value, $operator) {
                    $builder->filterKey('customer_id', $operator, $value);
                });
            } elseif (in_array($relation['key'], ['name', 'email', 'zip', 'address1', 'company', 'phone', 'fax'])) {
                $builder->orWhereHas('salesOrder.customer', function (Builder $builder) use ($relation, $value, $operator) {
                    $builder->filterKey($relation['key'], $operator, $value);
                });
            } else {
                $builder->orWhereHas('salesOrder.customer.shippingAddress', function (Builder $builder) use ($relation, $value, $operator) {
                    $builder->filterKey($relation['key'], $operator, $value);
                });
            }
        }, null, null, $conjunction);
    }

    public function scopeFilterAsnSent(Builder $builder, $operator, $value, $conjunction)
    {
        $function = $conjunction == 'and' ? 'where' : 'orWhere';

        $builder->{$function}(function(Builder $builder) use ($value, $operator) {
            $function = $value ? 'whereNotNull' : 'whereNull';
            $builder->{$function}('asn_last_sent_at');
            if(!$value) {
                $builder->where('order_status', '!=', PurchaseOrder::STATUS_DRAFT)
                    ->whereHas('destinationWarehouse', function (Builder $builder) {
                        $builder->where('type', Warehouse::TYPE_3PL);
                    });
            }
        });
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.purchase_order.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        if (Str::startsWith(func_get_arg(0), ['supplier_name.name', 'destination_name.', 'tags.'])) {
            return func_get_args();
        }

        return array_merge(collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all(), ['supplier_id']);
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['sku', 'purchase_order_number', 'order_status', 'submission_format', 'purchaseInvoices.supplier_invoice_number', 'supplier.name'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function extractToPdf(?string $reportType = null): array
    {
        if ($reportType == 'picklist') {
            return ['items' => PurchaseOrderPicklistTransformer::transform($this)];
        }

        $this->loadMissing('store', 'store.address');

        if ($this->supplier && $this->supplier->address) {
            $supplierAddress = $this->supplier->address->formatForInvoice();
        } else {
            $supplierAddress = '';
        }
        if ($this->store && $this->store->address) {
            $storeAddress = $this->store->address->formatForInvoice(false);
        } else {
            $storeAddress = '';
        }

        $lines = $this->purchaseOrderLines->map(function ($line) {
            return [
                'sku' => $line->product ? ($line->supplier_product->supplier_sku ?? $line->product->sku) : '',
                'description' => $line->description,
                'qty' => $line->quantity,
                'amount' => number_format($line->amount, 2),
                'subtotal' => number_format($line->subtotal, 2),
                'tax' => $line->tax_allocation,
                'discount' => 0,
                'discount_rate' => intval($line->discount_rate * 100),
            ];
        });
        if (! empty($this->supplier_notes)) {
            $lines->push(
                [
                    'sku' => 'Notes',
                    'description' => $this->supplier_notes ?? '',
                    'qty' => 0,
                    'amount' => '',
                    'subtotal' => '',
                    'tax' => 0,
                    'discount' => 0,
                    'discount_rate' => 0,
                ]
            );
        }

        $destinationAddress = '';
        $customer = null;
        if($this->dropshipping && $this->salesOrder) {
            $customer = $this->salesOrder->customer->name;
            $destinationAddress .= $this->salesOrder->shippingAddress->formatForInvoice();
        }
        else if($this->destination) {
            $destinationAddress .= $this->destination->formatForInvoice();
        }

        return [
            'order_number' => $this->purchase_order_number,
            'order_date' => Carbon::parse($this->purchase_order_date)->toFormattedDateString(),
            'store_address' => $storeAddress,
            'store_contact' => $this->store->address ? $this->store->address->email : null,
            'store_name' => $this->store->name,
            'vendor_address' => $supplierAddress,
            'vendor_name' => $this->supplier ? $this->supplier->name : '',
            'shipping_name' => $customer ?? $this->store->name,
            'shipping_address' => $destinationAddress,
            'shipping_method' => $this->shippingMethod ? $this->shippingMethod->name : ($this->requested_shipping_method ?: ''),
            'ship_by' => $this->shippingMethod && $this->shippingMethod->shippingCarrier ? $this->shippingMethod->shippingCarrier->name : '',
            'order_updated_at' => $this->updated_at->toFormattedDateString(),
            'total_units' => $this->purchaseOrderLines->whereNotNull('product_id')->sum('quantity'),
            'total' => number_format($this->total, 2),
            'total_discount' => number_format($this->discount, 2),
            'items_subtotal' => number_format($this->product_total, 2),
            'logo' => $this->store->logo_url ? url($this->store->logo_url) : null,
            'total_tax' => number_format($this->calculated_tax_total, 2), //TODO: test to make sure this works
            'comment' => $this->supplier_notes ?? '',
            'items' => $lines->toArray(),
        ];
    }

    public static function getExportableFields(): array
    {
        return PurchaseOrderDataImporter::getExportableFields();
    }

    public static function transformExportData(array $data): array
    {
        $grouped = BaseExporter::groupByLines($data, 'po_number', 'items');
        $data = empty($grouped) ? $data : $grouped;

        // Bind in fulfillment info
        $keys = array_keys($data[0]);

        if (in_array('fulfillment_info', $keys)) {
            foreach ($data as $key => $record) {
                $fulfillment = collect($record['fulfillment_info']);
                $shipped = $fulfillment->sum('shipment_item_quantity');
                $received = $fulfillment->sum('shipment_item_received');
                $data[$key]['fulfillment_info'] = [
                    'shipped' => $shipped,
                    'received' => $received,
                    'enroute' => max($shipped - $received, 0),
                ];
            }
        }

        return $data;
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new PurchaseOrderDataImporter(null, $filePath);
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->id;
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [
            'id' => $this->id,
            'purchase_order_number' => $this->purchase_order_number,
        ];
    }

    public function getTaxSummary(): array
    {
        $taxSummary = [];
        $this->purchaseOrderLines->load('taxRate')->each(function (PurchaseOrderLine $purchaseOrderLine) use (&$taxSummary) {
            if ($purchaseOrderLine->taxRate) {
                if (! isset($taxSummary[$purchaseOrderLine->taxRate->id])) {
                    $taxSummary[$purchaseOrderLine->taxRate->id] = [
                        'id' => $purchaseOrderLine->taxRate->id,
                        'name' => $purchaseOrderLine->taxRate->name,
                        'rate' => $purchaseOrderLine->taxRate->rate,
                        'amount' => 0,
                    ];
                }
                $taxSummary[$purchaseOrderLine->taxRate->id]['amount'] += $purchaseOrderLine->tax_allocation;
            }
        });

        return $taxSummary;
    }

    public function getTaxSummaryInTenantCurrency(): array
    {
        $taxSummary = [];
        $this->purchaseOrderLines->load('taxRate')->each(function (PurchaseOrderLine $purchaseOrderLine) use (&$taxSummary) {
            if ($purchaseOrderLine->taxRate) {
                if (! isset($taxSummary[$purchaseOrderLine->taxRate->id])) {
                    $taxSummary[$purchaseOrderLine->taxRate->id] = [
                        'id' => $purchaseOrderLine->taxRate->id,
                        'name' => $purchaseOrderLine->taxRate->name,
                        'rate' => $purchaseOrderLine->taxRate->rate,
                        'amount_in_tenant_currency' => 0,
                    ];
                }
                $taxSummary[$purchaseOrderLine->taxRate->id]['amount_in_tenant_currency'] += $purchaseOrderLine->tax_allocation_in_tenant_currency;
            }
        });

        return $taxSummary;
    }

    public function getReferenceNumber(): string
    {
        return $this->purchase_order_number;
    }

    public function getDestinationAddress(): Address
    {
        return $this->destinationWarehouse->address;
    }

    public function getFromEmail(): string
    {
        return $this->store->email;
    }
}
