<?php

namespace App\Models;

use App\Models\Concerns\HasAccountingTransactionLine;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * Class PurchaseInvoiceLine.
 *
 *
 * @property int $id
 * @property int $purchase_invoice_id
 * @property int $purchase_order_line_id
 * @property float $quantity_invoiced
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read float $sub_total
 * @property-read float $tax_value
 * @property-read float $discount_value
 * @property-read float $prorationOfPurchaseOrderLine
 * @property PurchaseInvoice $purchaseInvoice
 * @property PurchaseOrderLine $purchaseOrderLine
 */
class PurchaseInvoiceLine extends Model
{
    use HasAccountingTransactionLine, HasFactory;

    protected $casts = [
        'quantity_invoiced' => 'float',
    ];

    protected $fillable = [
        'purchase_invoice_id',
        'purchase_order_line_id',
        'quantity_invoiced',
        'updated_at',
    ];

    protected $touches = ['purchaseInvoice'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function purchaseInvoice()
    {
        return $this->belongsTo(PurchaseInvoice::class);
    }

    public function purchaseOrderLine()
    {
        return $this->belongsTo(PurchaseOrderLine::class);
    }

    public function accountingTransactionLine()
    {
        return $this->morphOne(AccountingTransactionLine::class, 'link');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getSubtotalAttribute(): float
    {
        return ($this->quantity_invoiced * $this->purchaseOrderLine->amount) - $this->purchaseOrderLine->discount_value;
    }

    public function prorationOfPurchaseOrderLine(): Attribute
    {
        return Attribute::get(
            fn () => $this->purchaseOrderLine->quantity == 0 ?
                0.00 : $this->quantity_invoiced / $this->purchaseOrderLine->quantity
        );
    }

    public function getDiscountValueAttribute()
    {
        return $this->quantity_invoiced * $this->purchaseOrderLine->amount * $this->purchaseOrderLine->discount / 100;
    }

    public function getTaxValueAttribute()
    {
        return $this->quantity_invoiced * $this->purchaseOrderLine->amount / 100;
    }
}
