<?php

namespace App\Models;

use App\Abstractions\FinancialDocumentInterface;
use App\Data\AccountingTransactionData;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasAccountingTransaction;
use App\Models\Concerns\HasFilters;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionDataFromPurchaseInvoice;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Spatie\Activitylog\LogOptions;
use Throwable;

/**
 * Class PurchaseInvoice.
 *
 *
 * @property int $id
 * @property int $purchase_order_id
 * @property Carbon $purchase_invoice_date
 * @property string $supplier_invoice_number
 * @property int $supplier_id
 * @property string $status
 * @property-read float $calculated_total
 * @property-read float $tax_total
 * @property-read float $total_paid

 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read PurchaseOrder $purchaseOrder
 * @property-read Supplier $supplier
 * @property-read Collection $purchaseInvoiceLines
 * @property-read AccountingTransaction $accountingTransaction
 */
class PurchaseInvoice extends Model implements FinancialDocumentInterface
{
    use Concerns\LogsActivity, HandleDateTimeAttributes, HasAccountingTransaction, HasFactory, HasFilters;

    const STATUS_UNPAID = 'unpaid';

    const STATUS_PAID = 'paid';

    const STATUS_PARTIALLY_PAID = 'partially_paid';

    const STATUS = [
        self::STATUS_UNPAID,
        self::STATUS_PAID,
        self::STATUS_PARTIALLY_PAID,
    ];

    protected $casts = [
        'purchase_invoice_date' => 'datetime',
    ];

    protected $fillable = [
        'purchase_invoice_date',
        'supplier_invoice_number',
        'status',
        'updated_at',
    ];

    protected $attributes = ['status' => self::STATUS_UNPAID];

    public function getAccountingDateFieldName(): string
    {
        return 'purchase_invoice_date';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function purchaseOrder()
    {
        return $this->belongsTo(PurchaseOrder::class);
    }

    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }

    public function purchaseInvoiceLines()
    {
        return $this->hasMany(PurchaseInvoiceLine::class);
    }

    public function payments()
    {
        return $this->morphMany(Payment::class, 'link');
    }

    public function fifoLayers()
    {
        return $this->morphMany(FifoLayer::class, 'link');
    }

    public function accountingTransaction()
    {
        return $this->morphOne(AccountingTransaction::class, 'link');
    }

    /**
     * @throws Throwable
     */
    public function getAccountingTransactionData(): AccountingTransactionData
    {
        return (new BuildAccountingTransactionDataFromPurchaseInvoice($this))->handle();
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return $this->purchaseOrder->accountingTransaction;
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function save(array $options = [])
    {
        // Set supplier_id if empty.
        if (empty($this->attributes['supplier_id'])) {
            $this->attributes['supplier_id'] = $this->purchaseOrder->supplier_id;
        }

        return parent::save($options);
    }

    public function delete()
    {
        $this->load(['purchaseInvoiceLines']);

        $this->purchaseInvoiceLines()->delete();
        // Set the invoice status of the purchase order
        $this->purchaseOrder->invoiced(now());

        return parent::delete();
    }

    /**
     * Syncs purchase invoice lines.
     */
    public function syncInvoiceLines(array $invoiceLines)
    {
        $existingLineIds = $this->purchaseInvoiceLines()->pluck('purchase_order_line_id')->toArray();
        $newLineIds = array_map(function ($invoice) {
            return $invoice['purchase_order_line_id'];
        }, $invoiceLines);

        $keptInvoiceIds = array_intersect($existingLineIds, $newLineIds);
        $additionalLineIds = array_diff($newLineIds, $keptInvoiceIds);
        $removedLineIds = array_diff($existingLineIds, $keptInvoiceIds);

        // Delete removed
        $this->purchaseInvoiceLines()->whereIn('purchase_order_line_id', $removedLineIds)->delete();

        // Add new line ids
        $additionalLines = collect($invoiceLines)->whereIn('purchase_order_line_id', $additionalLineIds)->toArray();
        $this->purchaseInvoiceLines()->createMany($additionalLines);

        // Update quantities for kept invoice ids
        $this->purchaseInvoiceLines()->whereIn('purchase_order_line_id', $keptInvoiceIds)
            ->each(function (PurchaseInvoiceLine $invoiceLine) use ($invoiceLines) {
                $matched = array_values(array_filter($invoiceLines, function ($line) use ($invoiceLine) {
                    return $line['purchase_order_line_id'] === $invoiceLine->purchase_order_line_id;
                }));
                if (count($matched) > 0) {
                    $invoiceLine->fill($matched[0]);
                    $invoiceLine->save();
                }
            });
    }

    /**
     * Aggregates purchase order invoice lines by either purchase order line ids
     * or line references on purchase order lines.
     */
    public static function aggregateInvoiceLines(array $invoiceLines): array
    {
        // Bind in purchase order line ids for lines with line references
        $invoiceLines = array_map(function ($line) {
            if (isset($line['purchase_order_line_reference']) && ! isset($line['purchase_order_line_id'])) {
                $line['purchase_order_line_id'] = PurchaseOrderLine::with([])->where('line_reference', $line['purchase_order_line_reference'])->firstOrFail(['id'])->id;
            }

            return $line;
        }, $invoiceLines);

        $uniqueLineIds = array_unique(array_map(function ($line) {
            if (isset($line['purchase_order_line_id'])) {
                return $line['purchase_order_line_id'];
            }

            return $line['description'];
        }, $invoiceLines));
        $uniqueLines = [];

        foreach ($uniqueLineIds as $lineId) {
            if (! is_numeric($lineId)) {
                // Add in lines without line ids, match by description
                $matched = array_values(array_filter($invoiceLines, function ($line) use ($lineId) {
                    return $line['description'] === $lineId;
                }));
            } else {
                $matched = array_values(array_filter($invoiceLines, function ($line) use ($lineId) {
                    return isset($line['purchase_order_line_id']) && $line['purchase_order_line_id'] === $lineId;
                }));
            }

            $uniqueLines[] = array_merge($matched[0], [
                'purchase_order_line_id' => $lineId,
                'quantity_invoiced' => array_sum(array_values(array_map(function ($line) {
                    return $line['quantity_invoiced'];
                }, $matched))),
            ]);
        }

        return $uniqueLines;
    }

    public function getCalculatedTotalAttribute()
    {
        $this->relationLoaded('purchaseInvoiceLines');

        $total = $this->purchaseInvoiceLines->sum('subtotal');

        return round($total, 2);
    }

    public function getTaxTotalAttribute()
    {
        $this->relationLoaded('purchaseInvoiceLines');

        $tax_total = $this->purchaseInvoiceLines->sum('tax_allocation');

        return round($tax_total, 2);

        return $totalDiscount;
    }

    /**
     * Gets the total amount paid for the sales order.
     */
    public function getTotalPaidAttribute(): float
    {
        $this->loadMissing('payments');

        return $this->payments()->sum('amount');
    }

    public function setPaymentStatus()
    {
        if ($this->total_paid - $this->calculated_total + $this->tax_total >= 0) {
            $this->status = static::STATUS_PAID;
        } elseif ($this->total_paid == 0) {
            $this->status = static::STATUS_UNPAID;
        } else {
            $this->status = static::STATUS_PARTIALLY_PAID;
        }

        $this->save();
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->purchase_order_id;
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    public function getMetadataForActivityLog(): ?array
    {
        return null;
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeAccountingReady(Builder $builder): Builder
    {
        return $builder;
    }
}
