<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Helpers;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use App\Observers\AddPackingSlipQueueObserver;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

/**
 * Class ProductImage.
 *
 *
 * @property int $id
 * @property int $product_id
 * @property string $url
 * @property string|null $name
 * @property int $sort_order
 * @property bool $is_primary
 * @property bool $resolvable
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read bool $is_external
 * @property-read string|null $relative_url
 */
class ProductImage extends Model implements Filterable, UniqueFieldsInterface
{
    use BulkImport, HasFactory, HasFilters;

    protected $casts = ['is_primary' => 'boolean', 'sort_order' => 'integer', 'resolvable' => 'boolean'];

    protected $fillable = ['url', 'name', 'sort_order', 'is_primary', 'resolvable', 'product_id'];

    protected $touches = ['product'];

    public static function getUniqueFields(): array
    {
        return ['product_id', 'url'];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function product()
    {
        return $this->belongsTo(Product::class)->select(['id', 'sku', 'name']);
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getUrlAttribute()
    {
        if (! empty($this->attributes['url']) && ! $this->is_external) {
            return Storage::disk('images')->url($this->attributes['url']);
        }

        return $this->attributes['url'] ?? null;
    }

    public function getRelativeUrlAttribute()
    {
        return $this->attributes['url'] ?? null;
    }

    public function getIsExternalAttribute()
    {
        return $this->isExternalUrl($this->attributes['url'] ?? null);
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * Remove image from the server if saved in it.
     */
    public function removeFromServer(bool $previousImage = false): bool
    {
        $url = $previousImage ? $this->getRawOriginal('url') : $this->attributes['url'];

        if (! empty($url) && ! $this->isExternalUrl($url)) {
            return Helpers::deleteImage($url);
        }

        return true;
    }

    /**
     * {@inheritDoc}
     */
    public function delete()
    {
        $this->removeFromServer();

        parent::delete();
    }

    /**
     * {@inheritDoc}
     */
    public function save(array $options = [])
    {
        if ($this->is_primary) {
            // reset primary image to product
            if (($this->exists && ! $this->getOriginal('is_primary')) || ! $this->exists) {
                self::with([])->where('product_id', $this->product_id)->update(['is_primary' => false]);
                // reprint sales orders packing slips
                AddPackingSlipQueueObserver::productPrimaryImageUpdated($this);
            }
        }

        // remove old image
        if ($this->getRawOriginal('url') != $this->attributes['url']) {
            $this->removeFromServer(true);
        }

        return parent::save($options);
    }

    private function isExternalUrl(?string $url)
    {
        return isset(parse_url($url)['scheme']);
    }

    public function updateData(array $data, $imageUrl)
    {
        $this->fill($data);
        $this->url = $imageUrl ?: $this->url;
        $this->resolvable = boolval($imageUrl ?: $this->resolvable);

        return $this->save();
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return ['url', 'name'];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->availableColumns();
    }
}
