<?php

namespace App\Models;

use App\Exporters\MapsExportableFields;
use App\Importers\DataImporter;
use App\Importers\DataImporters\ProductCategoryDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

/**
 * Class ProductCategory.
 *
 *
 * @property int $id
 * @property int|null $parent_id
 * @property string $name
 * @property bool $is_leaf
 * @property string $path
 * @property int $level
 * @property int|null $root_id
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $archived_at
 */
class ProductCategory extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable
{
    use Archive
  {
      archive as baseArchive;
      unarchived as baseUnarchived;
  }
    use HasFactory;
    use HasFilters, HasSort;

    protected $casts = ['is_leaf' => 'boolean', 'archived_at' => 'datetime', 'level' => 'integer'];

    protected $hidden = ['created_at', 'updated_at'];

    protected $fillable = ['name', 'parent_id'];

    /**
     * used to prevent joins with other tables, only use sub-query.
     *
     * @var bool
     */
    protected $relationsByJoin = false;

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function parent()
    {
        return $this->belongsTo(self::class, 'parent_id');
    }

    public function parents()
    {
        return $this->parent()->with('parents');
    }

    public function children()
    {
        return $this->hasMany(self::class, 'parent_id');
    }

    public static function buildTree($parentId = null, $collection = null)
    {
        if (!$collection) {
            $collection = ProductCategory::all();
        }

        $categories = $collection->where('parent_id', $parentId);

        return $categories->map(function ($item) use ($collection) {
            $children = ProductCategory::buildTree($item->id, $collection);
            return [
                'id' => $item->id,
                'name' => $item->name,
                'parent_id' => $item->parent_id,
                'is_leaf' => $item->is_leaf,
                'subcategories_count' => ProductCategory::getSubcategoriesCount($children),
                'children' => $children
            ];
        })->values();
    }

    public static function getSubcategoriesCount($children)
    {
        $count = count($children);
        foreach ($children as $child) {
            $count += ProductCategory::getSubcategoriesCount($child['children']);
        }

        return $count;
    }

    public function root()
    {
        return $this->belongsTo(self::class, 'root_id');
    }

    public function childrenWithSub()
    {
        return $this->children()->with('childrenWithSub');
    }

    public function products()
    {
        return $this->belongsToMany(Product::class, 'product_to_categories', 'category_id')
            ->using(ProductToCategory::class)
            ->withPivot('is_primary')
            ->withTimestamps();
    }

    public function attributeGroups()
    {
        return $this->belongsToMany(AttributeGroup::class, 'categories_to_attribute_groups', 'category_id')
            ->withTimestamps();
    }

    public function attributes()
    {
        return $this->belongsToMany(Attribute::class, 'categories_to_attributes', 'category_id')
            ->withTimestamps();
    }

    public function categoryToProducts()
    {
        return $this->hasMany(ProductToCategory::class, 'category_id');
    }

    public function categoryToAttributeGroups()
    {
        return $this->hasMany(CategoryToAttributeGroup::class, 'category_id');
    }

    public function categoryToAttributes()
    {
        return $this->hasMany(CategoryToAttribute::class, 'category_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    /*
    |--------------------------------------------------------------------------
    | Custom Functions
    |--------------------------------------------------------------------------
    */

    public function delete()
    {
        if (! $this->is_leaf) {
            return ['productCategory' => __('messages.category.delete_failed')];
        }

        // delete product associations with the category
        $this->products()->detach();
        // delete attribute group associations with the category
        $this->attributeGroups()->detach();
        // delete attribute associations with the category
        $this->attributes()->detach();

        $delete = parent::delete();

        // set parent to leaf if becomes childless
        if ($delete && $this->parent_id && $this->parent->children()->count() == 0) {
            $this->parent->is_leaf = true;
            $this->parent->save();
        }

        return $delete;
    }

    public function save(array $options = [])
    {
        // reload and rebuild category path to save
        $this->load('parents', 'childrenWithSub');

        // rebuild category path/level
        $path = $this->buildPath(null, [], false);

        $this->path = implode(' -> ', array_column($path, 'name'));
        $this->level = count($path);
        $this->root_id = $path[0]->id == $this->id ? null : $path[0]->id;

        // check if parent is leaf to reassign its products to the "Other" child
        if ($options['check_parent'] ?? true) {
            if ($this->parent_id) {
                $parentCategory = $this->parents;
                if ($parentCategory && $parentCategory->is_leaf) {
                    // if has assigned products
                    if ($parentCategory->categoryToProducts()->count()) {
                        // create a new category with name "Other"
                        $otherChild = new self();
                        $otherChild->name = 'Other';
                        $otherChild->is_leaf = true;
                        $otherChild->parent_id = $parentCategory->id;
                        // add as child to parent category
                        $otherChild->save(['check_parent' => false]);
                        // reassign products of parent category to "Other" category
                        $parentCategory->categoryToProducts()->update(['category_id' => $otherChild->id]);
                    }

                    // set parent category to un-leaf
                    $parentCategory->is_leaf = false;
                    $parentCategory->save(['check_parent' => false]);
                }
            }

            $this->is_leaf = $this->childrenWithSub->isEmpty();
        }

        if ($this->parent_id == 0) {
            $this->parent_id = null;
        }

        $save = parent::save($options);

        // rebuild its children path
        if ($save) {
            $ids = [];
            $this->getSubcategoriesIds(null, $ids, true);
        }

        return $save;
    }

    /**
     * {@inheritDoc}
     */
    public function archive(?callable $callback = null)
    {
        return $this->baseArchive(function () {
            $this->load('childrenWithSub');

            self::with([])->whereIn('id', $this->getSubcategoriesIds())->update(['archived_at' => now()]);
        });
    }

    /**
     * {@inheritDoc}
     */
    public function unarchived(?callable $callback = null)
    {
        return $this->baseUnarchived(function () use ($callback) {
            $this->load('childrenWithSub');

            self::with([])->whereIn('id', $this->getSubcategoriesIds())->update(['archived_at' => null]);

            if ($callback) {
                $callback();
            }
        });
    }

    /**
     * Build the category path.
     */
    private function buildPath(?self $category = null, array $path = [], bool $onlyName = true): array
    {
        if (is_null($category)) {
            $category = $this;
        }

        $path[] = $onlyName ? $category->name : $category;
        if ($category->parents) {
            return $this->buildPath($category->parents, $path, $onlyName);
        }

        return array_reverse($path);
    }

    /**
     * Get the category id and all its subcategories.
     */
    public function getSubcategoriesIds(?self $category = null, array &$ids = [], bool $rebuildChildrenPath = false): array
    {
        if (empty($category)) {
            $category = $this;
        }

        $ids[] = $category->id;
        /** @var self $child */
        foreach ($category->childrenWithSub as $child) {
            $this->getSubcategoriesIds($child, $ids);

            // rebuild child path
            if ($rebuildChildrenPath) {
                $child->path = implode(' -> ', $child->buildPath($child));
                $child->save();
            }
        }

        return array_unique($ids);
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.product_category.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        if (Str::startsWith(func_get_arg(0), ['category_tree.'])) {
            return func_get_args();
        }

        return array_merge(collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all(), [
            'parent_id',
        ]);
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new ProductCategoryDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return ProductCategoryDataImporter::getExportableFields();
    }
}
