<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\Exporters\MapsExportableFields;
use App\Importers\DataImporter;
use App\Importers\DataImporters\ProductBrandDataImporter;
use App\Importers\ImportableInterface;
use App\Models\Concerns\Archive;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * Class ProductBrand.
 *
 *
 * @property int $id
 * @property string $name
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $archived_at
 */
class ProductBrand extends Model implements Filterable, ImportableInterface, MapsExportableFields, Sortable, UniqueFieldsInterface
{
    use Archive, BulkImport, HasFilters, HasSort;
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = ['name'];

    protected $casts = ['name' => 'string', 'archive_at' => 'datetime'];

    public static function getUniqueFields(): array
    {
        return ['name'];
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function products()
    {
        return $this->hasMany(Product::class, 'brand_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function delete()
    {
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return parent::delete();
    }

    public function isUsed()
    {
        // used in products
        $this->loadCount('products');
        if ($this->products_count) {
            return [
                'Products' => trans_choice('messages.currently_used', $this->products_count, [
                    'resource' => 'product',
                    'model' => 'product brand('.$this->name.')',
                ]),
            ];
        }

        return false;
    }

    /**
     * Get available columns to show in datatable.
     */
    public function availableColumns(): array
    {
        return config('data_table.product_brand.columns');
    }

    /**
     * Get available columns to show in datatable.
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['name'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public function getImporter(string $filePath): DataImporter
    {
        return new ProductBrandDataImporter(null, $filePath);
    }

    public static function getExportableFields(): array
    {
        return ProductBrandDataImporter::getExportableFields();
    }
}
