<?php

namespace App\Models;

use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;

/**
 * Class PaymentType.
 *
 *
 * @property int $id
 * @property string $name
 * @property int $user_id
 * @property int $accounting_integration_id
 * @property string $accounting_integration_type
 * @property-read mixed $accountingIntegration
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 */
class PaymentType extends Model implements Filterable, Sortable
{
    use Archive, HasFilters, HasSort;
    use HasFactory;

    const SALES_CREDIT_PAYMENT_TYPE_NAME = 'Credit';

    const PAYMENT_TYPE_CASH = 'Cash';

    const PAYMENT_TYPE_AMAZON = 'Amazon Payment';

    const PAYMENT_TYPE_SHOPIFY = 'Shopify Payment';

    const PAYMENT_TYPE_MAGENTO = 'Magento Payment';

    const SYSTEM_PAYMENT_TYPES = [
        self::SALES_CREDIT_PAYMENT_TYPE_NAME,
        self::PAYMENT_TYPE_AMAZON,
        self::PAYMENT_TYPE_SHOPIFY,
        self::PAYMENT_TYPE_MAGENTO,
    ];

    protected $fillable = [
        'name',
        'accounting_integration_id',
        'accounting_integration_type',
    ];

    protected $casts = ['archived_at' => 'datetime'];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    public function accountingIntegration(): MorphTo
    {
        return $this->morphTo('accountingIntegration');
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function save(array $options = [])
    {
        if (! $this->exists && empty($this->user_id)) {
            $this->user_id = auth()->id();
        }

        return parent::save($options);
    }

    /**
     * delete payment type
     *
     * @throws Exception
     */
    public function delete(): bool|array|null
    {
        // If the model was used from any related relation, we will archived it and its variants
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return parent::delete();
    }

    /**
     * check if there is related purchase orders.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        // Credit payment type cannot be deleted: SKU-2313
        if ($this->name === self::SALES_CREDIT_PAYMENT_TYPE_NAME) {
            return [
                'payment_type' => trans_choice('messages.internal_not_deletable', null, [
                    'resource' => 'payment type',
                    'model' => 'payment type('.$this->name.')',
                ]),
            ];
        }

        // load related relations
        if (! isset($this->payments_count)) {
            $this->loadCount('payments');
        }

        if ($this->payments_count) {
            return [
                'payments' => trans_choice('messages.currently_used', $this->payments_count, [
                    'resource' => 'payment',
                    'model' => 'payment type('.$this->name.')',
                ]),
            ];
        }

        return false;
    }

    /**
     * @inheritDoc
     */
    public function availableColumns()
    {
        return config('data_table.payment_type.columns');
    }

    /**
     * @inheritDoc
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * @inheritDoc
     */
    public function generalFilterableColumns(): array
    {
        return ['name'];
    }

    /**
     * @inheritDoc
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public static function getForSalesCredit(): self
    {
        $paymentType = self::with([])->where('name', static::SALES_CREDIT_PAYMENT_TYPE_NAME)->first();
        if (! $paymentType) {
            // Create the payment type
            $paymentType = self::with([])->create(['name' => static::SALES_CREDIT_PAYMENT_TYPE_NAME]);
        }

        return $paymentType;
    }
}
