<?php

namespace App\Models;

use App\Jobs\GenerateSalesOrderFulfillmentPackingSlipsJob;
use App\Jobs\GenerateSalesOrderPackingSlipsJob;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

/**
 * Class PackingSlipQueue
 *
 * @property string $id
 * @property string $link_type
 * @property int $link_id
 * @property string $status
 * @property string $event
 * @property string $last_error
 * @property Carbon|null $queued_at
 * @property Carbon|null $create_at
 * @property Carbon|null $updated_at
 * @property-read Model|SalesOrder|SalesOrderFulfillment $link
 *
 * @method static Builder|static processing()
 * @method static Builder|static pending()
 */
class PackingSlipQueue extends Model
{
    protected $guarded = [];

    const STATUS_PENDING = 'pending';

    const STATUS_QUEUED = 'queued';

    const STATUS_IN_PROGRESS = 'in_progress';

    const STATUS_PRINTED = 'printed';

    const STATUS_FAILED = 'failed';

    const EVENT_CREATE = 'create';

    const EVENT_UPDATE = 'update';

    protected $table = 'packing_slip_queue';

    protected $attributes = ['status' => self::STATUS_PENDING];

    protected $casts = [
        'queued_at' => 'datetime',
    ];

    public function link()
    {
        return $this->morphTo('link');
    }

    public function scopeProcessing(Builder $builder): Builder
    {
        return $builder->whereIn('status', [
            self::STATUS_QUEUED,
            self::STATUS_IN_PROGRESS,
        ]);
    }

    public function scopePending(Builder $builder): Builder
    {
        // maybe we should add failed status but after sometime
        return $builder->whereIn('status', [self::STATUS_PENDING]);
    }

    public function dispatch()
    {
        if ($this->link_type == SalesOrder::class) {
            GenerateSalesOrderPackingSlipsJob::dispatch($this->link_id, $this)->onQueue('packing-slips');
        }

        if ($this->link_type == SalesOrderFulfillment::class) {
            GenerateSalesOrderFulfillmentPackingSlipsJob::dispatch($this->link_id, $this)->onQueue('packing-slips');
        }

        $this->status = self::STATUS_QUEUED;
        $this->queued_at = now();
        $this->save();
    }

    public function printed()
    {
        $this->delete();
    }

    public function printing()
    {
        $this->status = self::STATUS_IN_PROGRESS;
        $this->save();
    }

    public function failed(string $errorMessage)
    {
        $this->last_error = $errorMessage;
        $this->status = self::STATUS_PENDING; // back to pending
        $this->save();
    }
}
