<?php

namespace App\Models\Magento;

use App\Exceptions\IntegrationInstance\Magento\UnmappedStoreException;
use App\Exceptions\IntegrationInstance\OrderCurrencyNotFoundException;
use App\Helpers;
use App\Http\Requests\FulfillSalesOrderRequest;
use App\Http\Resources\Magento\OrderResource;
use App\Integrations\SalesChannelOrder;
use App\Jobs\ShipStation\AutoFulfillmentOrder;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Models\Currency;
use App\Models\IntegrationInstance;
use App\Models\Magento\Store as MagentoStore;
use App\Models\PaymentMethodMappingSalesChannelToSku;
use App\Models\PaymentType;
use App\Models\ProductListing;
use App\Models\ReturnReason;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturn;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\ShippingMethodMappingsSalesChannelToSku;
use App\Models\StoreMapping;
use App\Models\Warehouse;
use App\Repositories\WarehouseRepository;
use App\Services\SalesOrder\FulfillSalesOrderService;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property string $increment_id
 * @property int|null $sales_order_id
 * @property string $downloaded_by
 * @property string $updated_by
 * @property array $json_object
 * @property array $errors
 * @property Collection $refunds_map
 * @property Collection $returns_map
 * @property Collection $fulfillments_map
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property Carbon|null $archived_at
 * @property-read Carbon $magento_created_at
 * @property-read Carbon $magento_updated_at
 * @property-read IntegrationInstance $integrationInstance
 * @property-read SalesOrder|null $salesOrder
 *
 * @method static Builder|static latestOrder(int $integrationInstanceId, string $downloadedBy = null, string $column = 'updated_at')
 */
class Order extends Model implements Filterable, SalesChannelOrder, Sortable
{
    use Archive, HasFilters, HasSort;

    const LINE_ITEMS_QUERY = 'json_object.items';

    const LINE_ITEMS_QUERY_ID = 'item_id';

    const PRODUCT_LOCAL_QUERY_IDENTIFER = 'variant_id';

    const PRODUCT_LOCAL_FORIEN_IDENTIFER = 'product_id';

    public const CREATED_AT = 'sku_created_at';

    public const UPDATED_AT = 'sku_updated_at';

    protected $table = 'magento_orders';

    protected $casts = [
        'json_object' => 'array',
        'errors' => 'array',
        'refunds_map' => 'collection',
        'returns_map' => 'collection',
        'fulfillments_map' => 'collection',
        'archived_at' => 'datetime',
    ];

    protected $fillable = ['integration_instance_id', 'json_object', 'downloaded_by', 'updated_by'];

    public $dataTableKey = 'magento.order';

    public function integrationInstance()
    {
        return $this->belongsTo(IntegrationInstance::class);
    }

    public function salesOrder()
    {
        return $this->belongsTo(SalesOrder::class);
    }

    public function orderLineItems()
    {
        return $this->hasMany(OrderLineItem::class, 'magento_order_id');
    }

    public function isOpen(): bool
    {
        // TODO: review
        return $this->json_object['status'] == 'processing';
    }

    /**
     * Create sku sales order from magento order
     *
     * @throws Throwable
     */
    public function createSKUOrder(): SalesOrder|bool
    {
        return DB::transaction(function () {
            $existingOrder = $this->salesOrder;
            if (! $existingOrder) {
                $existingOrder = SalesOrder::with([])->where('sales_order_number', $this->increment_id)
                    ->whereHas('salesChannel', function ($builder) {
                        $builder->where('integration_instance_id', $this->integrationInstance->id);
                    })->first();

                if ($existingOrder) {
                    $this->sales_order_id = $existingOrder->id;
                    $this->save();

                    $this->updateSKUOrder();
                }
            }

            if ($existingOrder) {
                return $existingOrder;
            }

            $skuStoreId = null;
            $magentoStore = MagentoStore::where('magento_id', $this->json_object['store_id'])
                ->where('integration_instance_id', $this->integrationInstance->id)
                ->first();
            if ($magentoStore) {
                $magentoMapping = StoreMapping::where('magento_store_id', $magentoStore->id)
                    ->where('integration_instance_id', $this->integrationInstance->id)
                    ->first();

                if ($magentoMapping) {
                    $skuStoreId = $magentoMapping->sku_store_id;
                }
            }

            if (! $skuStoreId) {
                throw new UnmappedStoreException($this);
            }

            if (! $currencyId = Currency::query()->where('code', $this->json_object['order_currency_code'])->value('id')) {
                throw new OrderCurrencyNotFoundException($this->json_object['order_currency_code']);
            }

            $salesOrder = new SalesOrder();
            $salesOrder->sales_order_number = $this->increment_id;
            $salesOrder->sales_channel_id = $this->integrationInstance->salesChannel->id;
            $salesOrder->currency_id = $currencyId;
            $salesOrder->order_date = $this->json_object['created_at'];
            $salesOrder->store_id = $skuStoreId;
            $salesOrder->shipping_method_id = $this->getShippingMethodId();
            $salesOrder->requested_shipping_method = $this->json_object['shipping_description'] ?? null;
            $salesOrder->tax_total = $this->json_object['tax_amount'];
            $salesOrder->order_status = SalesOrder::STATUS_DRAFT;
            $salesOrder->is_tax_included = false;

            $salesOrder->setCustomer($this->getSkuCustomerAddress(), false, $this->integrationInstance->salesChannel->id);
            $salesOrder->setShippingAddress($this->getSkuShippingAddress());
            $salesOrder->setBillingAddress($this->getSkuBillingAddress());
            $salesOrder->save();

            $orderLines = $this->getSkuSalesOrderLines();
            $salesOrder->setSalesOrderLines($orderLines);
            $salesOrder->load('salesOrderLines');

            if ($this->json_object['status'] == 'canceled') {
                $this->markSalesOrderAsCancelled($salesOrder);

                return true;
            }

            $productLinesMapped = collect($orderLines)->where('is_product', true)
                ->where('product_id', null)->isEmpty();

            if ($productLinesMapped) {
                $salesOrder->approve();
                if (! $this->isPreAuditTrail()) {
                    $shipmentsMapped = $this->createFulfillments($salesOrder);
                    $this->handleReturns($salesOrder);
                } else {
                    // TODO: handle lines as no_audit_trail
                }
            }

            // Record order payments from payment and credit memos
            $this->createPayments($salesOrder);

            $this->archived_at = null;
            $this->errors = null;
            $this->sales_order_id = $salesOrder->id;
            $this->save();

            // is not PreAuditTrail and the Magento shipments mapped to sku fulfillments
            if (isset($shipmentsMapped) && $shipmentsMapped) {
                // Auto fulfill the sales order if the warehouses of lines belong to the automated warehouses
                (new AutoFulfillmentOrder($salesOrder))->fulfillOpenOrder();
            }

            return true;
        });
    }

    /**
     * Update sku sales order from Magento order
     *
     * @throws Throwable
     */
    public function updateSKUOrder(): bool
    {
        return DB::transaction(function () {
            $this->load('salesOrder');

            $salesOrder = $this->salesOrder;

            $salesOrder->shipping_method_id = $this->getShippingMethodId();
            $salesOrder->requested_shipping_method = $this->json_object['shipping_description'] ?? null;
            $salesOrder->tax_total = $this->json_object['tax_amount'];
            $salesOrder->setCustomer($this->getSkuCustomerAddress(), false, $this->integrationInstance->salesChannel->id);
            $salesOrder->setShippingAddress($this->getSkuShippingAddress());
            $salesOrder->setBillingAddress($this->getSkuBillingAddress());

            $orderLines = $this->getSkuSalesOrderLines();
            $salesOrder->setSalesOrderLines($orderLines, true);
            $salesOrder->load('salesOrderLines');

            if ($this->json_object['status'] == 'canceled') {
                $this->markSalesOrderAsCancelled($salesOrder);

                return true;
            }

            $productLinesMapped = collect($orderLines)->where('is_product', true)
                ->where('product_id', null)->isEmpty();

            if ($productLinesMapped) {
                $salesOrder->approve();
                if (! $this->isPreAuditTrail()) {
                    $shipmentsMapped = $this->createFulfillments($salesOrder);
                    $this->handleReturns($salesOrder);
                }
            }

            // Record order payments from payment and credit memos
            $this->createPayments($salesOrder);

            $this->archived_at = null;
            $this->errors = null;
            $this->sales_order_id = $salesOrder->id;
            $this->save();

            // is not PreAuditTrail and the Magento shipments mapped to sku fulfillments
            if (isset($shipmentsMapped) && $shipmentsMapped) {
                // Auto fulfill the sales order if the warehouses of lines belong to the automated warehouses
                (new AutoFulfillmentOrder($salesOrder))->fulfillOpenOrder();
            }

            return true;
        });
    }

    /**
     * when sku sales order deleted
     */
    public function skuOrderDeleted(): void
    {
        $this->sales_order_id = null;
        $this->fulfillments_map = null;
        $this->refunds_map = null;
        $this->returns_map = null;
        $this->save();
        $this->archive(); /** @see SKU-4130 */
    }

    /**
     * Create/Get the shipping method mapping and return the sku shipping method id
     */
    private function getShippingMethodId(): ?int
    {
        if (empty($this->json_object['shipping_description'])) {
            return null;
        }

        $mapping = ShippingMethodMappingsSalesChannelToSku::firstOrCreate([
            'sales_channel_id' => $this->integrationInstance->salesChannel->id,
            'sales_channel_method' => $this->json_object['shipping_description'],
        ]);

        return $mapping->shipping_method_id;
    }

    private function getSkuCustomerAddress(): ?array
    {
        $name = empty($this->json_object['customer_firstname']) && empty($this->json_object['customer_lastname']) ?
          null :
          (($this->json_object['customer_firstname'] ?? '').' '.($this->json_object['customer_lastname'] ?? ''));

        return [
            'name' => $name,
            'email' => $this->json_object['customer_email'] ?? null,
        ];
    }

    private function getSkuShippingAddress(): ?array
    {
        return $this->getSkuAddressFromMagento($this->json_object['extension_attributes']['shipping_assignments'][0]['shipping']['address'] ?? null);
    }

    private function getSkuBillingAddress(): ?array
    {
        return $this->getSkuAddressFromMagento($this->json_object['billing_address'] ?? null);
    }

    private function getSkuSalesOrderLines(): array
    {
        $lines = [];

        $salesChannel = $this->integrationInstance->salesChannel;
        $magentoLines = collect($this->json_object['items']);

        foreach ($magentoLines->where('product_type', '!=', 'configurable') as $item) {
            // the shipment_type (0 = ship together, 1 = separately) of the listing (the "separately" option, just separate by bundle options not by items)
            // we don't add the children of the bundle line because we can't ship it separately
            // TODO: we should save the bundle component quantities
            if (isset($item['parent_item']['product_type']) && $item['parent_item']['product_type'] == 'bundle') {
                continue;
            }

            /** @var ProductListing $listing */
            $listing = ProductListing::with([])->where('sales_channel_listing_id', $item['product_id'])
                ->where('sales_channel_id', $salesChannel->id)
                ->first();

            $itemAmount = $item['price'] ?: $item['parent_item']['price'] ?? 0;

            $lines[] = [
                'magento_product_id' => $item['product_id'], // only use when map listing
                'description' => $item['name'],
                'product_id' => $listing?->product_id,
                'product_listing_id' => $listing?->id,
                'quantity' => max(0, $item['qty_ordered'] - $item['qty_canceled']),
                'canceled_quantity' => $item['qty_canceled'],
                'sales_channel_line_id' => $item['item_id'],
                'is_product' => true,
                'warehouse_id' => $this->getPriorityWarehouseId($listing, $item),
                'amount' => $itemAmount - (($item['discount_amount'] ?? 0) ?: $item['parent_item']['discount_amount'] ?? 0),
                'discount_allocation' => 0,
                'tax_allocation' => $item['tax_amount'] ?? 0,
            ];
        }

        if (! empty($this->json_object['shipping_amount'])) {
            $lines[] = [
                'description' => $this->json_object['shipping_description'],
                'amount' => $this->json_object['shipping_amount'],
                'discount_allocation' => $this->json_object['shipping_discount_amount'],
                'tax_allocation' => $this->json_object['shipping_tax_amount'],
                'is_product' => false,
                'quantity' => 1,
            ];
        }

        return $lines;
    }

    /**
     * Handle Credit Memos
     *
     * @param  SalesOrder  $salesOrder
     *
     * const STATE_OPEN = 1;
     * const STATE_REFUNDED = 2;
     * const STATE_CANCELED = 3;
     *
     * @throws Throwable
     */
    private function handleRefunds(SalesOrder $salesOrder): void
    {
        $salesOrder->load('salesOrderLines');
        $refundsMap = $this->refunds_map ?: collect();

        // delete sales order fulfillments that canceled in Shopify
        $creditMemos = collect($this->json_object['credit_memos']);
        foreach ($refundsMap as $index => $refundMap) {
            $creditMemo = $creditMemos->firstWhere('entity_id', $refundMap['credit_memo_id']);
            // canceled
            if ((! $creditMemo || $creditMemo['state'] == 3) && ($skuSalesCredit = SalesCredit::with([])->find($refundMap['sales_credit_id']))) {
                $skuSalesCredit->delete();
                $refundsMap->offsetUnset($index);
            }
        }

        // open or refunded
        foreach ($creditMemos->where('state', '!=', 3) as $creditMemo) {
            $refundMap = $refundsMap->firstWhere('credit_memo_id', $creditMemo['entity_id']);
            if (! $refundMap || ! ($salesCredit = SalesCredit::find($refundMap['sales_credit_id']))) {
                $salesCredit = new SalesCredit([
                    'return_status' => SalesCredit::RETURN_STATUS_NOT_RETURNED,
                    'credit_date' => Carbon::parse($creditMemo['created_at']),
                    'store_id' => $this->integrationInstance->salesChannel->store_id,
                    'sales_credit_note' => collect($creditMemo['comments'])->pluck('comment')->implode(', '),
                    'to_warehouse_id' => Helpers::setting(Setting::KEY_SC_DEFAULT_WAREHOUSE, Warehouse::with([])->value('id')), ]);
                // credit_status is not in fillable attributes
                $salesCredit->credit_status = $creditMemo['state'] == 1 ? SalesCredit::CREDIT_STATUS_OPEN : SalesCredit::CREDIT_STATUS_CLOSED;

                $salesOrder->addSalesCredit($salesCredit);

                // add lines
                $lines = [];
                foreach ($creditMemo['items'] as $creditMemoLine) {
                    $orderLine = $salesOrder->salesOrderLines->firstWhere('sales_channel_line_id', $creditMemoLine['order_item_id']);

                    $lines[] = [
                        'quantity' => $creditMemoLine['qty'],
                        'description' => $creditMemoLine['name'],
                        'tax' => $creditMemoLine['tax_amount'],
                        'amount' => $creditMemoLine['price'],
                        'sales_order_line_id' => $orderLine->id,
                        'product_id' => $orderLine->product_id,
                    ];
                }
                // add shipping line
                if (! empty($creditMemo['shipping_amount'])) {
                    $lines[] = [
                        'quantity' => 1,
                        'description' => 'shipping',
                        'tax' => $creditMemo['shipping_tax_amount'],
                        'amount' => $creditMemo['shipping_amount'],
                    ];
                }
                $salesCredit->setSalesCreditLines($lines ?: false);

                $refundsMap->add(['sales_credit_id' => $salesCredit->id, 'credit_memo_id' => $creditMemo['entity_id']]);
            } else {
                if ($salesCredit->credit_status == SalesCredit::CREDIT_STATUS_OPEN && $creditMemo['state'] == 2) {
                    $salesCredit->credit_status     = SalesCredit::CREDIT_STATUS_CLOSED;
                    $salesCredit->sales_credit_note = collect($creditMemo['comments'])->pluck('comment')->implode(', ');
                    $salesCredit->save();
                }
            }
        }

        // save fulfillments map
        $this->refunds_map = $refundsMap->values();
        $this->save();
    }

    /**
     * Handle Returns
     *
     *
     * @throws Throwable
     */
    public function handleReturns(SalesOrder $salesOrder): void
    {
        $salesOrder->load('salesOrderLines');
        $returnsMap = $this->returns_map ?: collect();

        // TODO: check statuses, I could not find the available statuses
        $canceledStatuses = ['denied', 'rejected', 'closed'];

        // delete sales credits that canceled in Magento
        $returns = collect($this->json_object['returns']);
        foreach ($returnsMap as $index => $returnMap) {
            $return = $returns->firstWhere('entity_id', $returnMap['return_id']);
            if ((! $return || (in_array($return['status'], $canceledStatuses)))) {
                SalesCredit::with([])->find($returnMap['sales_credit_id'])?->delete();
                $returnsMap->offsetUnset($index);
            }
        }

        // not canceled
        foreach ($returns->whereNotIn('status', $canceledStatuses) as $return) {
            $returnMap = $returnsMap->firstWhere('return_id', $return['entity_id']);
            $updated = false;

            // check the mapped sales credit's data is different from the Magento return's data
            // if they were different, delete the sales credit to create it again
            if ($returnMap) {
                /** @var SalesCredit $salesCredit */
                $salesCredit = SalesCredit::find($returnMap['sales_credit_id']);
                if (! $salesCredit) {
                    $returnsMap = $returnsMap->where('return_id', '!=', $return['entity_id']);
                    $updated = true;
                }
                foreach ($return['items'] as $returnLine) {
                    $magentoLineItem = collect($this->json_object['items'])->firstWhere('item_id', $returnLine['order_item_id']);
                    if ($magentoLineItem['product_type'] == 'configurable') {
                        $magentoLineItem = collect($this->json_object['items'])->firstWhere('parent_item_id', $returnLine['order_item_id']);
                    } elseif ($magentoLineItem['product_type'] == 'simple' && ($magentoLineItem['parent_item']['product_type'] ?? null) == 'bundle') {
                        // we will assume complete bundle would be returned
                        if (collect($return['items'])->firstWhere('order_item_id', $magentoLineItem['parent_item_id'])) {
                            continue;
                        }

                        $magentoLineItem = collect($this->json_object['items'])->firstWhere('item_id', $magentoLineItem['parent_item_id']);
                        if ($magentoLineItem['qty_ordered'] == 1) {
                            $returnLine['qty_returned'] = 1;
                        } else {
                            throw new \Exception('the order has return on bundle product with quantity greater than 1');
                        }
                    }

                    $orderLine = $salesOrder->salesOrderLines->firstWhere('sales_channel_line_id', $magentoLineItem['item_id']);

                    /** @var SalesCreditLine $salesCreditLine */
                    $salesCreditLine = $salesCredit->salesCreditLines->firstWhere('sales_order_line_id', $orderLine->id);
                    // the credit line is not found or the received quantity is different or the line canceled
                    if (! $salesCreditLine || $salesCreditLine->received_quantity != $returnLine['qty_returned'] || in_array($returnLine['status'], ['rejected', 'denied'])) {
                        $salesCredit->delete();
                        $returnsMap = $returnsMap->where('return_id', '!=', $return['entity_id']);
                        $updated = true;
                        break;
                    }
                }
            }

            if (! $returnMap || $updated) {
                $salesCredit = new SalesCredit([
                    'credit_status' => SalesCredit::CREDIT_STATUS_OPEN,
                    'return_status' => SalesCredit::RETURN_STATUS_NOT_RETURNED,
                    'credit_date' => Carbon::parse($return['date_requested']),
                    'store_id' => $this->integrationInstance->salesChannel->store_id,
                    'sales_credit_note' => collect($return['comments'])->where('admin', null)->pluck('comment')->implode(', '),
                    'to_warehouse_id' => Helpers::setting(Setting::KEY_SC_DEFAULT_WAREHOUSE, Warehouse::with([])->value('id')), ]);

                $salesOrder->addSalesCredit($salesCredit);
                // the lines that have not been canceled
                $returnLines = collect($return['items'])->whereNotIn('status', ['rejected', 'denied']);
                $lines = [];
                foreach ($returnLines as $returnLine) {
                    $magentoLineItem = collect($this->json_object['items'])->firstWhere('item_id', $returnLine['order_item_id']);
                    if ($magentoLineItem['product_type'] == 'configurable') {
                        $magentoLineItem = collect($this->json_object['items'])->firstWhere('parent_item_id', $returnLine['order_item_id']);
                    } elseif ($magentoLineItem['product_type'] == 'simple' && ($magentoLineItem['parent_item']['product_type'] ?? null) == 'bundle') {
                        // we will assume complete bundle would be returned
                        if (collect($returnLines)->firstWhere('order_item_id', $magentoLineItem['parent_item_id'])) {
                            continue;
                        }

                        $magentoLineItem = collect($this->json_object['items'])->firstWhere('item_id', $magentoLineItem['parent_item_id']);
                        if ($magentoLineItem['qty_ordered'] == 1) {
                            $returnLine['qty_requested'] = 1;
                        } else {
                            throw new \Exception('the order has return on bundle product with quantity greater than 1');
                        }
                    }
                    $orderLine = $salesOrder->salesOrderLines->firstWhere('sales_channel_line_id', $magentoLineItem['item_id']);

                    $lines[$magentoLineItem['item_id']] = [
                        'quantity' => $returnLine['qty_requested'],
                        'description' => $orderLine->description,
                        //                      'tax'                 => $returnLine['tax_amount'],
                        'amount' => $orderLine->amount,
                        'sales_order_line_id' => $orderLine->id,
                        'product_id' => $orderLine->product_id,
                    ];
                }
                $salesCredit->setSalesCreditLines($lines ?: false);

                if ($returnLines->sum('qty_returned')) {
                    $salesCredit->load('salesCreditLines');

                    $receivedDate = collect($return['comments'])->firstWhere('status', 'received')['created_at'] ?? $salesCredit->credit_date;

                    $salesCreditReturn = new SalesCreditReturn([
                        'warehouse_id' => $salesCredit->to_warehouse_id,
                        'shipped_at' => $receivedDate,
                        'received_at' => $receivedDate,
                    ]);
                    $salesCredit->salesCreditReturns()->save($salesCreditReturn);

                    // reason: TODO: the Magento returned line returns the reason(integer value, need to get from another API), we can use it
                    // condition and action: TODO: the Magento returned line returns the condition(integer value, need to get from another API), we can use it
                    $returnReasonOther = ReturnReason::with([])->firstOrCreate(['name' => 'Other']);
                    $lines = [];
                    foreach ($returnLines->where('qty_returned', '>', 0) as $returnedLine) {
                        $magentoLineItem = collect($this->json_object['items'])->firstWhere('item_id', $returnedLine['order_item_id']);
                        if ($magentoLineItem['product_type'] == 'configurable') {
                            $magentoLineItem = collect($this->json_object['items'])->firstWhere('parent_item_id', $returnedLine['order_item_id']);
                        } elseif ($magentoLineItem['product_type'] == 'simple' && ($magentoLineItem['parent_item']['product_type'] ?? null) == 'bundle') {
                            // we will assume complete bundle would be returned
                            if (collect($returnLines)->firstWhere('order_item_id', $magentoLineItem['parent_item_id'])) {
                                continue;
                            }

                            $magentoLineItem = collect($this->json_object['items'])->firstWhere('item_id', $magentoLineItem['parent_item_id']);
                            if ($magentoLineItem['qty_ordered'] == 1) {
                                $returnedLine['qty_returned'] = 1;
                            } else {
                                throw new \Exception('the order has return on bundle product with quantity greater than 1');
                            }
                        }
                        $orderLine = $salesOrder->salesOrderLines->firstWhere('sales_channel_line_id', $magentoLineItem['item_id']);

                        $salesCreditLine = $salesCredit->salesCreditLines->firstWhere('sales_order_line_id', $orderLine->id);
                        if (! is_null($salesCreditLine->product_id)) {
                            $lines[$magentoLineItem['item_id']] = [
                                'sales_credit_line_id' => $salesCreditLine->id,
                                'quantity' => $returnedLine['qty_returned'],
                                'product_id' => $salesCreditLine->product_id,
                                'action' => SalesCreditReturnLine::ACTION_ADD_TO_STOCK,
                                'reason_id' => $returnReasonOther->id,
                            ];
                        }
                    }
                    $salesCreditReturn->setReturnLines($lines);

                    $salesCredit->returned($salesCreditReturn->received_at);
                }
                $returnsMap->push(['sales_credit_id' => $salesCredit->id, 'return_id' => $return['entity_id']]);
            }
        }

        // save returns map
        $this->returns_map = $returnsMap->values();
        $this->save();
    }

    /**
     * Marks the sales order as cancelled.
     */
    protected function markSalesOrderAsCancelled(SalesOrder $salesOrder): SalesOrder
    {
        $salesOrder->close();
        $salesOrder->canceled_at = $salesOrder->canceled_at ?: now();
        $salesOrder->save();

        // delete(cancel) the sales order fulfillments
        $salesOrder->salesOrderFulfillments->where('status', '!=', SalesOrderFulfillment::STATUS_FULFILLED)->each->delete(true);

        return $salesOrder;
    }

    /**
     * Add payments to the sales order
     * this function should call after all steps (createFulfillments, handleReturns)
     */
    private function createPayments(SalesOrder $salesOrder): void
    {
        /** @var PaymentMethodMappingSalesChannelToSku $mapping */
        $mapping = PaymentMethodMappingSalesChannelToSku::query()->firstOrCreate([
            'sales_channel_id' => $this->integrationInstance->salesChannel->id,
            'sales_channel_method' => $this->json_object['payment']['method'],
        ]);
        $magentoPaymentType = $mapping->paymentType;
        // unmapped
        if (! $mapping->paymentType) {
            $magentoPaymentType = PaymentType::with([])->firstOrCreate(['name' => $this->json_object['payment']['method']]);
            // link to mapping
            $mapping->payment_type_id = $magentoPaymentType->id;
            $mapping->save();
        }

        $payments = [];

        // 3 = STATE_CANCELED
        foreach (collect($this->json_object['invoices'])->where('state', '!=', 3) as $invoice) {
            $payments[] = [
                'payment_date' => $invoice['created_at'],
                'payment_type_id' => $magentoPaymentType->id,
                'amount' => $invoice['grand_total'],
                'currency_code' => $invoice['order_currency_code'],
                'external_reference' => $invoice['transaction_id'] ?? null,
            ];
        }

        foreach (collect($this->json_object['credit_memos'])->where('state', '!=', 3) as $creditMemo) {
            $payments[] = [
                'payment_date' => $creditMemo['created_at'],
                'payment_type_id' => $magentoPaymentType->id,
                'amount' => -$creditMemo['grand_total'],
                'currency_code' => $creditMemo['order_currency_code'],
                'external_reference' => $creditMemo['transaction_id'] ?? null,
            ];

            // add refunded quantity to lines
            foreach ($creditMemo['items'] as $creditMemoLine) {
                $magentoOrderLine = collect($this->json_object['items'])->firstWhere('item_id', $creditMemoLine['order_item_id']);
                if ($magentoOrderLine['product_type'] == 'configurable') {
                    // the variant line return in another line
                    continue;
                } elseif ($magentoOrderLine['product_type'] == 'simple' && ($magentoOrderLine['parent_item']['product_type'] ?? null) == 'bundle') {
                    // we will only take the bundle product
                    continue;
                }

                /** @var SalesOrderLine $orderLine */
                $orderLine = $salesOrder->salesOrderLines->firstWhere('sales_channel_line_id', $creditMemoLine['order_item_id']);
                $orderLine->quantity -= $creditMemoLine['qty'];
                $orderLine->canceled_quantity += $creditMemoLine['qty'];
                $orderLine->save();
            }
        }

        $salesOrder->payments()->delete(); // Clear any existing payments
        $salesOrder->payments()->createMany($payments);
        $salesOrder->setPaymentStatus();
    }

    public function createFulfillments(SalesOrder $salesOrder): bool
    {
        $salesOrder->load('salesOrderLines');
        $fulfillmentsMap = $this->fulfillments_map ?: collect();
        $result = true;

        // delete sales order fulfillments that canceled in Magento
        $magentoShipments = collect($this->json_object['shipments']);
        foreach ($fulfillmentsMap as $index => $fulfillmentMap) {
            $magentoShipment = $magentoShipments->firstWhere('entity_id', $fulfillmentMap['magento_shipment_id']);
            if (! $magentoShipment && ($skuFulfillment = SalesOrderFulfillment::with([])->find($fulfillmentMap['sku_fulfillment_id']))) {
                $skuFulfillment->delete(true);
                $fulfillmentsMap->offsetUnset($index);
            }
        }

        foreach ($magentoShipments as $shipment) {
            $shipmentLines = collect($shipment['items'])
                ->map(function ($line) {
                    $magentoOrderLine = collect($this->json_object['items'])->firstWhere('item_id', $line['order_item_id']);

                    if ($magentoOrderLine['product_type'] == 'configurable') {
                        $magentoOrderLine = collect($this->json_object['items'])->firstWhere('parent_item_id', $line['order_item_id']);
                        $line['order_item_id'] = $magentoOrderLine['item_id'];
                    } elseif ($magentoOrderLine['product_type'] == 'simple' && ($magentoOrderLine['parent_item']['product_type'] ?? null) == 'bundle') {
                        return null;
                    }

                    return $line;
                })->filter()->values();

            $orderLines = $salesOrder->salesOrderLines->whereIn('sales_channel_line_id', $shipmentLines->pluck('order_item_id'))->groupBy(['warehouse_id']);

            foreach ($orderLines as $warehouseId => $salesOrderLines) {
                $fulfillmentMap = $fulfillmentsMap->firstWhere('magento_shipment_id', $shipment['entity_id']);

                // the shipment is not mapped or the fulfillment deleted from sku
                if (! $fulfillmentMap || ! ($salesOrderFulfillment = SalesOrderFulfillment::find($fulfillmentMap['sku_fulfillment_id']))) {
                    $fulfillment = [
                        'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
                        'fulfilled_at' => $shipment['updated_at'],
                        'requested_shipping_method_id' => $salesOrder->shipping_method_id,
                        'requested_shipping_method' => $salesOrder->requested_shipping_method,
                        'fulfilled_shipping_method' => $shipment['tracks'][0]['carrier_code'] ?? null,
                        'tracking_number' => count($shipment['tracks']) ? collect($shipment['tracks'])->pluck('track_number')->implode(', ') : null,
                        'warehouse_id' => $warehouseId,
                        'fulfillment_lines' => $salesOrderLines->map(function ($orderLine) use ($shipmentLines) {
                            $shipmentLine = $shipmentLines->firstWhere('order_item_id', $orderLine->sales_channel_line_id);

                            return [
                                'sales_order_line_id' => $orderLine->id,
                                'quantity' => $shipmentLine['qty'],
                            ];
                        })->toArray(),
                    ];

                    // TODO: can't fulfill
                    try {
                        $fulfillRequest = FulfillSalesOrderRequest::createFromCustom($fulfillment, 'POST', ['sales_order' => $salesOrder]);
                        if (is_array($salesOrderFulfillment = FulfillSalesOrderService::make($salesOrder)->fulfill($fulfillRequest))) {
                            Log::debug("Can't create a fulfillment from a Magento shipment for {$this->increment_id}", $salesOrderFulfillment);
                            $result = false;

                            continue;
                        }
                    } catch (Throwable $exception) {
                        Log::debug("Can't create a fulfillment from a Magento shipment for {$this->increment_id}: {$exception->getMessage()}", $exception instanceof ValidationException ? $exception->errors() : []);
                        $result = false;

                        continue;
                    }

                    // We do this to avoid resending fulfillments to Magento when running the Submit Tracking Info command
                    $salesOrderFulfillment->submitted_to_sales_channel_at = now();
                    $salesOrderFulfillment->save();

                    // TODO: maybe the Magento shipment parse to many fulfillments
                    $fulfillmentsMap->add([
                        'sku_fulfillment_id' => $salesOrderFulfillment->id,
                        'magento_shipment_id' => $shipment['entity_id'],
                    ]);
                } else {
                    // update sku fulfillment if it was not have tracking number
                    if ($salesOrderFulfillment && empty($salesOrderFulfillment->tracking_number) && collect($shipment['tracks'])->isNotEmpty()) {
                        $salesOrderFulfillment->fulfilled_at = $shipment['updated_at'];
                        $salesOrderFulfillment->fulfilled_shipping_method = $shipment['tracks'][0]['carrier_code'] ?? null;
                        $salesOrderFulfillment->tracking_number = collect($shipment['tracks'])->pluck('track_number')->implode(', ');
                        $salesOrderFulfillment->save();
                    }
                }
            }
        }

        // save fulfillments map
        $this->fulfillments_map = $fulfillmentsMap->values();
        $this->save();

        return $result;
    }

    public function isPreAuditTrail(): bool
    {
        if (! $this->integrationInstance->audit_trail_start_date || $this->integrationInstance->audit_trail_start_date->lt(Carbon::parse($this->json_object['created_at'], 'UTC'))) {
            return false;
        } else {
            return true;
        }
    }

    private function getSkuAddressFromMagento(?array $address): ?array
    {
        if (empty($address)) {
            return null;
        }

        $skuAddress['name'] = "{$address['firstname']} {$address['lastname']}";
        $skuAddress['company'] = $address['company'] ?? null;
        $skuAddress['email'] = $address['email'] ?? null;
        $skuAddress['phone'] = $address['telephone'];
        $skuAddress['address1'] = $address['street'][0] ?? null;
        $skuAddress['address2'] = $address['street'][1] ?? null;
        $skuAddress['address3'] = empty($address['street'][2]) ? null : implode(', ', array_slice($address['street'] ?? [], 2));
        $skuAddress['city'] = $address['city'];
        $skuAddress['province'] = $address['region'] ?? null;
        $skuAddress['province_code'] = $address['region_code'] ?? null;
        $skuAddress['zip'] = $address['postcode'];
        $skuAddress['country_code'] = $address['country_id'];

        return $skuAddress;
    }

    private function getPriorityWarehouseId($product, $item, ?bool $defaultToFirst = true): ?int
    {
        if (! $product) {
            return null;
        }

        /** @var WarehouseRepository $warehouses */
        $warehouses = app()->make(WarehouseRepository::class);

        $product = $product instanceof ProductListing ? $product->product : $product;

        return $warehouses->getPriorityWarehouseIdForProduct($product, $item['qty_ordered'], $defaultToFirst,
            $this->getSkuShippingAddress());
    }

    public function mapFulfillmentToSkuFulfillment(int $magentoShipmentId, int $skuFulfillmentId)
    {
        $fulfillmentsMap = $this->fulfillments_map ?: [];
        $fulfillmentsMap[] = ['sku_fulfillment_id' => $skuFulfillmentId, 'magento_shipment_id' => $magentoShipmentId];
        $this->fulfillments_map = $fulfillmentsMap;
        $this->save();
    }

    public static function getOrderDateAttributeName(): string
    {
        return "JSON_EXTRACT(`json_object`, '$.created_at')";
    }

    public function scopeLatestOrder(Builder $builder, int $integrationInstanceId, ?string $downloadedBy = null, string $column = 'updated_at')
    {
        $builder->where('integration_instance_id', $integrationInstanceId)
            ->when($downloadedBy, function (Builder $builder) use ($downloadedBy) {
                $builder->where(function (Builder $builder) use ($downloadedBy) {
                    $builder->where('downloaded_by', $downloadedBy);
                    $builder->orWhere('updated_by', $downloadedBy);
                });
            })
            ->latest($column);
    }

    /**
     * Scope to filter by json_object->created_at
     * this scope as an example to filter on datetime field(in json column), you can customize format and timezone
     * in Magento, we can filter and sort by create_at/updated_at fields without casting
     */
    public function scopeWhereOrderDate(Builder $builder, string $operator, $value): void
    {
        $orderDateAttribute = static::getOrderDateAttributeName();
        $value = $value instanceof Carbon ? $value->toDateTimeString() : $value;
        // we can convert the timezone and format of the order date value based on what it gets from the sales channel
        $builder->whereRaw("CONVERT_TZ(STR_TO_DATE(JSON_UNQUOTE({$orderDateAttribute}), '%Y-%m-%d %H:%i:%s'), 'UTC', 'UTC') {$operator} '{$value}'");
    }

    public function getMagentoCreatedAtAttribute()
    {
        return Carbon::parse($this->json_object['created_at']);
    }

    public function getMagentoUpdatedAtAttribute()
    {
        return Carbon::parse($this->json_object['updated_at']);
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.magento.order.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['increment_id', 'customer_firstname', 'customer_lastname', 'customer_email'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public static function getResource()
    {
        return OrderResource::class;
    }

    /**
     * {@inheritDoc}
     */
    public function getSkuOrderLines(): array
    {
        return $this->getSkuSalesOrderLines();
    }

    /**
     * {@inheritDoc}
     */
    public function findMatchingLines(Collection $orderLines, ProductListing $productListing)
    {
        return $orderLines->where('magento_product_id', $productListing->sales_channel_listing_id);
    }

    /**
     * {@inheritDoc}
     */
    public function getSkuShippingMethodId()
    {
        return $this->getShippingMethodId();
    }

    /**
     * {@inheritDoc}
     */
    public function isFullyShipped(): bool
    {
        // we don't use it when mapping
    }

    /**
     * {@inheritDoc}
     */
    public function isPartiallyShipped(): bool
    {
        // we don't use it when mapping
    }

    /**
     * {@inheritDoc}
     */
    public function partiallyFulfill(SalesOrder $salesOrder)
    {
        // we don't use it when mapping
    }
}
