<?php

namespace App\Models;

use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\InventoryMovementTypeException;
use App\Exceptions\SupplierWarehouseCantHaveInventoryMovementsException;
use App\Exporters\MapsExportableFields;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Concerns\LogsActivity;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Repositories\StockTakeRepository;
use App\Services\StockTake\OpenStockTakeException;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;
use Spatie\Activitylog\LogOptions;

/**
 * Class InventoryMovement.
 *
 *
 * @property int $id
 * @property Carbon $inventory_movement_date
 * @property int $product_id
 * @property float $quantity
 * @property string $type
 * @property string $inventory_status
 * @property int $warehouse_id
 * @property int $warehouse_location_id
 * @property int $link_id
 * @property string $link_type
 * @property int $layer_id
 * @property string $layer_type
 * @property string|null $reference
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property FifoLayer|null $fifo_layer
 * @property FifoLayer|BackorderLayer $layer
 * @property BackorderQueue|null $backorder_queue
 * @property Product $product
 * @property-read int|null $parent_link_id
 * @property-read string $layer_name
 * @property-read Warehouse $warehouse
 * @property-read Model|SalesOrderLine $link
 * @property int $aging_quantity
 */
class InventoryMovement extends Model implements Filterable, MapsExportableFields, Sortable
{
    use BulkImport;
    use HandleDateTimeAttributes;
    use HasFactory;
    use HasFilters;
    use HasParentLink;
    use HasSort;
    use LogsActivity;

    /**
     * Inventory State.
     */
    const INVENTORY_STATUS_ACTIVE = 'active';

    const INVENTORY_STATUS_RESERVED = 'reserved'; // due to sales order placed but not yet shipped

    const INVENTORY_STATUS_IN_TRANSIT = 'in_transit'; // due to transfers between warehouses

    const INVENTORY_STATUS = [
        self::INVENTORY_STATUS_ACTIVE,
        self::INVENTORY_STATUS_RESERVED,
        self::INVENTORY_STATUS_IN_TRANSIT,
    ];

    /**
     * Inventory Movement Types.
     */
    const TYPE_PURCHASE_RECEIPT = 'purchase_receipt';

    const TYPE_TRANSFER = 'transfer'; // between warehouses

    const TYPE_SALE = 'sale';

    const TYPE_ADJUSTMENT = 'adjustment';

    const TYPE_STOCK_TAKE = 'stock_take';

    const TYPE_ASSEMBLY = 'assembly';

    const TYPE_RETURN = 'return';

    const TYPE_RECLASSIFICATION = 'reclassification';

    const TYPES = [
        self::TYPE_PURCHASE_RECEIPT,
        self::TYPE_TRANSFER,
        self::TYPE_RECLASSIFICATION,
        self::TYPE_SALE,
        self::TYPE_ADJUSTMENT,
        self::TYPE_STOCK_TAKE,
        self::TYPE_ASSEMBLY,
        self::TYPE_RETURN,
    ];

    /**
     * Layer Types.
     */
    const LAYER_TYPE_FIFO = 'fifo';

    const LAYER_TYPE_BACKORDER = 'backorder';

    const LAYER_TYPES = [
        FifoLayer::class => self::LAYER_TYPE_FIFO,
        BackorderQueue::class => self::LAYER_TYPE_BACKORDER,
    ];

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    protected $casts = [
        'inventory_movement_date' => 'datetime',
        'quantity' => 'float',
        'layer_id' => 'integer',
        'fulfilled_quantity' => 'integer',
    ];

    protected $guarded = [];

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    public function getMetadataForActivityLog(): ?array
    {
        return [];
    }

    public function getParentSubjectIdForActivityLog(): int
    {
        return $this->id;
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function warehouseLocation()
    {
        return $this->belongsTo(WarehouseLocation::class);
    }

    public function layer()
    {
        return $this->morphTo('layer');
    }

    public function link()
    {
        return $this->morphTo('link');
    }

    public function salesCreditReturnLine(): BelongsTo
    {
        return $this->belongsTo(SalesCreditReturnLine::class, 'link_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getFifoLayerAttribute()
    {
        return $this->layer instanceof FifoLayer ? $this->layer : null;
    }

    public function setFifoLayerAttribute($value)
    {
        $this->layer_id = $value;
        $this->layer_type = FifoLayer::class;
    }

    public function getBackorderQueueAttribute()
    {
        return $this->layer instanceof BackorderQueue ? $this->layer : null;
    }

    public function setBackorderQueueAttribute($value)
    {
        $this->layer_id = $value;
        $this->layer_type = BackorderQueue::class;
    }

    public function getLayerNameAttribute()
    {
        if (! isset(self::LAYER_TYPES[$this->layer_type])) {
            return null;
        }

        return self::LAYER_TYPES[$this->layer_type];
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function save(array $options = [])
    {
        if ($this->isDirty('warehouse_id') && $this->getOriginal('warehouse_id') != null) {
            customlog('SKU-6240', 'Warehouse changing for inventory movement '.$this->id.' (SKU: '.$this->product?->sku.')', [
                'warehouse_id_from' => $this->getOriginal('warehouse_id'),
                'warehouse_id_to' => $this->warehouse_id,
                'debug' => debug_pretty_string(),
            ]);
        }

        $product = Product::find($this->product_id);

        // Bundles and matrix products cannot have inventory movements
        if ($product && in_array($product->type, [Product::TYPE_BUNDLE, Product::TYPE_MATRIX])) {
            throw new InventoryMovementTypeException();
        }

        // Prevent inventory movement if there's a stock take
        // at the warehouse and for the product unless the movement
        // is for the given stock take in question.
        $openStockTake = StockTakeRepository::findOpenStockTake($this->warehouse_id, $this->product_id);
        if ($openStockTake) {
            $isThisStockTake = ($this->link && $this->link instanceof StockTakeItem && $this->link->stockTake->id === $openStockTake->id);
            if (! $isThisStockTake) {
                throw new OpenStockTakeException($openStockTake, 'Inventory activity locked as a result of open stock take: '.$openStockTake->id);
            }
        }

        if (empty($this->warehouse_location_id)) {
            $this->warehouse_location_id = $this->warehouse->defaultLocation->id ?? null;
        }

        // Supplier warehouses cannot have inventory movements
        $warehouse = Warehouse::with([])->findOrFail($this->warehouse_id);
        if ($warehouse->isSupplierWarehouse()) {
            throw new SupplierWarehouseCantHaveInventoryMovementsException('Supplier warehouse cannot have inventory movements.');
        }

        // cache the reference
        if ($this->link_type) {
            $this->loadMissing('link');
            $this->reference = $this->link?->getReference();
        }

        return parent::save($options);
    }

    /**
     * Reassign fifo-layer of the inventory movement.
     *
     *
     * @throws InsufficientStockException|OpenStockTakeException
     */
    public function reassignFifoLayer(?int $fifoLayerId = null)
    {
        $fifoLayerId = $fifoLayerId ?: $this->layer_id;

        $this->product->load([
            'activeFifoLayers' => function (HasMany $builder) use ($fifoLayerId) {
                $builder->where('id', '!=', $fifoLayerId);
            },
        ]);

        $activeFifoLayer = $this->product->getCurrentFifoLayerForWarehouse($this->warehouse);

        // if product does not have active fifo layer
        if (! $activeFifoLayer) {
            throw new InsufficientStockException($this->product_id);
        }

        $requiredQuantity = abs($this->quantity);
        $remainQuantity = 0;

        // if current fifo layer can't fulfill all required quantity.
        if ($requiredQuantity > $activeFifoLayer->available_quantity) {
            $remainQuantity = $requiredQuantity - $activeFifoLayer->available_quantity;

            $requiredQuantity = $activeFifoLayer->available_quantity;
        }

        // reassign sales order link with fifo-layer
        if ($this->link_type == SalesOrderLine::class) {
            // detach previous link between sales order line and fifo-layer
            if ($this->exists) {
                $this->link->fifoLayers()->detach($this->layer_id);
            }

            // add link between sales order line and active fifo layer
            $this->link->fifoLayers()->attach($activeFifoLayer->id, ['quantity' => $requiredQuantity]);

            // update/create reserved movement for this active movement
            $reservedMovement = $this->link->inventoryMovements()
                ->where('inventory_status', self::INVENTORY_STATUS_RESERVED)
                ->where('layer_type', FifoLayer::class)
                ->where('layer_id', $this->layer_id)
                ->first();
            if (! $reservedMovement) {
                $reservedMovement = $this->replicate();
                $reservedMovement->inventory_status = self::INVENTORY_STATUS_RESERVED;
            }

            $reservedMovement->fifo_layer = $activeFifoLayer->id;
            $reservedMovement->quantity = $requiredQuantity;
            $reservedMovement->save();

            // TODO: we need to reassign reserved movements for SalesOrderFulfillmentLine
        }

        // set quantity and fifo layer for inventory movement
        $this->fifo_layer = $activeFifoLayer->id;
        $this->quantity = -$requiredQuantity;
        $this->save();

        // fulfill from active fifo-layer
        $activeFifoLayer->fulfilled_quantity += $requiredQuantity;
        $activeFifoLayer->save();

        // new movement for remaining quantity
        if ($remainQuantity) {
            $newMovement = $this->replicate();
            $newMovement->quantity = -$remainQuantity;
            $newMovement->layer_id = null; // to prevent update reserved quantity of origin movement
            $newMovement->reassignFifoLayer($fifoLayerId);
        }
    }

    public function split(int $quantity)
    {
        $this->quantity += $quantity;
        $this->save();

        // update reserved movement quantity and sales order line link with fifo-layer
        if ($this->link_type == SalesOrderLine::class) {
            // add link between sales order line and active fifo layer
            $this->link->fifoLayers()->syncWithoutDetaching([$this->layer_id => ['quantity' => abs($this->quantity)]]);

            // update/create reserved movement for this active movement
            $reservedMovement = $this->link->inventoryMovements()
                ->where('inventory_status', self::INVENTORY_STATUS_RESERVED)
                ->where('layer_type', FifoLayer::class)
                ->where('layer_id', $this->layer_id)
                ->first();

            $reservedMovement->quantity = abs($this->quantity);
            $reservedMovement->save();
        }

        $newMovement = $this->replicate();
        $newMovement->quantity = -$quantity;
        $newMovement->layer_id = null; // to prevent update reserved quantity of origin movement
        $newMovement->reassignFifoLayer($this->layer_id);
    }

    public function reverseLayer(): void
    {
        if ($this->inventory_status !== self::INVENTORY_STATUS_ACTIVE) {
            return;
        }
        if ($this->layer_type === BackorderQueue::class) {
            $this->layer()->delete();
        } else {
            $fifoLayer = $this->fifo_layer;
            if ($fifoLayer) {
                $fifoLayer->fulfilled_quantity = max(0, $fifoLayer->fulfilled_quantity - ($this->inventory_status === self::INVENTORY_STATUS_ACTIVE ? abs($this->quantity) : 0));
                $fifoLayer->save();
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.inventory_movement.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        if (Str::startsWith(func_get_arg(0), ['type.', 'warehouse.'])) {
            return func_get_args();
        }

        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['reference', 'type', 'product.sku'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', false, 'ID'),
            'inventory_movement_date' => Exporter::makeExportableField('inventory_movement_date', false),
            'sku' => Exporter::makeExportableField('sku', false),
            'product_name' => Exporter::makeExportableField('product_name', false),
            'type.name' => Exporter::makeExportableField('type', false),
            'inventory_status' => Exporter::makeExportableField('inventory_status', false),
            'quantity' => Exporter::makeExportableField('quantity', false),
            'warehouse.name' => Exporter::makeExportableField('warehouse_name', false),
            'layer.name' => Exporter::makeExportableField('layer_name', false),
            'layer.layer_id' => Exporter::makeExportableField('layer_id', false),
            'reference' => Exporter::makeExportableField('reference', false),
            'created_at' => Exporter::makeExportableField('created_at', false),
            'updated_at' => Exporter::makeExportableField('updated_at', false),
        ];
    }

    public function scopeWithInventoryDetails($builder)
    {
        $builder->selectRaw("SUM(CASE WHEN inventory_status != '".self::INVENTORY_STATUS_IN_TRANSIT."' THEN quantity ELSE 0 END) as in_warehouse_quantity")
            ->selectRaw("SUM(CASE WHEN inventory_status = '".self::INVENTORY_STATUS_ACTIVE."' THEN quantity ELSE 0 END) as in_warehouse_availability_quantity")
            ->selectRaw("SUM(CASE WHEN inventory_status = '".self::INVENTORY_STATUS_RESERVED."' AND type != '".self::TYPE_TRANSFER."' THEN quantity ELSE 0 END) as in_warehouse_reserved_quantity")
            ->selectRaw("SUM(CASE WHEN inventory_status = '".self::INVENTORY_STATUS_IN_TRANSIT."' THEN quantity ELSE 0 END) as in_warehouse_transit_quantity");
    }

    public function scopeGroupInventory($builder, $columns)
    {
        $builder->selectRaw(implode(',', $columns))->groupBy($columns);
    }

    public function scopeFilterLayer(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        if ($relation['key'] == 'layer_type') {
            $value = str_replace('-', '\\', $value);

            return $builder->filterKey('layer_type', $operator, $value, $conjunction);
        }

        return false; // default behavior
    }

    public function scopeSortLayer(Builder $builder, array $relation, string $ascending)
    {
        return $builder->sortKey($relation['key'], $ascending);
    }

    public function scopeBeforeInventoryDate(Builder $query, $date)
    {
        return $query->where('inventory_movement_date', '<=', Carbon::parse($date));
    }
}
