<?php

namespace App\Models;

use App\Contracts\HasReference;
use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Exporters\BaseExporter;
use App\Exporters\MapsExportableFields;
use App\Exporters\TransformsExportData;
use App\Managers\ProductInventoryManager;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\DB;

/**
 * Class InventoryAssembly.
 *
 * @property string $action
 * @property int $warehouse_id
 * @property Carbon $action_date
 */
class InventoryAssembly extends Model implements Filterable, HasReference, MapsExportableFields, Sortable, TransformsExportData
{
    use Archive, HasFilters, HasSort;

    const INVENTORY_ASSEMBLY_ACTION_ASSEMBLE = 'assemble';

    const INVENTORY_ASSEMBLY_ACTION_DISASSEMBLE = 'disassemble';

    const INVENTORY_ASSEMBLY_ACTIONS = [
        self::INVENTORY_ASSEMBLY_ACTION_ASSEMBLE,
        self::INVENTORY_ASSEMBLY_ACTION_DISASSEMBLE,
    ];

    /**
     * @var array
     */
    protected $fillable = [
        'warehouse_id',
        'action_date',
    ];

    /**
     * @var array
     */
    protected $casts = ['action_date' => 'datetime'];

    // Allow filtering by joins
    protected $relationsByJoin = true;

    /*
     |--------------------------------------------------------------------------
     | Relations
     |--------------------------------------------------------------------------
     */

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class, 'warehouse_id');
    }

    public function inventoryAssemblyLines(): HasMany
    {
        return $this->hasMany(InventoryAssemblyLine::class);
    }

    public function getReference(): ?string
    {
        return $this->id;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.inventory_assembly.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['id', 'action'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function inventoryMovements()
    {
        return $this->morphMany(InventoryMovement::class, 'link');
    }

    /**
     * @return bool|mixed|null
     */
    public function delete()
    {
        //Load deletable relationships

        $this->load('inventoryAssemblyLines');

        return DB::transaction(function () {
            // Delete related models

            /*
             * Is this needed if the db has cascading?
             */
            // Delete transfer lines
            if ($this->inventoryAssemblyLines) {
                $this->inventoryAssemblyLines->each(function (InventoryAssemblyLine $line) {
                    $line->delete();
                });
            }
            // cache inventory for lines
            $this->updateProductsInventory();

            // Delete the inventory assembly
            return parent::delete();
        });
    }

    public function updateProductsInventory()
    {
        $productIds = $this->inventoryAssemblyLines->pluck('product_id')->toArray();

        (new ProductInventoryManager($productIds))->setUpdateAverageCost(false)->updateProductInventoryAndAvgCost();
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', true, 'ID'),
            'warehouse.name' => Exporter::makeExportableField('warehouse_name'),
            'action_date' => Exporter::makeExportableField('action_date'),
            'action' => Exporter::makeExportableField('action'),
            'items.id' => Exporter::makeExportableField('item_id'),
            'quantity' => Exporter::makeExportableField('item_quantity'),
            'name' => Exporter::makeExportableField('item_name'),
            'sku' => Exporter::makeExportableField('item_sku'),
            'product_type' => Exporter::makeExportableField('item_product_type'),
            'unit_cost' => Exporter::makeExportableField('item_unit_cost'),
            'available_stock.available' => Exporter::makeExportableField('item_available_stock'),
            'created_at' => Exporter::makeExportableField('created_at', false),
            'updated_at' => Exporter::makeExportableField('updated_at', false),
        ];
    }

    public static function transformExportData(array $data): array
    {
        return BaseExporter::groupByLines($data, 'id', 'items');
    }
}
