<?php

namespace App\Models;

use App\Abstractions\FinancialDocumentInterface;
use App\Abstractions\IsLineInterface;
use App\Contracts\HasReference;
use App\Data\AccountingTransactionData;
use App\DataTable\Exports\DataTableExporter as Exporter;
use App\Exporters\MapsExportableFields;
use App\Helpers;
use App\Managers\WarehouseTransferManager;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HandleDateTimeAttributes;
use App\Models\Concerns\HasAccountingTransaction;
use App\Models\Concerns\HasAccountingTransactionLine;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use App\Services\Accounting\Actions\FinancialDocuments\BuildAccountingTransactionDataFromInventoryAdjustment;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\InventoryManagement\NegativeInventoryEvent;
use App\Services\InventoryManagement\PositiveInventoryEvent;
use App\Services\StockTake\OpenStockTakeException;
use Carbon\Carbon;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use InvalidArgumentException;
use Throwable;

/**
 * Class InventoryAdjustmentController.
 *
 *
 * @property int $id
 * @property Carbon $adjustment_date
 * @property int $product_id
 * @property int $warehouse_id
 * @property float $quantity
 * @property string $notes
 * @property Warehouse $warehouse
 * @property Product $product
 * @property bool $is_variable_cost
 * @property int|null $assembled_adjustment_id
 * @property int|null $link_id
 * @property string|null $link_type
 * @property Carbon|null $archived_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read float $unit_cost
 * @property-read ?int $parent_link_id
 * @property-read Collection|InventoryMovement[] $inventoryMovements
 * @property-read Collection|FifoLayer[] $fifoLayers
 * @property-read FifoLayer $creationFifoLayer
 */
class InventoryAdjustment extends Model implements Filterable, FinancialDocumentInterface, HasReference, IsLineInterface, MapsExportableFields, NegativeInventoryEvent, PositiveInventoryEvent, Sortable
{
    use Archive, HandleDateTimeAttributes, HasAccountingTransaction, HasAccountingTransactionLine, HasFilters, HasParentLink, HasSort;
    use HasFactory;

    const TYPE_INCREASE = 'increase';

    const TYPE_DECREASE = 'decrease';

    const TYPE_SET = 'set';

    const ADJUSTMENT_TYPES = [
        self::TYPE_INCREASE,
        self::TYPE_DECREASE,
        self::TYPE_SET,
    ];
    const REQUEST_LINK_TYPE_PO = 'purchase_order';

    const REQUEST_LINK_TYPE_IM = 'inventory_movement';

    const REQUEST_LINK_TYPE_IA = 'inventory_adjustment';

    const REQUEST_LINK_TYPE_SC = 'sales_credit';

    const REQUEST_LINK_TYPE_WT = 'warehouse_transfer';

    const REQUEST_LINK_TYPE_ST = 'stock_take';

    const REQUEST_LINK_TYPE_IAS = 'assembly';

    const REQUEST_LINK_TYPES = [
        PurchaseOrderShipmentReceiptLine::class => self::REQUEST_LINK_TYPE_PO,
        // This should be first in PO group for filters to work.
        PurchaseOrderLine::class => self::REQUEST_LINK_TYPE_PO,
        InventoryMovement::class => self::REQUEST_LINK_TYPE_IM,
        self::class => self::REQUEST_LINK_TYPE_IA,
        SalesCreditReturnLine::class => self::REQUEST_LINK_TYPE_SC,
        WarehouseTransferShipmentReceiptLine::class => self::REQUEST_LINK_TYPE_WT,
        StockTakeItem::class => self::REQUEST_LINK_TYPE_ST,
        InventoryAssemblyLine::class => self::REQUEST_LINK_TYPE_IAS,
    ];

    const ADJUSTMENT_REASON_OVER_FULFILLMENT = 'over_fulfillment';


    protected $fillable = [
        'adjustment_date',
        'product_id',
        'warehouse_id',
        'quantity',
        'notes',
        'link_id',
        'link_type',
        'reason',
        'updated_at'
    ];

    protected $casts = ['adjustment_date' => 'datetime', 'quantity' => 'float'];

    public function getReference(): string
    {
        return $this->id;
    }

    public function getDrawer(): Model
    {
        return $this;
    }

    public function getLineParent(): Model
    {
        return $this;
    }


    /**
     * @throws Throwable
     */
    public function getAccountingTransactionData(): AccountingTransactionData
    {
        return (new BuildAccountingTransactionDataFromInventoryAdjustment($this))->handle();
    }

    public function getParentAccountingTransaction(): ?AccountingTransaction
    {
        return null;
    }

    public function getAccountingDateFieldName(): string
    {
        return 'adjustment_date';
    }


    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function inventoryMovements()
    {
        $relation = $this->morphMany(InventoryMovement::class, 'link');

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (InventoryMovement $inventoryMovement) {
                $inventoryMovement->delete();
            });
        });

        return $relation;
    }

    public function link(): MorphTo
    {
        return $this->morphTo('link');
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    public function scopeAccountingReady(Builder $builder): Builder
    {
        return $builder;
    }

    public function scopeFilterLink(Builder $builder, array $relation, string $operator, $value, $conjunction)
    {
        $function = $conjunction === 'and' ? 'where' : 'orWhere';
        $prefix = 'App-Models-';

        $value = str_replace($prefix, '', $value);
        $builder = $builder->whereHas('link');
        $builder = $builder->{$function}(function ($query) use ($value, $operator) {
            switch ($operator) {
                case 'contains':
                    $operator = 'LIKE';
                    $query->where(function ($query) use ($operator, $value) {
                        $query->where('link_type', $operator, "%{$value}%");
                    });
                    break;
                case 'doesNotContain':
                    $operator = 'NOT LIKE';
                    $query->where(function ($query) use ($operator, $value) {
                        $namespace = "App\\\Models\\\\$value";
                        $query->Where('link_type', $operator, "%{$namespace}%");
                    });
                    break;
                case 'startsWith':
                    $operator = 'LIKE';

                    $query->where(function ($query) use ($operator, $value) {
                        $namespace = "App\\\Models\\\\$value";
                        $query->where('link_type', $operator, "{$namespace}%");
                    });
                    break;
                case 'endsWith':
                    $operator = 'LIKE';
                    $query->where('link_type', $operator, "%{$value}");
                    break;
                case '!=':
                    $namespace = "App\\\Models\\\\$value";

                    $query->where('link_type', 'not like', $namespace);
                    break;
                case '=':
                    $namespace = "App\\\Models\\\\$value";
                    $query->where('link_type', 'like', $namespace);
                    break;
                default:
                    $namespace = "App\\\Models\\\\$value";

                    $query->where('link_type', $operator, $namespace);
                    break;
            }
        });

        $operatorTypesWithNull = ['!=', 'doesNotContain', 'isEmpty', 'isNotEmpty'];
        if (in_array($operator, $operatorTypesWithNull)) {
            $foreignKeyName = $this->{$relation['name']}()->getForeignKeyName();
            if (in_array($operator, ['!=', 'doesNotContain', 'isEmpty'])) {
                $builder->orWhereNull($foreignKeyName);
            }
            if ($operator === 'isNotEmpty') {
                $builder->orWhereNotNull($foreignKeyName);
            }
        }

        return $builder;
    }

    private function makeLinkTypeFromName($name): array
    {
        if (in_array($name, self::REQUEST_LINK_TYPES)) {
            return array_keys(self::REQUEST_LINK_TYPES, $name);
        }

        return [__NAMESPACE__.'\\'.ucfirst(Str::camel($name))];
    }

    public function fifoLayers()
    {
        $relation = $this->morphMany(FifoLayer::class, 'link');

        $relation->onDelete(function (Builder $builder) {
            return $builder->each(function (FifoLayer $fifoLayer) {
                $fifoLayer->delete();
            });
        });

        return $relation;
    }

    public function creationFifoLayer(): MorphOne
    {
        return $this->morphOne(FifoLayer::class, 'link');
    }

    public function accountingTransaction()
    {
        return $this->morphOne(AccountingTransaction::class, 'link');
    }

    public function accountingTransactionLine()
    {
        return $this->morphOne(AccountingTransactionLine::class, 'link');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getUnitCostAttribute()
    {
        return $this->getFifoLayer()->avg_cost ?? $this->product->getCurrentFifoLayerForWarehouse($this->warehouse)?->avg_cost ?? $this->product->unit_cost ?? 0;
    }

    public function getLinkNameAttribute()
    {
        if ($this->link_type === PurchaseOrder::class) {
            return 'Purchase Receipt';
        }

        return null;
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    /**
     * Deletes the inventory adjustment.
     *
     * @throws Throwable
     */
    public function delete(bool $autoApplyStock = true): bool|array
    {
        /**
         * We handle the inventory movements and
         * fifo balances through the inventory
         * manager and delete the adjustment.
         */
        return DB::transaction(function () use ($autoApplyStock) {
            $manager = InventoryManager::with($this->warehouse_id, $this->product);
            if ($this->quantity < 0) {
                /**
                 * Deleting a negative movement, we need to
                 * put back used inventory.
                 */
                $manager->reverseNegativeEvent($this, $autoApplyStock);
            } elseif ($this->quantity > 0) {
                /**
                 * Deleting a positive movement, we attempt to
                 * charged back an used inventory and release.
                 */
                $manager->removeAllStockFrom($this);
            }

            $this->inventoryMovements()->delete();

            if ($this->link_type == WarehouseTransferLine::class)
            {
                /** @var WarehouseTransferLine $warehouseTransferLine */
                $warehouseTransferLine = $this->link;

                $result = parent::delete();

                app(WarehouseTransferManager::class)->setTransferReceiptStatus($warehouseTransferLine->warehouseTransfer);

                return $result;
            }

            // Delete the warehouse transfer
            return parent::delete();
        });
    }

    public function backorderQueueReleases()
    {
        return $this->morphMany(BackorderQueueRelease::class, 'link');
    }

    /**
     * Checks if an inventory adjustment cannot be deleted.
     *
     *
     * @deprecated
     */
    public function isDeletable(): bool
    {
        // Insufficient stock will be handled
        // in the inventory manager for deletion.
        // The manager may choose to take stock
        // from other sources, such as sales orders
        // if needed.
        return true;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return config('data_table.inventory_adjustment.columns');
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        if (Str::startsWith(func_get_arg(0), ['warehouse.'])) {
            return func_get_args();
        }

        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return ['adjustment_date', 'notes', 'reference', 'link_id', 'link_type'];
    }

    /**
     * {@inheritDoc}
     */
    public function sortableColumns()
    {
        return collect($this->availableColumns())->where('sortable', 1)->pluck('data_name')->all();
    }

    public static function getExportableFields(): array
    {
        return [
            'id' => Exporter::makeExportableField('id', false, 'ID'),
            'adjustment_date' => Exporter::makeExportableField('adjustment_date', false),
            'sku' => Exporter::makeExportableField('sku', false),
            'warehouse.name' => Exporter::makeExportableField('warehouse_name', false),
            'notes' => Exporter::makeExportableField('notes', false),
            'quantity' => Exporter::makeExportableField('quantity', false),
            'product_name' => Exporter::makeExportableField('product_name', false),
            'created_at' => Exporter::makeExportableField('created_at', false),
            'updated_at' => Exporter::makeExportableField('updated_at', false),
        ];
    }

    public function createInventoryMovementsForLayers(array $layers, ?Carbon $dateOverride = null): void
    {
        foreach ($layers as $layerInfo) {
            if (! ($layerInfo['layer'] instanceof FifoLayer)) {
                throw new InvalidArgumentException("Inventory adjustment can only have Fifo Layers, {$layerInfo['layer_type']} given.");
            }
            $this->makeAdjustmentMovement(-$layerInfo['quantity'], $layerInfo['layer']);
        }
    }

    public function getReductionActiveMovements(): Arrayable
    {
        return $this->inventoryMovements()->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '<', 0)
            ->get();
    }

    public function clearMovements(): void
    {
        $this->inventoryMovements()->delete();
    }

    /**
     * @throws OpenStockTakeException
     */
    public function reduceQtyWithSiblings(int $quantity, InventoryMovement $representingMovement): void
    {
        $representingMovement->quantity += $quantity;
        if ($representingMovement->quantity == 0) {
            $representingMovement->delete();
        } else {
            $representingMovement->save();
        }
    }

    public function createFifoLayer(int $quantity, ?float $unitCost = null, ?int $productId = null): FifoLayer
    {
        $fifoLayer = new FifoLayer;
        $fifoLayer->fifo_layer_date = $this->getEventDate();
        $fifoLayer->product_id = $this->product_id;
        $fifoLayer->original_quantity = $quantity;
        $fifoLayer->total_cost = $quantity * ($unitCost ?: $this->unit_cost);
        $fifoLayer->warehouse_id = $this->warehouse_id;

        // Add the fifo layer to the adjustment
        $this->fifoLayers()->save($fifoLayer);

        return $fifoLayer;
    }

    public function getUnitCost(): float
    {
        return $this->unit_cost > 0 ? $this->unit_cost : $this->product->getUnitCostAtWarehouse($this->warehouse_id);
    }

    public function getOriginatingMovement(): ?InventoryMovement
    {
        return $this->inventoryMovements->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '>', 0)
            ->first();
    }

    public function getFifoLayer(): ?FifoLayer
    {
        return $this->fifoLayers()->first();
    }

    public function createInventoryMovements(int $quantity, FifoLayer $fifoLayer): void
    {
        $this->makeAdjustmentMovement($quantity, $fifoLayer);
    }

    private function makeAdjustmentMovement(int $quantity, FifoLayer $fifoLayer): InventoryMovement
    {
        /** @var InventoryMovement $inventoryMovement */
        $inventoryMovement = $this->inventoryMovements()->where('layer_id', $fifoLayer->id)->first();
        if (! $inventoryMovement) {
            $inventoryMovement = new InventoryMovement();
        }
        $inventoryMovement->inventory_movement_date = $this->getEventDate();
        $inventoryMovement->product_id = $this->product_id;
        $inventoryMovement->type = InventoryMovement::TYPE_ADJUSTMENT;
        $inventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
        $inventoryMovement->warehouse_id = $this->warehouse_id;
        $inventoryMovement->quantity += $quantity; // Caters for updates as well.
        $inventoryMovement->warehouse_location_id = $this->warehouse->defaultLocation->id ?? null;
        $inventoryMovement->link_id = $this->id;
        $inventoryMovement->link_type = self::class;
        $fifoLayer->inventoryMovements()->save($inventoryMovement);

        return $inventoryMovement;
    }

    public function getLinkType(): string
    {
        return self::class;
    }

    public function getId(): int
    {
        return $this->id;
    }

    public function refreshFromFifoCostUpdate(float $amount = 0): void
    {
        $this->updated_at = now();
        $this->save();
    }

    public function getProductId(): int
    {
        return $this->product_id;
    }

    public function getWarehouseId(): int
    {
        return $this->warehouse_id;
    }

    public function getEventDate(): Carbon
    {
        return $this->adjustment_date->max(Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE), 'UTC'));
    }

    public function getQuantity(): int
    {
        return $this->quantity;
    }

    public function getType(): string
    {
        return InventoryMovement::TYPE_ADJUSTMENT;
    }
}
