<?php

namespace App\Models;

use App\Helpers;
use App\Http\Resources\Amazon\ProductResource;
use App\Http\Resources\Amazon\ShippingServiceResource as AmazonShippingServiceResource;
use App\Http\Resources\Shipstation\ShippingServiceResource as ShipstationShippingServiceResource;
use App\Http\Resources\Shopify\ShopifyShippingServiceResource as ShopifyShippingServiceResource;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use InvalidArgumentException;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\Ebay\Entities\EbayOrder;
use Modules\Ebay\Entities\EbayProduct;
use Modules\WooCommerce\Entities\WooCommerceOrder;
use Modules\WooCommerce\Entities\WooCommerceProduct;
use Sonnenglas\AmazonMws\Enum;

/**
 * Class Integration.
 *
 *
 * @property int $id
 * @property string $name
 * @property string|null $description
 * @property array|null $integration_settings_template
 * @property string|null $image_url
 * @property bool $supports_multiple_instances
 * @property string $integration_type
 * @property string|null $integration_sub_type
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read array $credentials
 * @property-read Collection|IntegrationInstance[] $integrationInstances
 * @property-read Carbon|null $last_sync
 */
class Integration extends Model implements Filterable
{
    use HasFactory;
    use HasFilters;

    private static $skuIntegration;

    protected $guarded = [];

    const NAME_SKU_IO = 'SKU.io';

    const NAME_AMAZON_US = 'Amazon US';

    const NAME_SHIPSTATION = 'ShipStation';

    const NAME_SHIPMYORDERS = 'ShipMyOrders';

    const NAME_EBAY = 'Ebay';

    const NAME_SHOPIFY = 'Shopify';

    const NAME_STARSHIPIT = 'Starshipit';

    const NAME_MAGENTO = 'Magento';

    const NAME_XERO = 'Xero';
    const NAME_VERACORE = 'Veracore';

    const NAME_WOOCOMMERCE = 'WooCommerce';

    const NAME_QBO = 'QuickBooks Online';

    const TYPE_SALES_CHANNEL = 'sales_channel';

    const TYPE_SHIPPING_PROVIDER = 'shipping_provider';

    const TYPE_ACCOUNTING = 'accounting';

    const ACCOUNTING = 'accounting';

    protected $casts = [
        'integration_settings_template' => 'array',
        'supports_multiple_instances' => 'boolean',
        'last_sync' => 'datetime',
    ];

    public function integrationInstances()
    {
        return $this->hasMany(IntegrationInstance::class);
    }

    public function isSalesChannel(): bool
    {
        return $this->integration_type === self::TYPE_SALES_CHANNEL;
    }

    public function isShippingProvider(): bool
    {
        return $this->integration_type === self::TYPE_SHIPPING_PROVIDER;
    }

    /**
     * developer credentials based on integration name.
     */
    public function getCredentialsAttribute(): array
    {
        switch ($this->name) {
            case self::NAME_AMAZON_US:
                return [
                    'developerId' => env('AMZ_US_DEVELOPER_ID'),
                    'keyId' => env('AMZ_US_DEVELOPER_ACCESS_KEY'),
                    'secretKey' => env('AMZ_US_DEVELOPER_SECRET_KEY'),
                    'marketplaceId' => Enum::MARKETPLACE_US,
                ];
            case self::NAME_EBAY:
                $isProduction = env('EBAY_ENVIRONMENT') == 'productions';

                return [
                    'appId' => env($isProduction ? 'EBAY_US_APP_ID' : 'EBAY_SANDBOX_US_APP_ID'),
                    'certId' => env($isProduction ? 'EBAY_US_CERT_ID' : 'EBAY_SANDBOX_US_CERT_ID'),
                    'devId' => env($isProduction ? 'EBAY_US_DEV_ID' : 'EBAY_SANDBOX_US_DEV_ID'),
                    'ruName' => env('EBAY_US_RU_NAME_ID'),
                ];
            default:
                return [];
        }
    }

    public function synced(?Carbon $datetime = null)
    {
        $this->last_sync = $datetime ?? now();
        $this->save();
    }

    /**
     * Get validation rules for connection setting based on integration name.
     */
    public function getConnectionSettingsRules(bool $update = false): array
    {
        $rules = [];
        switch ($this->name) {
            case self::NAME_AMAZON_US:
                $rules = ['merchantId' => 'required', 'authToken' => 'required'];
                break;
            case self::NAME_SHIPSTATION:
                $rules = ['apiKey' => 'required', 'apiSecret' => 'required'];
                break;
            case self::NAME_STARSHIPIT:
                $rules = ['apiKey' => 'required', 'subscriptionKey' => 'required'];
                break;
            case self::NAME_MAGENTO:
                $rules = [
                    'oauth_consumer_key' => 'required',
                    'oauth_consumer_secret' => 'required',
                    'access_token' => 'required',
                    'secret_token' => 'required',
                    'store_base_url' => 'required',
                ];
                break;
            case self::NAME_XERO:
                //$rules = ['clientId' => 'required', 'clientSecret' => 'required'];
                break;
        }

        $rules = Helpers::addPrefixToArrayKeys($rules, 'connection_settings');

        return $update ? $this->addSometimesRule($rules) : $rules;
    }

    /**
     * Get validation rules based on integration type.
     */
    public function getIntegrationTypeRules(bool $update = false): array
    {
        $rules = match ($this->integration_type) {
            self::TYPE_SALES_CHANNEL => ($this->name != self::NAME_MAGENTO && $this->name != self::NAME_EBAY)
                ? ['integration_settings.settings.store.id' => 'required|exists:stores,id']
                : [],
            default => [],
        };

        return $update ? $this->addSometimesRule($rules) : $rules;
    }

    /**
     * Get the appropriate model based on integration name.
     *
     * @param  bool  $isFBA
     */
    public function getModelPath($type): string
    {
        $models = [
            self::NAME_AMAZON_US => [
                'products' => AmazonProduct::class,
                'orders' => AmazonOrder::class,
            ],
            self::NAME_SHOPIFY => [
                'products' => \App\Models\Shopify\ShopifyProduct::class,
                'orders' => \App\Models\Shopify\ShopifyOrder::class,
            ],
            self::NAME_MAGENTO => [
                'products' => \App\Models\Magento\Product::class,
                'orders' => \App\Models\Magento\Order::class,
            ],
            self::NAME_EBAY => [
                'products' => EbayLegacyProduct::class,
                'orders' => EbayOrder::class,
            ],
            self::NAME_WOOCOMMERCE => [
                'products' => WooCommerceProduct::class,
                'orders' => WooCommerceOrder::class,
            ],
        ];

        $modelPath = Arr::get($models, $this->name.'.'.$type, null);

        if (is_null($modelPath)) {
            throw new InvalidArgumentException("Add model path for records type '$type' for this '{$this->name}'");
        }

        return $modelPath;
    }

    /**
     * Get the appropriate model based on integration name.
     *
     * @param  bool  $isFBA
     */
    public function getProductsModelPath(): string
    {
        return $this->getModelPath('products');
    }

    /**
     * Get the appropriate model based on integration name.
     */
    public function getShippingServiceModelPath(): string
    {
        switch ($this->name) {
            case self::NAME_AMAZON_US:
                return \App\Models\Amazon\ShippingService::class;
            case self::NAME_SHOPIFY:
                return \App\Models\Shopify\ShopifyShippingService::class;
            default:
                throw new InvalidArgumentException('add model path for this integration');
        }
    }

    /**
     * Get the appropriate resource on integration name.
     */
    public function getShippingServiceResourcePath(): string
    {
        switch ($this->name) {
            case self::NAME_SHIPSTATION:
                return ShipstationShippingServiceResource::class;
            case self::NAME_AMAZON_US:
                return AmazonShippingServiceResource::class;
            case self::NAME_SHOPIFY:
                return ShopifyShippingServiceResource::class;
            default:
                throw new InvalidArgumentException('add resource path for this integration');
        }
    }

    /**
     * Get the appropriate model based on integration name.
     */
    public function getOrderModelPath(): string
    {
        return $this->getModelPath('orders');
    }

    /**
     * Get the appropriate product resource based on integration name.
     */
    public function getProductResourcePath(): string
    {
        switch ($this->name) {
            case self::NAME_AMAZON_US:
                return ProductResource::class;
            case self::NAME_SHOPIFY:
                return \App\Http\Resources\Shopify\ShopifyProductResource::class;
            default:
                throw new InvalidArgumentException('add resource path for this integration');
        }
    }

    /**
     * Get product keys based on integration name.
     *
     *
     * @return string[]
     */
    public function getProductKeys(bool $isFBA = false): array
    {
        switch ($this->name) {
            case self::NAME_AMAZON_US:
                return $isFBA ? ['sku' => 'sku', 'name' => 'product-name'] : ['sku' => 'seller-sku', 'name' => 'item-name'];
            default:
                throw new InvalidArgumentException('add model path for this integration');
        }
    }

    /**
     * Add sometimes rule for required attributes.
     */
    private function addSometimesRule(array $rules): array
    {
        foreach ($rules as $attribute => $attributeRules) {
            if (is_array($attributeRules) && in_array('required', $attributeRules)) {
                $attributeRules = Arr::prepend($attributeRules, 'sometimes');
            } else {
                $attributeRules = 'sometimes|'.$attributeRules;
            }

            $rules[$attribute] = $attributeRules;
        }

        return $rules;
    }

    public static function getSkuIntegration()
    {
        if (! isset(static::$skuIntegration)) {
            static::$skuIntegration = self::with([])->where('name', self::NAME_SKU_IO)->first();
        }

        return static::$skuIntegration;
    }

    /**
     * {@inheritDoc}
     */
    public function availableColumns()
    {
        return ['name', 'description', 'integration_type'];
    }

    /**
     * {@inheritDoc}
     */
    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    /**
     * {@inheritDoc}
     */
    public function generalFilterableColumns(): array
    {
        return $this->availableColumns();
    }
}
