<?php

namespace App\Models;

use App\Abstractions\UniqueFieldsInterface;
use App\DataTable\DataTableModelInterface;
use App\DataTable\DataTableModelTrait;
use App\Enums\FinancialAlertTypeEnum;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Contracts\Filterable;
use App\Models\Contracts\Sortable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Arr;

/**
 * @property int $id
 * @property int $sales_order_line_financial_id
 * @property int $financial_alert_classification_id
 * @property FinancialAlertTypeEnum $alert_type
 * @property string $note
 * @property int $archived_by_user_id
 * @property ?Carbon $created_at
 * @property ?Carbon $updated_at
 * @property ?Carbon $archived_at
 * @property-read SalesOrderLineFinancial $salesOrderLineFinancial
 * @property-read FinancialAlertClassification $financialAlertClassification
 * @property-read User $archivedByUser
 */
class FinancialAlert extends Model implements DataTableModelInterface, Filterable, Sortable, UniqueFieldsInterface
{
    use DataTableModelTrait;
    use HasFactory;
    use HasFilters;
    use HasSort;
    use Archive;

    protected $guarded = [];

    protected $casts = [
        'archived_at' => 'datetime',
        'alert_type' => FinancialAlertTypeEnum::class,
    ];

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */
    public static function getUniqueFields(): array
    {
        return ['sales_order_line_financial_id'];
    }

    public static function specialLabels(): array
    {
        return [
            'id' => 'ID',
            'salesOrder' => 'Order #',
            'sku' => 'SKU',
            'total_revenue' => 'Revenue',
            'cogs' => 'COGS',
            'cogs_returned' => 'Returned COGS',
            'cost_allocated' => 'Cost Allocated',
            'total_cost' => 'Cost',
            'profit_margin' => 'Margin (%)',
        ];
    }

    public function availableColumns(): array
    {
        $this->availableColumns = [
            'id' => 'integer',
            'salesOrder' => 'string',
            'sku' => 'string',
            'alert_type' => 'alert_type',
            'total_revenue' => 'decimal',
            'cogs' => 'decimal',
            'cogs_returned' => 'decimal',
            'cost_allocated' => 'decimal',
            'allocated_details' => 'json',
            'total_cost' => 'decimal',
            'profit' => 'decimal',
            'profit_margin' => 'decimal',
            'classification' => 'string',
            'note' => 'string',
            'resolution' => 'string',
            'archived_by_user_id' => 'integer',
            'archived_by_user' => 'string',
            'created_at' => 'datetime',
            'updated_at' => 'datetime',
            'archived_at' => 'datetime',
        ];
        return $this->availableColumns;
    }

    public function visibleColumns(): array
    {
        return array_keys(Arr::except($this->availableColumns, ['archived_by_user_id', 'archived_at', 'archived_by_user']));
    }

    public function filterableColumns(): array
    {
        return Arr::except(array_merge(array_keys($this->availableColumns), [
           'archived_at',
           'archived_by_user_id',
            'archived_by_user',
        ]), ['allocated_details']);
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function salesOrderLineFinancial(): BelongsTo
    {
        return $this->belongsTo(SalesOrderLineFinancial::class);
    }

    public function financialAlertClassification(): BelongsTo
    {
        return $this->belongsTo(FinancialAlertClassification::class);
    }

    public function archivedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'archived_by_user_id');
    }
}
