<?php

namespace App\Models;

use App\Jobs\ForecastInventoryJob;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

/**
 * Class Currency.
 *
 *
 * @property int $id
 * @property string $code
 * @property float $conversion
 * @property bool $is_default
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 */
class Currency extends Model implements Filterable
{
    use HasFactory;
    use HasFilters;

    protected $casts = ['is_default' => 'boolean', 'conversion' => 'float'];

    protected $fillable = ['code', 'conversion', 'is_default'];

    private static $default = null;

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function constantCurrency()
    {
        return $this->belongsTo(\App\Models\Constant\ConstantsCurrency::class, 'code', 'code');
    }

    public function salesOrders()
    {
        return $this->hasMany(SalesOrder::class);
    }

    public function purchaseOrders()
    {
        return $this->hasMany(PurchaseOrder::class);
    }

    public function productPricingTiers()
    {
        return $this->hasMany(ProductPricingTier::class, 'currency_code', 'code');
    }

    public function supplierPricingTiers()
    {
        return $this->hasMany(SupplierPricingTier::class, 'currency_code', 'code');
    }

    /*
    |--------------------------------------------------------------------------
    | Accessors & Mutators
    |--------------------------------------------------------------------------
    */

    public function getNameAttribute()
    {
        return $this->constantCurrency->name ?? null;
    }

    /*
    |--------------------------------------------------------------------------
    | Functions
    |--------------------------------------------------------------------------
    */

    public function save(array $options = [])
    {

        $rebuildForecasting = false;
        if ($this->is_default) {
            // reset default currency
            if (($this->exists && ! $this->getOriginal('is_default')) || ! $this->exists) {
                self::with([])->update(['is_default' => false]);
                $rebuildForecasting = true;
            }

            // set conversation to 1
            $this->conversion = 1;
        }

        $saved = parent::save($options);

        if($saved && $rebuildForecasting){
            dispatch(app(ForecastInventoryJob::class))->onQueue('forecasting');
        }

        return $saved;
    }

    public function delete()
    {
        if ($usage = $this->isUsed()) {
            return $usage;
        }

        return parent::delete();
    }

    /**
     * Determined if the currency is used.
     *
     * @return array|bool
     */
    public function isUsed()
    {
        $usage = [];

        if ($this->is_default) {
            $usage['is_default'] = __('messages.currency.delete_failed');
        }

        $relatedRelations = ['salesOrders', 'productPricingTiers', 'supplierPricingTiers'];
        $this->loadCount($relatedRelations);

        foreach ($relatedRelations as $relatedRelation) {
            $countKeyName = str_replace('-', '_', Str::kebab($relatedRelation)).'_count';
            if ($this->{$countKeyName}) {
                $relatedName = Str::singular(str_replace('-', ' ', Str::kebab($relatedRelation)));

                $usage[$relatedRelation] = trans_choice('messages.currently_used', $this->{$countKeyName}, [
                    'resource' => $relatedName,
                    'model' => 'currency('.$this->code.')',
                ]);
            }
        }

        return count($usage) ? $usage : false;
    }

    /**
     * Get default currency.
     *
     * if the user does not set default currency, return first currency
     */
    public static function default(): self
    {
        if (! static::$default) {
            static::$default = static::with([])->where('is_default', true)->first() ?: self::with([])->first();
        }

        return static::$default;
    }

    public function availableColumns()
    {
        return [];
    }

    public function filterableColumns(): array
    {
        return [
            'code',
            'conversion',
            'is_default',
            'created_at',
            'updated_at',
        ];
    }

    public function generalFilterableColumns(): array
    {
        return $this->filterableColumns();
    }
}
