<?php

namespace App\Models\Concerns;

use DateTime;
use DateTimeInterface;

trait WebClient
{
    /**
     * Serialize data
     *
     * @param  mixed  $data the data to serialize
     * @param  null  $type the OpenAPIToolsType of the data
     * @param  null  $format the format of the OpenAPITools type of the data
     * @return float|object|array|bool|int|string|null serialized form of $data
     */
    public static function sanitizeForSerialization(mixed $data, $type = null, $format = null): float|object|array|bool|int|string|null
    {
        if (is_scalar($data) || $data === null) {
            return $data;
        } elseif ($data instanceof DateTime) {
            return ($format === 'date') ? $data->format('Y-m-d') : $data->format(DateTimeInterface::ATOM);
        } elseif (is_array($data)) {
            foreach ($data as $property => $value) {
                $data[$property] = self::sanitizeForSerialization($value);
            }

            return $data;
        } elseif (is_object($data)) {
            $values = [];
            foreach ($data as $property => $value) {
                $values[$property] = self::sanitizeForSerialization($value);
            }

            return (object) $values;
        } else {
            return (string) $data;
        }
    }

    /**
     * Take value and turn it into a string suitable for inclusion in
     * the query, by imploding comma-separated if it's an object.
     * If it's a string, pass through unchanged. It will be url-encoded
     * later.
     *
     * @param  string[]|string|DateTime  $object an object to be serialized to a string
     * @return DateTime|string the serialized object
     */
    public static function toQueryValue($object): DateTime|string
    {
        if (is_array($object)) {
            return implode(',', $object);
        } else {
            return self::toString($object);
        }
    }

    /**
     * Take value and turn it into a string suitable for inclusion in
     * the parameter. If it's a string, pass through unchanged
     * If it's a datetime object, format it in ISO8601
     *
     * @param  string|DateTime  $value the value of the parameter
     * @return DateTime|string the header string
     */
    public static function toString($value): DateTime|string
    {
        if ($value instanceof DateTime) { // datetime in ISO8601 format
            return $value->format(DateTimeInterface::ATOM);
        } else {
            return $value;
        }
    }

    /**
     * @param  string[]  $accept
     */
    public function selectHeadersForMultipart(array $accept): array
    {
        $headers = $this->selectHeaders($accept, []);

        unset($headers['Content-Type']);

        return $headers;
    }

    /**
     * @param  string[]  $accept
     * @param  string[]  $contentTypes
     */
    public function selectHeaders(array $accept, array $contentTypes): array
    {
        $headers = [];

        $accept = $this->selectAcceptHeader($accept);
        if ($accept !== null) {
            $headers['Accept'] = $accept;
        }

        $headers['Content-Type'] = $this->selectContentTypeHeader($contentTypes);

        return $headers;
    }

    /**
     * Return the header 'Accept' based on an array of Accept provided
     *
     * @param  string[]  $accept Array of header
     * @return string|null Accept (e.g. application/json)
     */
    private function selectAcceptHeader(array $accept): ?string
    {
        if (count($accept) === 0 || (count($accept) === 1 && $accept[0] === '')) {
            return null;
        } elseif (preg_grep("/application\/json/i", $accept)) {
            return 'application/json';
        } else {
            return implode(',', $accept);
        }
    }

    /**
     * Return the content type based on an array of content-type provided
     *
     * @param  string[]  $contentType Array fo content-type
     * @return string Content-Type (e.g. application/json)
     */
    private function selectContentTypeHeader(array $contentType): string
    {
        if (count($contentType) === 0 || (count($contentType) === 1 && $contentType[0] === '')) {
            return 'application/json';
        } elseif (preg_grep("/application\/json/i", $contentType)) {
            return 'application/json';
        } else {
            return implode(',', $contentType);
        }
    }
}
