<?php

namespace App\Models\Concerns;

use App\Enums\TrackedJobLogStatusEnum;
use App\Models\TrackedJobLog;
use Error;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Carbon;

/*
 * TODO: Add completed_tasks, total_tasks, and progress_percentage to TrackedJobLog
 */
trait TrackableJobTrait
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected TrackedJobLog $trackedJobLogRecord;

    protected function serializeArguments($arguments): array
    {
        return array_map(function ($argument) {
            if ($argument instanceof Model) {
                return [
                    'type' => 'eloquent_model',
                    'model' => get_class($argument),
                    'id' => $argument->getKey(),
                ];
            }

            // ... handle other complex types as needed ...

            return $argument;
        }, $arguments);
    }

    protected function initializeTrackedJob($relatedModel, $jobName = null): void
    {
        $serializedArgs = $this->serializeArguments(func_get_args());
        $this->trackedJobLogRecord = TrackedJobLog::create([
            'job_type' => static::class,
            'queue' => $this->queue ?? 'default',
            'job_name' => $jobName ?? static::class,
            'status' => TrackedJobLogStatusEnum::PENDING,
            'related_model_type' => get_class($relatedModel),
            'related_model_id' => $relatedModel->id,
            'job_data' => json_encode($serializedArgs),
        ]);
    }

    public function trackProgress(TrackedJobLogStatusEnum $status, $message = null): void
    {
        $data = [
            'status' => $status,
            'messages' => $message,
        ];
        switch ($status) {
            case TrackedJobLogStatusEnum::PROCESSING:
                $data['job_id'] = $this->job->getJobId();
                $data['attempted_at'] = Carbon::now()->format('Y-m-d H:i:s.u');
                break;
            case TrackedJobLogStatusEnum::COMPLETED:
            case TrackedJobLogStatusEnum::FAILED:
                $data['completed_at'] = Carbon::now()->format('Y-m-d H:i:s.u');
                break;
            default:
                break;
        }
        $this->trackedJobLogRecord->update($data);
    }

    public function failed(Exception|Error $exception): void
    {
        $this->trackProgress(TrackedJobLogStatusEnum::FAILED, $exception->getMessage());
    }
}
